﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;
using StatsDataSource.Storage;
using StatsDataSource.ObjectModel;
using System.Diagnostics;

namespace StatsDataSource.Utils
{
	class XMLStatsParser
	{
		//////////////////////////////////////////////////////////////////////////
		XmlReader m_reader;
		XmlDocument m_document;
		StatsRepository m_repository;
		int m_lastTimeStamp;

		GameNode m_currNode;
		//////////////////////////////////////////////////////////////////////////

		public XMLStatsParser(XmlReader reader, StatsRepository repo, int timeStamp)
		{
			m_reader = reader;
			m_document = new XmlDocument();
			m_repository = repo;
			m_lastTimeStamp = timeStamp;
		}

		//////////////////////////////////////////////////////////////////////////

		public void Parse()
		{
			m_document.Load(m_reader);

			for (int i = 0; i != m_document.ChildNodes.Count; ++i)
				if (m_document.ChildNodes[i].Name != "xml")
					Traverse(m_document.ChildNodes[i]);
		}

		private void Traverse(XmlNode node)
		{
			if (VisitNode(node))
			{
				for (int i = 0; i != node.ChildNodes.Count; ++i)
					Traverse(node.ChildNodes[i]);

				LeaveNode(node);
			}
		}

		//////////////////////////////////////////////////////////////////////////

		private bool VisitNode(XmlNode node)
		{
			ScopeDesc sd = m_repository.Registry.GetScopeDesc(node.Name);
			ElementDesc ed = m_repository.Registry.GetElementDesc(node.Name);
			if (sd != null)
				return VisitScope(node, sd);
			if (ed != null)
				return VisitElement(node, ed);

			switch (node.Name)
			{
				case "timelines": return true;
				case "timeline": return VisitEventGroup(node);
				default: return VisitState(node);
			}
		}

		private void LeaveNode(XmlNode node)
		{
			if (node.Name != "timelines")
				m_currNode = m_currNode.Parent;
		}

		//////////////////////////////////////////////////////////////////////////

		private bool VisitScope(XmlNode node, ScopeDesc desc)
		{
			Locator l = new Locator(-1, desc.ID, 0, 0, ++m_lastTimeStamp);
			m_currNode = (m_currNode != null)
					? ((GameScope)m_currNode).AddChild(l)
					: m_repository.AddRootScope(l);

			AddStates(node);
			return true;
		}

		private bool VisitElement(XmlNode node, ElementDesc desc)
		{
			Debug.Assert(m_currNode.Locator.IsScope());
			GameScope sc = (GameScope)m_currNode;

			++m_lastTimeStamp;
			Locator loc = new Locator(desc.ID, sc.Desc.ID, desc.ID, m_lastTimeStamp, m_lastTimeStamp);
			m_currNode = sc.AddChild(loc);

			AddStates(node);
			return true;
		}

		private bool VisitEventGroup(XmlNode node)
		{
			string name = node.Attributes["name"].Value;
			StatDesc desc = m_repository.Registry.GetEventDesc(name);
			if (desc == null)
				desc = m_repository.Registry.RegisterEvent(name);

			GameEventGroup gr = m_currNode.AddEventGroup(desc);

			for (int i = 0; i != node.ChildNodes.Count; ++i)
			{
				XmlNode evnt = node.ChildNodes[i];
				Int64 time = Int64.Parse(evnt.Attributes["time"].Value);

				List<KeyValuePair<string, string>> prms = new List<KeyValuePair<string, string>>();
				ExtractEventParameters(evnt, prms);

				gr.AddEvent(time, prms.ToArray());
			}

			return false;
		}

		private void ExtractEventParameters(XmlNode evnt, List<KeyValuePair<string, string>> prms)
		{
			for (int i = 0; i != evnt.Attributes.Count; ++i)
			{
				XmlAttribute a = evnt.Attributes[i];
				if (a.Name != "time")
					prms.Add(new KeyValuePair<string, string>(a.Name, a.Value));
			}

			if (evnt.ChildNodes.Count != 0)
			{
				for (int i = 0; i != evnt.ChildNodes[0].Attributes.Count; ++i)
				{
					XmlAttribute a = evnt.ChildNodes[0].Attributes[i];
					prms.Add(new KeyValuePair<string, string>(a.Name, a.Value));
				}
			}
		}

		GameState m_currState;
		private bool VisitState(XmlNode node)
		{
			StatDesc desc = GetStateDesc(node.Name);

			m_currState = (m_currState != null)
					? m_currState.AddChild(desc, null)
					: m_currNode.AddState(desc, null);

			for (int i = 0; i != node.Attributes.Count; ++i)
			{
				XmlAttribute attr = node.Attributes[i];
				StatDesc cd = GetStateDesc(attr.Name);
				m_currState.AddChild(cd, attr.Value);
			}

			for (int i = 0; i != node.ChildNodes.Count; ++i)
			{
				VisitState(node.ChildNodes[i]);
			}

			m_currState = m_currState.Parent;

			return false;
		}

		//////////////////////////////////////////////////////////////////////////

		private StatDesc GetStateDesc(string name)
		{
			StatDesc desc = m_repository.Registry.GetStateDesc(name);
			if (desc == null)
				desc = m_repository.Registry.RegisterState(name);
			return desc;
		}

		private void AddStates(XmlNode n)
		{
			for (int i = 0; i != n.Attributes.Count; ++i)
			{
				XmlAttribute a = n.Attributes[i];

				StatDesc sd = m_repository.Registry.GetStateDesc(a.Name);

				if (sd == null)
					sd = m_repository.Registry.RegisterState(a.Name);

				m_currNode.AddState(sd, a.Value);
			}
		}

		//////////////////////////////////////////////////////////////////////////
	}
}
