﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;
using StatsDataSource.ObjectModel;
using StatsDataSource.Storage;

namespace StatsDataSource.Utils
{
	class XMLStatsSerializer : IDisposable, IStatsVisitor
	{
		private XmlTextWriter m_xml;

		////////////////////////////////////////////////////////////

		public XMLStatsSerializer(string fileName)
		{
			m_xml = new XmlTextWriter(fileName, Encoding.UTF8);
			m_xml.Formatting = Formatting.Indented;
		}

		public void Dispose()
		{
			if (m_xml != null)
				m_xml.Close();
		}

		////////////////////////////////////////////////////////////

		public bool VisitScope(GameScope scope)
		{
			m_xml.WriteStartElement(scope.Desc.Name);
			return true;
		}

		public void LeaveScope(GameScope scope)
		{
			m_xml.WriteEndElement();
		}

		public bool VisitElement(GameElement elem)
		{
			m_xml.WriteStartElement(elem.Desc.Name);
			return true;
		}

		public void LeaveElement(GameElement elem)
		{
			m_xml.WriteEndElement();
		}

		////////////////////////////////////////////////////////////

		public bool VisitState(GameState state)
		{
			if (state.Children.Count == 0)
			{
				if (state.Value != null)
					WriteAttribute(state.Desc.Name, state.Value);
			}
			else
			{
				m_xml.WriteStartElement(state.Desc.Name);
			}
			return true;
		}

		public void LeaveState(GameState state)
		{
			if (state.Children.Count != 0)
				m_xml.WriteEndElement();
		}

		////////////////////////////////////////////////////////////

		public bool StartEventGroups(GameNode node)
		{
			m_xml.WriteStartElement("timelines");
			return true;
		}

		public bool VisitEventGroup(GameEventGroup evntGroup)
		{
			m_xml.WriteStartElement("timeline");
			m_xml.WriteStartAttribute("name");
			m_xml.WriteValue(evntGroup.Desc.Name);
			m_xml.WriteEndAttribute();
			return true;
		}

		public void LeaveEventGroup(GameEventGroup evntGroup)
		{
			m_xml.WriteEndElement();
		}

		public void EndEventGroups(GameNode node)
		{
			m_xml.WriteEndElement();
		}

		public void VisitEvent(GameEvent evnt)
		{
			m_xml.WriteStartElement("val");
			m_xml.WriteStartAttribute("time");
			m_xml.WriteValue(evnt.TimeMillisecs.ToString());
			m_xml.WriteEndAttribute();

			if (evnt.Parameters.Count == 1)
			{
				WriteAttribute(evnt.Parameters.Keys[0], evnt.Parameters.Values[0]);
			}
			else if (evnt.Parameters.Count > 1)
			{
				m_xml.WriteStartElement("param");

				foreach (KeyValuePair<string,string> a in evnt.Parameters)
					WriteAttribute(a.Key, a.Value);

				m_xml.WriteEndElement();
			}

			m_xml.WriteEndElement();
		}

		private void WriteAttribute<T>(string name, T value)
		{
			m_xml.WriteStartAttribute(name);
			m_xml.WriteValue(value);
			m_xml.WriteEndAttribute();
		}
	}

}
