﻿using System;
using System.Collections.Generic;
using System.Text;
using StatsDataSource.ObjectModel;
using SlimDX;
using System.Drawing;
using StatsDataSource.Storage;
using System.Diagnostics;

namespace StatsGameK01
{
	class GlobalTreeUpdateVisitor : IStatsVisitor
	{
		MinimapForm m_form;
		MiniMap m_map;

		EEventArchType m_currAT;
		EEventOwner m_currOwner;
		GameEventGroup m_positionGroup;
		PathData m_path;

		public GlobalTreeUpdateVisitor(MinimapForm form)
		{
			m_form = form;
		}

		public bool VisitScope(GameScope scope)
		{
			if (scope.Parent != null)
				return true;

			m_map = null;

			if (scope.GetTag(Tags.MAP) != null)
				return false;

			m_map = m_form.LoadMap(scope);
			if (m_map != null)
			{
				m_map.ClearFullBatch();
				return true;
			}
			return false;
		}
		public void LeaveScope(GameScope scope) { }

		public bool VisitElement(GameElement elem) { return true; }
		public void LeaveElement(GameElement elem) { }

		public bool VisitState(GameState state) { return false; }
		public void LeaveState(GameState state) { }

		public bool StartEventGroups(GameNode node)
		{
			m_positionGroup = m_form.GetPositionTrack(node);

			if (m_positionGroup != null)
				foreach (var e in m_positionGroup.Events)
					if (e.GetTag(Tags.POSTION) == null)
						e.SetTag(Tags.POSTION, m_form.GetEventPosition(e));

			return m_positionGroup != null;
		}

		public bool VisitEventGroup(GameEventGroup evntGroup)
		{
			m_currAT = m_form.GetEventType(evntGroup.Desc.Name);
			m_currOwner = m_form.GetEventOwner(evntGroup);

			if (m_currAT == EEventArchType.Path)
			{
				var clr = m_currOwner == EEventOwner.AI ?
					Color.Blue :
					(m_currOwner == EEventOwner.Player)
						? Color.Red : Color.RoyalBlue;

				m_path = new PathData(m_currOwner);
				m_map.FullPaths[evntGroup.Node] = m_path;
			}

			return m_currAT != EEventArchType.Num;
		}

		public void LeaveEventGroup(GameEventGroup evntGroup)
		{ }

		public void EndEventGroups(GameNode node) { m_positionGroup = null; }

		public void VisitEvent(GameEvent evnt)
		{
			if (m_currAT != EEventArchType.Path)
			{
				MarkerInfo inf;
				if (!m_form.GetMarkerInfo(evnt, m_positionGroup, m_map, m_currOwner, out inf))
					return;

				var place = m_map.FullQTree.Insert(inf);
				Debug.Assert(place != null);
			}
			else
			{
				var pos = m_map.WorldToImage((Vector2)evnt.GetTag(Tags.POSTION));
				m_path.Points.Add(new Vector3(pos.X, pos.Y, 0));
				m_path.Events.Add(evnt);
			}
		}
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	
	class GlobalTreeIncrementalUpdater
	{
		MinimapForm m_form;
		MiniMap m_map;

		EEventArchType m_currAT;
		EEventOwner m_currOwner;
		GameEventGroup m_positionGroup;
		PathData m_path;

		List<GameEvent> m_pendingEvents;
		List<GameEvent> m_pendingPositions;

		public GlobalTreeIncrementalUpdater(MinimapForm form, List<GameEvent> pendings, List<GameEvent> positions)
		{
			m_form = form;
			m_pendingEvents = pendings;
			m_pendingPositions = positions;
		}

		public void ProcessMarkerEvent(GameEvent evnt)
		{
			MarkerInfo inf;
			if (!m_form.GetMarkerInfo(evnt, m_positionGroup, m_map, m_currOwner, out inf))
				return;

			var place = m_map.FullQTree.Insert(inf);
			Debug.Assert(place != null);
		}

		public void ProcessPositionEvent(GameEvent evnt)
		{
			var pos = m_map.WorldToImage((Vector2)evnt.GetTag(Tags.POSTION));
			m_path.Points.Add(new Vector3(pos.X, pos.Y, 0));
			m_path.Events.Add(evnt);
		}
		
		void UpdateGroupData(GameEventGroup g)
		{
			m_currAT = m_form.GetEventType(g.Desc.Name);
			m_currOwner = m_form.GetEventOwner(g);
		}

		void UpdateNodeData(GameNode node)
		{
			var session = node;

			while (session.Parent != null)
				session = session.Parent;

			object tag = session.GetTag(Tags.MAP);
			m_map = tag == null ?
				m_form.LoadMap((GameScope)session) :
				(MiniMap)tag;
		}

		void ProcessMarkerEvents(List<GameEvent> events)
		{
			events.Sort(new EventNodeGroupCmp());

			GameEventGroup curGroup = null;
			GameNode curNode = null;

			for (int i = 0; i != events.Count; )
			{
				var e = events[i];
				var g = e.Group;
				var n = g.Node;

				if (g != curGroup)
					UpdateGroupData(g);

				if (n != curNode)
				{
					UpdateNodeData(n);
					m_map.ClearFullBatch();
					m_positionGroup = m_form.GetPositionTrack(n);
				}

				curNode = n;
				curGroup = g;

				if (m_currOwner != EEventOwner.Num && m_map != null && m_positionGroup != null && m_positionGroup.Events.Count != 0)
				{
					ProcessMarkerEvent(e);
					events.RemoveAt(i);
				}
				else
					++i;
			}
		}

		void ProcessPositionEvents(List<GameEvent> events)
		{
			events.Sort(new EventNodeGroupCmp());

			GameEventGroup curGroup = null;
			GameNode curNode = null;

			for (int i = 0; i != events.Count; ++i)
			{
				var e = events[i];
				var g = e.Group;
				var n = g.Node;

				if (g != curGroup)
					UpdateGroupData(g);

				if (n != curNode)
				{
					UpdateNodeData(n);

					if (!m_map.FullPaths.TryGetValue(n, out m_path))
					{
						m_path = new PathData(m_currOwner);
						m_map.FullPaths[n] = m_path;
					}
				}

				curNode = n;
				curGroup = g;

				ProcessPositionEvent(e);
			}
		}

		public void ProcessPendingEvents()
		{
			foreach (var p in m_pendingPositions)
				p.SetTag(Tags.POSTION, m_form.GetEventPosition(p));

			ProcessMarkerEvents(m_pendingEvents);

			ProcessPositionEvents(m_pendingPositions);
		}
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	class EventNodeGroupCmp : IComparer<GameEvent>
	{
		public int Compare(GameEvent e1, GameEvent e2)
		{
			if (e1.Group.Node != e2.Group.Node)
				return e1.Group.Node.Locator.TimeStamp < e2.Group.Node.Locator.TimeStamp ? -1 : 1;
			if (e1.Group.Desc.ID == e2.Group.Desc.ID)
				return 0;
			return e1.Group.Desc.ID < e2.Group.Desc.ID ? -1 : 1;
		}
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
}
