﻿using System;
using System.Collections.Generic;
using System.Text;
using SlimDX.Direct3D9;
using SlimDX;
using System.Runtime.InteropServices;
using System.Drawing;
using System.Xml;
using System.IO;
using System.Reflection;
using StatsDataSource.ObjectModel;

namespace StatsGameK01
{
	public interface IMiniMap
	{
		string Name { get; }

		string MetadataPath { get; }

		string ImagePath { get; }

		RectangleF WorldCoords { get; }

		RectangleF ImageCoords { get; }

		QuadTree<MarkerInfo> FullQTree { get; }

		QuadTree<MarkerInfo> FilteredQTree { get; }

		Dictionary<GameNode, PathData> FullPaths { get; }

		Dictionary<GameNode, PathData> FilteredPaths { get; }

		PointF ImageToWorld(PointF w);

		PointF WorldToImage(PointF w);
	}

	//////////////////////////////////////////////////////////////////////////

	public interface IMiniMapView : IMiniMap
	{
		IMiniMapRenderer Renderer { get; }

		Texture Texture { get; }

		VertexBuffer Vertices { get; }

		RenderingBatch PrepareFullBatch();

		void ClearFullBatch();

		RenderingBatch PrepareFilteredBatch();

		void ClearFilteredBatch();
	}

	//////////////////////////////////////////////////////////////////////////

	class MiniMap : IMiniMapView
	{
		#region IMiniMap

		public string Name { get; set; }

		
		public string MetadataPath { get; set; }

		public string ImagePath { get; set; }

		
		public RectangleF WorldCoords { get; set; }

		public RectangleF ImageCoords { get; set; }


		public QuadTree<MarkerInfo> FullQTree { get; private set; }

		public QuadTree<MarkerInfo> FilteredQTree { get; private set; }

		public Dictionary<GameNode, PathData> FullPaths { get; private set; }

		public Dictionary<GameNode, PathData> FilteredPaths { get; private set; }


		public PointF ImageToWorld(PointF i)
		{
			Vector2 w = ImageToWorld(new Vector2(i.X, i.Y));
			return new PointF(w.X, w.Y);
		}

		public PointF WorldToImage(PointF w)
		{
			Vector2 i = WorldToImage(new Vector2(w.X, w.Y));
			return new PointF(i.X, i.Y);
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region IMiniMapView

		public IMiniMapRenderer Renderer { get; private set; }

		public Texture Texture { get; private set; }

		public VertexBuffer Vertices { get; private set; }

		RenderingBatch FullRenderBatch;
		RenderingBatch FilteredRenderBatch;

		public RenderingBatch PrepareFullBatch()
		{
			if (FullRenderBatch == null)
		  {
				FullRenderBatch = new RenderingBatch(Renderer.Percise);
				FullQTree.AcceptVisitor(FullRenderBatch);
		  }
			return FullRenderBatch;
		}

		public void ClearFullBatch()
		{
			FullRenderBatch = null;
		}

		public RenderingBatch PrepareFilteredBatch()
		{
			if (FilteredRenderBatch == null)
			{
				FilteredRenderBatch = new RenderingBatch(Renderer.Percise);
				FilteredQTree.AcceptVisitor(FilteredRenderBatch);
			}
			return FilteredRenderBatch;
		}

		public void ClearFilteredBatch()
		{
			FilteredRenderBatch = null;
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Impl

		const float TERMINAL_SIZE = 3.5f;

		public MiniMap(IMiniMapRenderer renderer, string name, Texture tex, RectangleF rect)
		{
			Renderer = renderer;
			Name = name;
			Texture = tex;
			WorldCoords = rect;

			var desc = Texture.GetLevelDescription(0);
			ImageCoords = new RectangleF(-desc.Width * 0.5f, -desc.Height * 0.5f, desc.Width, desc.Height);

			Vertices = MiniMapRenderer.CreateBillboard(renderer.Device, ImageCoords.Width, ImageCoords.Height);

			FullQTree = new QuadTree<MarkerInfo>(ImageCoords, TERMINAL_SIZE, new MarkerQTAgent());
			FilteredQTree = new QuadTree<MarkerInfo>(ImageCoords, TERMINAL_SIZE, new MarkerQTAgent());

			FullPaths = new Dictionary<GameNode, PathData>();
			FilteredPaths = new Dictionary<GameNode, PathData>();
		}

		public Vector2 ImageToWorld(Vector2 i)
		{
			Vector2 w;
			float aspect_x = (float)WorldCoords.Width / ImageCoords.Width;
			float aspect_y = (float)WorldCoords.Height / ImageCoords.Height;

			w.X = ((float)ImageCoords.Width * 0.5f - i.Y) * aspect_x + WorldCoords.X;
			w.Y = (i.X + (float)ImageCoords.Height * 0.5f) * aspect_y + WorldCoords.Y;
			return w;
		}

		public Vector2 WorldToImage(Vector2 w)
		{
			Vector2 img;
			float aspect_x = (float)ImageCoords.Width / WorldCoords.Width;
			float aspect_y = (float)ImageCoords.Height / WorldCoords.Height;

			img.X = (w.Y - WorldCoords.Y) * aspect_y - (float)ImageCoords.Height * 0.5f;
			img.Y = (WorldCoords.X - w.X) * aspect_x + (float)ImageCoords.Width * 0.5f; 
			return img;
		}

		#endregion
	}

	////////////////////////////////////////////////////////////////////////////

	internal class MapMetaData
	{
		public string MinimapFile { get; private set; }
		public RectangleF Rect { get; private set; }

		public MapMetaData()
		{ }

		static public MapMetaData Load(string path)
		{
			if (File.Exists(path))
			{
				using (var r = XmlReader.Create(path))
				{
					MapMetaData md = new MapMetaData();
					if (md.Load(r))
						return md;
				}
			}
			return null;
		}

		public bool Load(XmlReader xml)
		{
			float startx = float.MinValue, starty = float.MinValue, endx = float.MinValue, endy = float.MinValue;

			while (xml.Read())
			{
				if (xml.NodeType == XmlNodeType.Element && xml.Name == "MiniMap")
				{
					for (int i = 0; i != xml.AttributeCount; ++i)
					{
						xml.MoveToAttribute(i);
						switch (xml.Name)
						{
							case "Filename": MinimapFile = xml.Value; break;
							case "startX": startx = float.Parse(xml.Value); break;
							case "startY": starty = float.Parse(xml.Value); break;
							case "endX": endx = float.Parse(xml.Value); break;
							case "endY": endy = float.Parse(xml.Value); break;
						}
					}

					break;
				}
			}

			if (MinimapFile == null || startx == float.MinValue || starty == float.MinValue || endx == float.MinValue || endy == float.MinValue)
				return false;

			Rect = new RectangleF(startx, starty, endx - startx, endy - starty);
			return true;
		}
	}

	////////////////////////////////////////////////////////////////////////////

	internal class MarkerQTAgent : IQTreeAgent<MarkerInfo>
	{
		public Vector2 ExtractPosition(MarkerInfo data)
		{
			return data.PositionIS;
		}
	}

	////////////////////////////////////////////////////////////////////////////

	public class RenderingBatch : IQuadTreeVisitor<MarkerInfo>
	{
		public List<Vector2>[] SpriteGroups { get; private set; }
		public List<KeyValuePair<Vector2, int>> DensityPoints { get; private set; }
		public int MaxDensity;
		bool percise;

		// TODO: refactor
		const int MAX_SPRITES = 16;

		public RenderingBatch(bool percise)
		{
			this.percise = percise;
			SpriteGroups = new List<Vector2>[MAX_SPRITES];

			for (int i = 0; i != MAX_SPRITES; ++i)
				SpriteGroups[i] = new List<Vector2>();

			DensityPoints = new List<KeyValuePair<Vector2, int>>();
			MaxDensity = 0;
		}

		public bool EnterSection(QuadSectionNode<MarkerInfo> section, EQTreeQuater quater)
		{
			// TODO: Add clipping
			return true;
		}

		public void LeaveSection(QuadSectionNode<MarkerInfo> section, EQTreeQuater quater)
		{
		}

		public void VisitTerminal(QuadTerminalNode<MarkerInfo> terminal, EQTreeQuater quater)
		{
			if (terminal.Items.Count > 0)
			{
				var agent = (MarkerQTAgent)terminal.Tree.Agent;

				int max = percise ? terminal.Items.Count : 1;

				Vector2 pos;
				for(int i = 0; i != max; ++i)
				{
					pos = agent.ExtractPosition(terminal.Items[i]);
					SpriteGroups[MarkerRenderMode.Instance.GetSpriteID(terminal.Items[i])].Add(pos);
				}

				pos = agent.ExtractPosition(terminal.Items[0]);
				DensityPoints.Add(new KeyValuePair<Vector2, int>(pos, terminal.Items.Count));
				MaxDensity = Math.Max(MaxDensity, terminal.Items.Count);
			}
		}
	}

	////////////////////////////////////////////////////////////////////////////

}
