﻿using System;
using System.Collections.Generic;
using System.Text;
using SlimDX.Direct3D9;
using SlimDX;
using System.Drawing;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using System.Threading;
using System.IO;
using System.Reflection;

namespace StatsGameK01
{
	//////////////////////////////////////////////////////////////////////////

	public interface IMiniMapRenderer
	{
		Device Device { get; }

		PresentParameters DeviceParams { get; }

		IMiniMapView Map { get; }

		Matrix ViewProj { get; }

		float Scale { get; }

		bool Percise { get; set; }


		IRenderMode ActiveMode { get; set; }

		List<IRenderMode> GetRenderModes();

		void RegisterRenderMode(IRenderMode mode);


		void Redraw();

		QuadVertex[] CreateQuad(RectangleF rect);

		QuadVertex[] CreateFullscreenQuad();
	}

	//////////////////////////////////////////////////////////////////////////

	class MiniMapRenderer : IResettable, IMiniMapRenderer
	{
		#region IMiniMapRenderer

		public Device Device { get; private set; }

		public PresentParameters DeviceParams { get; private set; }

		private IMiniMapView m_map;
		public IMiniMapView Map
		{
			get { return m_map; }
			set
			{
				m_map = value;
				RenderView.Visible = m_map != null;
				Position = Position;
				Draw();
			}
		}

		public Matrix ViewProj { get; set; }

		public float Scale { get; private set; }

		public bool Percise { get; set; }


		IRenderMode m_activeMode;
		public IRenderMode ActiveMode 
		{
			get { return m_activeMode; }
			set 
			{ 
				m_activeMode = value;
				if (m_activeMode != null && !m_activeMode.IsInitialized)
					m_activeMode.Init(this);
				Draw();
			}
		}

		List<IRenderMode> m_modes = new List<IRenderMode>();
		public List<IRenderMode> GetRenderModes()
		{
			return m_modes;
		}

		public void RegisterRenderMode(IRenderMode mode)
		{
			m_modes.Add(mode);
		}

		public void Redraw()
		{
			if (Map != null)
			{
				Map.ClearFilteredBatch();
				Map.ClearFullBatch();
			}
			this.Draw();
		}

		public QuadVertex[] CreateQuad(RectangleF rect)
		{
			QuadVertex[] fsq = new QuadVertex[] {
				new QuadVertex(new Vector4( rect.X - 0.5f,			rect.Y - 0.5f,				0.5f, 1.0f), new Vector2(0, 0) ),
        new QuadVertex(new Vector4( rect.Width - 0.5f,	rect.Y - 0.5f,				0.5f, 1.0f), new Vector2(1, 0) ),
        new QuadVertex(new Vector4( rect.X - 0.5f,			rect.Height - 0.5f,		0.5f, 1.0f), new Vector2(0, 1) ),
        new QuadVertex(new Vector4( rect.Width - 0.5f,	rect.Height - 0.5f,		0.5f, 1.0f), new Vector2(1, 1) ),
			};
			return fsq;
		}

		public QuadVertex[] CreateFullscreenQuad()
		{
			return CreateQuad(new RectangleF(0, 0, DeviceParams.BackBufferWidth, DeviceParams.BackBufferHeight));
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Properties		

		internal RenderView RenderView { get; set; }

		private PointF m_position = new PointF(0, 0);
		public PointF Position
		{
			get { return m_position; }
			set 
			{ 
				m_position = value;

				if (Map == null)
					return;

				float texW2 = (float)Map.ImageCoords.Width * 0.5f;
				float texH2 = (float)Map.ImageCoords.Height * 0.5f;
				float viewW2 = (float)RenderView.Width * 0.5f / Scale;
				float viewH2 = (float)RenderView.Height * 0.5f / Scale;
				float minx = - texW2 + viewW2;
				float miny = - texH2 + viewH2;
				float maxx = texW2 - viewW2;
				float maxy = texH2 - viewH2;
				m_position.X = ScrollHorizontal ? Math.Min(Math.Max(minx, m_position.X), maxx) : 0;
				m_position.Y = ScrollVertical ? Math.Min(Math.Max(miny, m_position.Y), maxy) : 0;

				UpdateMatrices();
			}
		}

		public bool ScrollHorizontal { get { return Map == null ? false : RenderView.Width < Map.ImageCoords.Width * Scale; } }
		public bool ScrollVertical { get { return Map == null ? false : RenderView.Height < Map.ImageCoords.Height * Scale; } }

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Initialization

		public MiniMapRenderer(RenderView renderView)
		{
			Scale = 1.0f;
			ViewProj = Matrix.Identity;
			Percise = false;

			RenderView = renderView;
			RenderView.OnDraw += Draw;
			RenderView.MouseDown += RenderView_MouseDown;
			RenderView.MouseUp += RenderView_MouseUp;
			RenderView.MouseMove += RenderView_MouseMove;
			RenderView.Parent.Parent.MouseWheel += RenderView_MouseWheel;
			RenderView.SizeChanged += RenderView_SizeChanged;

			DeviceParams = new PresentParameters()
			{
				BackBufferHeight = SystemInformation.PrimaryMonitorSize.Height,
				BackBufferWidth = SystemInformation.PrimaryMonitorSize.Width
			};

			Device = new Device(new Direct3D(), 0, DeviceType.Hardware, RenderView.Handle, CreateFlags.HardwareVertexProcessing, DeviceParams);

			Position = Position;
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Drawing

		private void UpdateMatrices()
		{
			var view = Matrix.LookAtLH(new Vector3(m_position.X, m_position.Y, -10.0f), new Vector3(m_position.X, m_position.Y, 0), new Vector3(0, 1.0f, 0));
			var proj = Matrix.OrthoLH(RenderView.Width / Scale, RenderView.Height / Scale, 1.0f, 100.0f);
			ViewProj = view * proj;
		}

		public Result OnLostDevice()
		{
			foreach (var mode in m_modes)
				mode.OnLostDevice();
			return ResultCode.Success;
		}

		public Result OnResetDevice()
		{
			foreach (var mode in m_modes)
				mode.OnResetDevice();
			return ResultCode.Success;
		}

		public void Draw()
		{
			if (!TestDevice() || !RenderView.Visible)
				return;

			ActiveMode.Draw();
		}

		private bool TestDevice()
		{
			var code = Device.TestCooperativeLevel();
			if (code == ResultCode.DeviceLost)
			{
				Thread.Sleep(1000);
				return false;
			}
			else if (code == ResultCode.DeviceNotReset)
			{
				OnLostDevice();
				DeviceParams.BackBufferHeight = SystemInformation.PrimaryMonitorSize.Height;
				DeviceParams.BackBufferWidth = SystemInformation.PrimaryMonitorSize.Width;
				Device.Reset(DeviceParams);
				OnResetDevice();
				Draw();
				return false;
			}
			return true;
		}

		public void OnResize()
		{
			RenderView_SizeChanged(null, null);
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Scrolling

		Point m_capturePoint;
		bool m_scrolled;
		private void RenderView_MouseDown(object sender, MouseEventArgs e)
		{
			RenderView.Capture = true;
			m_capturePoint = RenderView.PointToScreen(e.Location);
			m_scrolled = false;
		}

		public delegate void ClickedDeleg(Point position);
		public event ClickedDeleg Clicked;

		private void RenderView_MouseUp(object sender, MouseEventArgs e)
		{
			RenderView.Capture = false;
			RenderView.Cursor = Cursors.Default;

			if (!m_scrolled && Clicked != null)
				Clicked(e.Location);
		}

		private void RenderView_MouseMove(object sender, MouseEventArgs e)
		{
			if (RenderView.Capture)
			{
				RenderView.Cursor = Cursors.Hand;

				var pnt = RenderView.PointToScreen(e.Location);
				float dx = ((float)(m_capturePoint.X - pnt.X)) / Scale;
				float dy = ((float)(m_capturePoint.Y - pnt.Y)) / Scale;

				if (m_capturePoint != pnt)
				{
					m_capturePoint = pnt;
					m_scrolled = true;

					Position = new PointF(Position.X + dx, Position.Y - dy);
					Draw();
				}
			}
		}

		void RenderView_MouseWheel(object sender, MouseEventArgs e)
		{
			if (e.Delta != 0)
			{
				Scale += e.Delta > 0 ? 0.15f : -0.15f;
				Scale = Math.Min(Math.Max(0.1f, Scale), 3.0f);
				Position = Position;
				Draw();
			}
		}

		void RenderView_SizeChanged(object sender, EventArgs e)
		{
			// Trigger reevaluation of position
			Position = Position;
			Draw();
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Helpers
		public Vector2 ScreenToWorld(Point sspos)
		{
			Vector3 ray = Vector3.Unproject(new Vector3(sspos.X, sspos.Y, 0), 0, 0, RenderView.Width, RenderView.Height, 0, 1.0f, ViewProj);
			return new Vector2(ray.X, ray.Y);
		}

		public static VertexBuffer CreateBillboard(Device device, float w, float h)
		{
			float w2 = w * 0.5f;
			float h2 = h * 0.5f;

			var verts = new MarkerVertex[] {
				new MarkerVertex(new Vector3(-w2,	-h2,	0),	new Vector2(0, 1)),
        new MarkerVertex(new Vector3( w2,	-h2,	0),	new Vector2(1, 1)),
				new MarkerVertex(new Vector3(-w2,	 h2,	0),	new Vector2(0, 0)),
				new MarkerVertex(new Vector3( w2,	 h2,	0),	new Vector2(1, 0)) 
			};

			var buf = new VertexBuffer(device, verts.Length * MarkerVertex.Size, Usage.WriteOnly, VertexFormat.None, Pool.Managed);
			DataStream stream = buf.Lock(0, 0, LockFlags.None);
			stream.WriteRange(verts);
			buf.Unlock();

			return buf;
		}

		public static VertexBuffer CreatePolygon(Device device, int sides, float radius)
		{
			int nverts = sides + 2;
			var buf = new VertexBuffer(device, SplatVertex.Size * nverts, Usage.WriteOnly, VertexFormat.None, Pool.Managed);
			DataStream stream = buf.Lock(0, 0, LockFlags.None);

			var center = new SplatVertex(new Vector3(0, 0, 0), 1.0f);
			stream.Write(center);

			for (int i = 0; i != sides + 1; ++i)
			{
				float sector = 6.283f / (float)sides * i;
				float x = (float)Math.Cos(sector) * radius;
				float y = (float)Math.Sin(sector) * radius;
				var vert = new SplatVertex(new Vector3(x, y, 0), 0.0f);
				stream.Write(vert);
			}

			buf.Unlock();
			return buf;
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////
	}
}
