﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using StatsDataSource.Core;
using StatsGameK01.Properties;
using StatsDataSource.ObjectModel;
using System.IO;
using StatsDataSource.Storage;
using System.Diagnostics;
using SlimDX;
using StatsDataSource.Filters;

namespace StatsGameK01
{
	public partial class MinimapForm : Form, IFilteredStatsListener
	{
		//////////////////////////////////////////////////////////////////////////

		#region Members

		internal MinimapPlugin Plugin { get; set; }

		internal MiniMapRenderer Renderer { get; private set; }

		internal StatDesc TeamDesc { get; private set; }

		internal Form MainForm { get; private set; }

		#endregion

		//////////////////////////////////////////////////////////////////////////

		public MinimapForm()
		{
			InitializeComponent();
		}

		private void MinimapForm_Load(object sender, EventArgs e)
		{
			Renderer = new MiniMapRenderer(renderView);
			var mm = new MarkerRenderMode();
			Renderer.RegisterRenderMode(mm);
			Renderer.RegisterRenderMode(new DensityRenderMode());
			Renderer.ActiveMode = mm;

			Renderer.Clicked += MiniMap_Clicked;
			renderView.MouseMove += MiniMap_MouseMove;
			MainForm = (Form)Plugin.MainForm;

			Plugin.Core.ScopesChanged += Core_ScopesChanged;
			MainForm.LocationChanged += MainForm_LocationChanged;
			MainForm_LocationChanged(null, null);
			Plugin.Core.Repository.RegisterFilteredListener(this);
		}

		//////////////////////////////////////////////////////////////////////////

		#region Minimap

		private StatDesc m_map_path_desc;
		internal string GetMinimapPath(GameScope scope)
		{
			if(m_map_path_desc == null)
				m_map_path_desc = Plugin.Core.Repository.Registry.GetStateDesc("map_path");

			GameState state;
			if (m_map_path_desc != null && scope.States.TryGetValue(m_map_path_desc.ID, out state) && !string.IsNullOrEmpty(state.Value))
			{
				return Path.GetFullPath(Path.Combine(Path.Combine(Settings.Default.GamePath, "GameCrysis2"), state.Value));
			}
			return null;
		}

		private StatDesc m_map_desc;
		internal string GetMinimapName(GameScope scope)
		{
			if(m_map_desc == null)
				m_map_desc = Plugin.Core.Repository.Registry.GetStateDesc("map");

			GameState state;
			if (m_map_desc != null && scope.States.TryGetValue(m_map_desc.ID, out state) && !string.IsNullOrEmpty(state.Value))
			{
				return state.Value;
			}
			return null;
		}

		void Core_ScopesChanged(StatsCore sender, List<GameScope> scopes)
		{
			if (scopes.Count == 1)
			{
				Plugin.ActiveMap = LoadMap(scopes[0]);
			}
		}

		public void SetMap(IMiniMap map)
		{
			Plugin.ActiveMap = map;
			Renderer.Map = (MiniMap)map;
			UpdateView();
		}

		private void UpdateView()
		{
			if (!updateTimer.Enabled && Renderer.Map != null)
			{
				updateTimer.Start();
			}
		}

		private void updateTimer_Tick(object sender, EventArgs e)
		{
			var map = Renderer.Map;

			if (map != null)
			{
				var s = new FilteredTreeSynchronizer(map);
				s.Synchronize();
				Renderer.Draw();
			}
			updateTimer.Stop();
		}

		private void tabModes_SelectedIndexChanged(object sender, EventArgs e)
		{
			Renderer.ActiveMode = tabModes.SelectedIndex == 0
				? (IRenderMode)MarkerRenderMode.Instance
				: tabModes.SelectedIndex == 1
				? (IRenderMode)DensityRenderMode.Instance
				: null;
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Picking

		void GetToolTipText(List<MarkerInfo> items, out string header, out string text)
		{
			header = items.Count + " Event(s)";

			var sb = new StringBuilder(300 * items.Count);
			int e = 0;
			foreach (var it in items)
			{
				sb.Append("Type: "); sb.Append(it.Event.Group.Desc.Name); sb.Append('\n');

				sb.Append("Owner: "); sb.Append(it.Event.Group.Node.Name); sb.Append('\n');

				sb.Append("Time: ");
				sb.AppendFormat("{0:00}:{1:00}:{2:00}", it.Event.Time.Hours, it.Event.Time.Minutes, it.Event.Time.Seconds);
				sb.Append('\n');

				if (it.Event.Parameters.Count != 0)
					sb.Append("Parameters: ");

				bool first = true;
				int p = 0;
				foreach (var pair in it.Event.Parameters)
				{
					if (first)
						first = false;
					else if (p % 3 == 0)
						sb.Append(",\n");
					else
						sb.Append(", ");

					sb.Append(pair.Key); sb.Append(" = "); sb.Append(pair.Value);
					++p;
				}

				sb.Append('\n'); sb.Append('\n');

				++e;
				if (e == 4 && items.Count > 4)
				{
					sb.Append("...");
					break;
				}
			}
			text = sb.ToString().TrimEnd(new char[] { '\n' });
		}


		bool m_tooltip = false;
		Point m_ttpos;
		void MiniMap_Clicked(Point position)
		{
			if (Renderer.Map != null && !m_tooltip)
			{
				m_ttpos = position;

				var pos = Renderer.ScreenToWorld(position);

				var prox = new QuadTreeProximity<MarkerInfo>(pos, 2);
				Renderer.Map.FilteredQTree.AcceptVisitor(prox);

				if (prox.Items.Count == 0)
					return;

				string header, text;
				GetToolTipText(prox.Items, out header, out text);

				eventToolTip.ToolTipTitle = header;

				for (int i = 0; i != 2; ++i)
					eventToolTip.Show(text, renderView, position);

				m_tooltip = true;
			}
		}


		void MiniMap_MouseMove(object sender, MouseEventArgs e)
		{
			if (m_tooltip && m_ttpos != e.Location)
			{
				eventToolTip.Hide(renderView);
				m_tooltip = false;
			}
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Action export

		internal IPluginAction[] FillActions()
		{
			return new IPluginAction[] { new DelegAction("&Options/&Game path...", GamePath_Clicked) };
		}

		private void GamePath_Clicked(IPluginAction action)
		{
			folderBrowserDialog1.SelectedPath = Settings.Default.GamePath;

			if (folderBrowserDialog1.ShowDialog() == DialogResult.OK)
			{
				Settings.Default.GamePath = folderBrowserDialog1.SelectedPath;
				Settings.Default.Save();

				Core_ScopesChanged(Plugin.Core, Plugin.Core.CurrentScopes);
			}
		}

		void MainForm_LocationChanged(object sender, EventArgs e)
		{
			Location = new Point(MainForm.Location.X + MainForm.Width, MainForm.Location.Y);
		}
		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Event utils

		internal EEventArchType GetEventType(string name)
		{
			switch (name)
			{
				case "position": return EEventArchType.Path;
				case "explode": return EEventArchType.Marker;
				default: return EEventArchType.Marker_Deduced;
			}
		}

		internal EEventOwner GetEventOwner(GameEventGroup g)
		{
			if (TeamDesc == null)
				TeamDesc = Plugin.Core.Repository.Registry.GetStateDesc("team");

			if (TeamDesc != null)
			{
				if (g.Node.Name == "player")
				{
					GameState ts;
					if (g.Node.States.TryGetValue(TeamDesc.ID, out ts))
						return ts.Value == "1" ? EEventOwner.Player : EEventOwner.Player_AI_Team;
				}
				else
				{
					return EEventOwner.AI;
				}
			}

			return EEventOwner.Num;
		}

		internal Vector2 GetEventPosition(GameEvent e)
		{
			try
			{
				float fx = float.Parse(e.Parameters["x"]);
				float fy = float.Parse(e.Parameters["y"]);
				return new Vector2(fx, fy);
			}
			catch
			{
				Debug.Assert(false);
				return new Vector2(float.MinValue, float.MinValue);
			}
		}

		private StatDesc m_posDesc;
		internal StatDesc GetPosDesc()
		{
			if (m_posDesc == null)
				m_posDesc = Plugin.Core.Repository.Registry.GetEventDesc("position");

			return m_posDesc;
		}

		internal MiniMap LoadMap(GameScope scope)
		{
			var tag = scope.GetTag(Tags.MAP);
			if (tag != null)
				return (MiniMap)tag;

			var map_path = GetMinimapPath(scope);
			if (map_path == null)
				return null;

			var map_name = GetMinimapName(scope);

			IMiniMap map = Plugin.LoadMap(map_name, map_path);
			if (map == null)
			{
				MessageBox.Show(this, "Failed to load mini map in: \"" + map_path + "\",\ncheck game path settings", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
				return null;
			}

			scope.SetTag(Tags.MAP, map);
			return (MiniMap)map;
		}

		internal MarkerInfo CreateMarkerInfo(GameEvent e, Vector2 pos, MiniMap map, EEventOwner owner)
		{
			return new MarkerInfo(e, map.WorldToImage(pos), pos, owner);
		}

		internal bool GetMarkerInfo(GameEvent e, GameEventGroup posGroup, MiniMap map, EEventOwner owner, out MarkerInfo info)
		{
			object tag = e.GetTag(Tags.MARKER);
			if (tag != null)
			{
				info = (MarkerInfo)tag;
			}
			else
			{
				info = new MarkerInfo();

				var pe = posGroup.FindClosest(e.TimeMillisecs);
				if (pe == null)
					return false;

				info = CreateMarkerInfo(e, (Vector2)pe.GetTag(Tags.POSTION), map, owner);
				e.SetTag(Tags.MARKER, info);
			}

			return true;
		}

		internal GameEventGroup GetPositionTrack(GameNode n)
		{
			GameEventGroup pg;
			var posdesc = GetPosDesc();
			if (posdesc == null || !n.EventGroups.TryGetValue(posdesc.ID, out pg))
				return null;
			return pg;
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region IFilteredStatsListener

		private bool inModificationBatch = false;

		public void ModificationBatchStarted()
		{
			inModificationBatch = true;
		}

		List<GameEvent> PendingEvents = new List<GameEvent>();

		public void ModificationBatchEnded(IncrementalUpdate update)
		{
			inModificationBatch = false;

			if (update == null)
			{
				var v = new GlobalTreeUpdateVisitor(this);
				Plugin.Core.Repository.AcceptVisitor(v);
				UpdateView();
			}
			else if(update.AddedEvents.Count != 0)
			{
				var pd = GetPosDesc();
				List<GameEvent> pos = new List<GameEvent>();

				foreach (var e in update.AddedEvents)
				{
					if (e.Group.Desc != pd)
						PendingEvents.Add(e);
					else
						pos.Add(e);
				}

				var v = new GlobalTreeIncrementalUpdater(this, PendingEvents, pos);
				v.ProcessPendingEvents();
				UpdateView();
			}
		}

		public void RepositoryNodeAdded(GameNode node)
		{
		}

		public void RepositoryStateAdded(GameState state)
		{
		}

		public void RepositoryEventGroupAdded(GameEventGroup group)
		{
		}

		public void RepositoryEventAdded(GameEvent evnt)
		{
			if (inModificationBatch)
				return;

			if(evnt.FilterState != EFilterState.Failed)
				UpdateView();
		}

		public void RepositoryFilterChanged(CompoundFilter active, EFilterUpdateType type)
		{
			UpdateView();
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Mode settings

		private void cbTotalDensity_CheckedChanged(object sender, EventArgs e)
		{
			DensityRenderMode.UseTotalDensity = cbTotalDensity.Checked;
			UpdateView();
		}

		#endregion

	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

}
