﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Drawing;
using StatsDataSource.ObjectModel;
using System.Diagnostics;
using SlimDX;

namespace StatsGameK01
{
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	public interface IQTreeAgent<T>
	{
		Vector2 ExtractPosition(T data);
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	#region Nodes

	//////////////////////////////////////////////////////////////////////////

	public enum EQTreeNodeType { Section, Terminal }
	public enum EQTreeQuater { NE, SE, SW, NW, Root }

	//////////////////////////////////////////////////////////////////////////

	public abstract class QuadNode<T>
	{
		private WeakReference m_tree;
		public QuadTree<T> Tree
		{ 
			get { return m_tree != null ? (QuadTree<T>)m_tree.Target : null; }
			protected set { m_tree = new WeakReference(value); }
		}

		private WeakReference m_parent;
		public QuadSectionNode<T> Parent
		{
			get { return m_parent != null ? (QuadSectionNode<T>)m_parent.Target : null; }
			protected set { m_parent = new WeakReference(value); }
		}

		public readonly RectangleF Area;

		public readonly EQTreeNodeType Type;

		public QuadNode(QuadTree<T> tree, EQTreeNodeType type, RectangleF area, QuadSectionNode<T> parent)
		{
			Tree = tree;
			Type = type;
			Area = area;

			if (parent != null)
				Parent = parent;
		}

		public abstract void AcceptVisitor(IQuadTreeVisitor<T> visitor, EQTreeQuater quater);
	}

	//////////////////////////////////////////////////////////////////////////

	public class QuadSectionNode<T> : QuadNode<T>
	{
		public QuadNode<T>[] Quaters { get; private set; }

		public QuadSectionNode(QuadTree<T> tree, RectangleF area, QuadSectionNode<T> parent)
			: base(tree, EQTreeNodeType.Section, area, parent)
		{
		}

		public bool isEmpty()
		{
			return Quaters == null;
		}

		public override void AcceptVisitor(IQuadTreeVisitor<T> visitor, EQTreeQuater quater)
		{
			if (visitor.EnterSection(this, quater) && !isEmpty())
			{
				for (int i = 0; i != 4; ++i)
					if (Quaters[i] != null)
						Quaters[i].AcceptVisitor(visitor, (EQTreeQuater)i);
			}
			visitor.LeaveSection(this, quater);
		}

		public void Split(float terminalSize)
		{
			Quaters = new QuadNode<T>[4];

			float w2 = Area.Width * 0.5f;
			float h2 = Area.Height * 0.5f;

			if (w2 <= terminalSize && h2 <= terminalSize)
			{
				Quaters[0] = new QuadTerminalNode<T>(Tree, new RectangleF(Area.X + w2, Area.Y, w2, h2), this);
				Quaters[1] = new QuadTerminalNode<T>(Tree, new RectangleF(Area.X + w2, Area.Y + h2, w2, h2), this);
				Quaters[2] = new QuadTerminalNode<T>(Tree, new RectangleF(Area.X, Area.Y + h2, w2, h2), this);
				Quaters[3] = new QuadTerminalNode<T>(Tree, new RectangleF(Area.X, Area.Y, w2, h2), this);
			}
			else
			{
				Quaters[0] = new QuadSectionNode<T>(Tree, new RectangleF(Area.X + w2, Area.Y, w2, h2), this);
				Quaters[1] = new QuadSectionNode<T>(Tree, new RectangleF(Area.X + w2, Area.Y + h2, w2, h2), this);
				Quaters[2] = new QuadSectionNode<T>(Tree, new RectangleF(Area.X, Area.Y + h2, w2, h2), this);
				Quaters[3] = new QuadSectionNode<T>(Tree, new RectangleF(Area.X, Area.Y, w2, h2), this);
			}

		}
	}

	//////////////////////////////////////////////////////////////////////////

	public class QuadTerminalNode<T> : QuadNode<T>
	{
		public List<T> Items { get; private set; }

		public QuadTerminalNode(QuadTree<T> tree, RectangleF area, QuadSectionNode<T> parent)
			: base(tree, EQTreeNodeType.Terminal, area, parent)
		{
			Items = new List<T>();
		}

		public override void AcceptVisitor(IQuadTreeVisitor<T> visitor, EQTreeQuater quater)
		{
			visitor.VisitTerminal(this, quater);
		}
	}

	#endregion

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	#region Visitors

	public interface IQuadTreeVisitor<T>
	{
		bool EnterSection(QuadSectionNode<T> section, EQTreeQuater quater);
		void LeaveSection(QuadSectionNode<T> section, EQTreeQuater quater);
		void VisitTerminal(QuadTerminalNode<T> terminal, EQTreeQuater quater);
	}

	//////////////////////////////////////////////////////////////////////////

	public class QTreeInserter<T> : IQuadTreeVisitor<T>
	{
		public QuadTerminalNode<T> place;
		public float terminalSize;
		public T to_insert;

		private bool isInArea(RectangleF area, Vector2 pos)
		{
			float x = pos.X - area.X;
			float y = pos.Y - area.Y;
			return x >= 0 && y >= 0 && x < area.Width && y < area.Height;
		}

		public bool EnterSection(QuadSectionNode<T> section, EQTreeQuater quater)
		{
			if (!isInArea( section.Area, section.Tree.Agent.ExtractPosition(to_insert) ))
				return false;

			if (section.isEmpty())
				section.Split(terminalSize);

			return true;
		}

		public void LeaveSection(QuadSectionNode<T> section, EQTreeQuater quater)
		{
		}

		public void VisitTerminal(QuadTerminalNode<T> terminal, EQTreeQuater quater)
		{
			if (isInArea(terminal.Area, terminal.Tree.Agent.ExtractPosition(to_insert)))
			{
				place = terminal;
				terminal.Items.Add(to_insert);
			}
		}
	}

	//////////////////////////////////////////////////////////////////////////

	public class QuadTreeProximity<T> : IQuadTreeVisitor<T>
	{
		private Vector2 m_position;
		private float m_tolerance;
		public List<T> Items { get; private set; }


		public QuadTreeProximity(Vector2 position, float tolerance)
		{
			m_position = position;
			m_tolerance = tolerance;
			Items = new List<T>();
		}

		private bool isInArea(RectangleF area, Vector2 pos, float tolerance)
		{
			float x = pos.X - area.X;
			float y = pos.Y - area.Y;
			return x + tolerance >= 0 && y + tolerance >= 0 && x - tolerance < area.Width && y - tolerance < area.Height;
		}

		public bool EnterSection(QuadSectionNode<T> section, EQTreeQuater quater)
		{
			return isInArea(section.Area, m_position, m_tolerance);
		}

		public void LeaveSection(QuadSectionNode<T> section, EQTreeQuater quater)
		{
		}

		public void VisitTerminal(QuadTerminalNode<T> terminal, EQTreeQuater quater)
		{
			foreach (T item in terminal.Items)
			{
				Vector2 pos = terminal.Tree.Agent.ExtractPosition(item);
				if ((pos - m_position).Length() <= m_tolerance)
					Items.Add(item);
			}
		}
	}

	//////////////////////////////////////////////////////////////////////////

	public class QTreeCleaner<T> : IQuadTreeVisitor<T>
	{
		public bool EnterSection(QuadSectionNode<T> section, EQTreeQuater quater) { return true; }
		public void LeaveSection(QuadSectionNode<T> section, EQTreeQuater quater) { }
		public void VisitTerminal(QuadTerminalNode<T> terminal, EQTreeQuater quater)
		{
			terminal.Items.Clear();
		}
	}

	//////////////////////////////////////////////////////////////////////////

	#endregion

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	#region QuadTree

	public class QuadTree<T>
	{
		public QuadSectionNode<T> Root { get; private set; }

		public IQTreeAgent<T> Agent { get; set; }

		private float m_terminalSize;
		public float TerminalSize
		{
			get { return m_terminalSize; }
			set
			{
				if (m_terminalSize != value)
				{
					m_terminalSize = value;
					ClearPartition();
				}
			}
		}

		public QuadTree(RectangleF area, float terminalSize, IQTreeAgent<T> agent)
		{
			Agent = agent;
			m_terminalSize = terminalSize;
			Root = new QuadSectionNode<T>(this, area, null);
		}

		private QTreeCleaner<T> m_cleaner = new QTreeCleaner<T>();
		public void ClearContent()
		{
			AcceptVisitor(m_cleaner);
		}

		public void ClearPartition()
		{
			var area = Root.Area;
			Root = null;
			Root = new QuadSectionNode<T>(this, area, null);
		}

		private QTreeInserter<T> m_inserter = new QTreeInserter<T>();
		public QuadTerminalNode<T> Insert(T data)
		{
			m_inserter.place = null;
			m_inserter.terminalSize = TerminalSize;
			m_inserter.to_insert = data;

			AcceptVisitor(m_inserter);
			return m_inserter.place;
		}

		public void AcceptVisitor(IQuadTreeVisitor<T> visitor)
		{
			Root.AcceptVisitor(visitor, EQTreeQuater.Root);
		}
	}

	#endregion

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
}
