﻿using System;
using System.Collections.Generic;
using System.Text;
using SlimDX.Direct3D9;
using System.Drawing;
using SlimDX;
using System.IO;
using System.Reflection;

namespace StatsGameK01
{
	//////////////////////////////////////////////////////////////////////////

	public interface IRenderMode : IResettable
	{
		string Name { get; }

		bool IsInitialized { get; }

		void Init(IMiniMapRenderer renderer);

		void Draw();
	}


	//////////////////////////////////////////////////////////////////////////

	public interface IMarkerRenderMode : IRenderMode
	{
		void AddViewCallback(IMarkerViewCallback clb);

		void RemoveViewCallback(IMarkerViewCallback clb);
	}

	public interface IMarkerViewCallback
	{
		int GetSpriteID(MarkerInfo mi);

		Color GetPathColor(PathData path);
	}

	//////////////////////////////////////////////////////////////////////////

	class MarkerRenderMode : IMarkerRenderMode
	{
		#region IRenderMode Members

		public static MarkerRenderMode Instance;

		public string Name { get { return "Markers"; } }

		IMiniMapRenderer m_renderer;
		Line PathLine;
		Effect MapEffect;
		Effect MarkEffect;
		Texture Sprites;
		VertexBuffer SpriteBillboard;

		int NumSprites;

		public bool IsInitialized { get; set; }

		public MarkerRenderMode()
		{
			Instance = this;
			AddViewCallback(new DefaultCallback());
		}

		public void Init(IMiniMapRenderer renderer)
		{
			m_renderer = renderer;
			string resources = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "Resources");

			PathLine = new Line(m_renderer.Device);

			MapEffect = Effect.FromFile(m_renderer.Device, Path.Combine(resources, "MapEffect.fx"), ShaderFlags.None);
			MarkEffect = Effect.FromFile(m_renderer.Device, Path.Combine(resources, "MarkerEffect.fx"), ShaderFlags.None);

			Sprites = Texture.FromFile(m_renderer.Device, Path.Combine(resources, "Markers.dds"));
			var ld = Sprites.GetLevelDescription(0);
			NumSprites = 4;//ld.Width >> 4;

			SpriteBillboard = MiniMapRenderer.CreateBillboard(m_renderer.Device, 16.0f, 16.0f);

			IsInitialized = true;
		}

		public Result OnLostDevice()
		{
			MapEffect.OnLostDevice();
			MarkEffect.OnLostDevice();
			PathLine.OnLostDevice();
			return ResultCode.Success;
		}

		public Result OnResetDevice()
		{
			MapEffect.OnResetDevice();
			MarkEffect.OnResetDevice();
			PathLine.OnResetDevice();
			return ResultCode.Success;
		}

		public void Draw()
		{
			m_renderer.Device.Clear(ClearFlags.Target, Color.Black, 1.0f, 0);

			if (m_renderer.Map != null)
			{
				m_renderer.Device.BeginScene();

				RenderingBatch batch = m_renderer.Map.PrepareFilteredBatch();

				DrawMap();
				DrawPaths();

				DrawMarkers(batch);

				m_renderer.Device.EndScene();
			}

			m_renderer.Device.Present();
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region IMarkerRenderMode Members

		List<IMarkerViewCallback> Callbacks = new List<IMarkerViewCallback>();

		public int GetSpriteID(MarkerInfo mi)
		{
			int id = -1;
			for (int i = Callbacks.Count - 1; i >= 0; --i)
			{
				id = Callbacks[i].GetSpriteID(mi);
				if (id != -1)
					break;
			}

			return id == -1 ? 0 : id;
		}

		public Color GetPathColor(PathData path)
		{
			Color clr = Color.Black;
			for (int i = Callbacks.Count - 1; i >= 0; --i)
			{
				clr = Callbacks[i].GetPathColor(path);
				if (clr != Color.Black)
					break;
			}

			return clr;
		}

		public void AddViewCallback(IMarkerViewCallback clb)
		{
			Callbacks.Add(clb);
			if(m_renderer != null)
				m_renderer.Redraw();
		}

		public void RemoveViewCallback(IMarkerViewCallback clb)
		{
			Callbacks.Remove(clb);
			if (m_renderer != null)
				m_renderer.Redraw();
		}

		class DefaultCallback : IMarkerViewCallback
		{
			public int GetSpriteID(MarkerInfo mi)
			{
				return mi.Owner == EEventOwner.Player 
					? 0 
					: mi.Owner == EEventOwner.AI 
					? 1 
					: 2;
			}

			public Color GetPathColor(PathData path)
			{
				return path.Owner == EEventOwner.AI 
					? Color.Blue
					: path.Owner == EEventOwner.Player
					? Color.Red 
					: Color.RoyalBlue;
			}
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Impl

		private void DrawMap()
		{
			m_renderer.Device.SetStreamSource(0, m_renderer.Map.Vertices, 0, MapVertex.Size);
			m_renderer.Device.VertexDeclaration = MapVertex.GetDeclaration(m_renderer.Device);

			MapEffect.Technique = "tech_main";
			MapEffect.SetTexture("texture0", m_renderer.Map.Texture);
			MapEffect.SetValue<Matrix>("mWVP", m_renderer.ViewProj);

			int npass = MapEffect.Begin();
			for (int i = 0; i != npass; ++i)
			{
				MapEffect.BeginPass(0);
				m_renderer.Device.DrawPrimitives(PrimitiveType.TriangleStrip, 0, 2);
				MapEffect.EndPass();
			}

			MapEffect.End();
		}

		private void DrawPaths()
		{
			m_renderer.Device.SetRenderState(RenderState.ZEnable, false);

			foreach (var p in m_renderer.Map.FilteredPaths)
				if (p.Value.Points.Count > 1)
					DrawPath(p.Value.Points.ToArray(), GetPathColor(p.Value));

			m_renderer.Device.SetRenderState(RenderState.ZEnable, true);
		}

		private void DrawPath(Vector3[] points, Color clr)
		{
			if (points.Length < 2)
				return;

			PathLine.Width = 4.0f;

			PathLine.Begin();
			PathLine.DrawTransformed(points, m_renderer.ViewProj, clr);
			PathLine.End();
		}

		private void DrawMarkers(RenderingBatch batch)
		{
			m_renderer.Device.SetStreamSource(0, SpriteBillboard, 0, MarkerVertex.Size);
			m_renderer.Device.VertexDeclaration = MarkerVertex.GetDeclaration(m_renderer.Device);
			MarkEffect.Technique = "tech_main";
			MarkEffect.SetTexture("texture0", Sprites);
			MarkEffect.SetValue<Matrix>("mWVP", m_renderer.ViewProj);
			MarkEffect.SetValue<float>("fNumSprites", (float)NumSprites);
			MarkEffect.SetValue<float>("fScale", m_renderer.Scale);

			for (int i = 0; i != NumSprites; ++i)
				DrawMarker(i, batch.SpriteGroups[i]);
		}

		private void DrawMarker(int sprite, List<Vector2> poss)
		{
			MarkEffect.SetValue<float>("fSprite", (float)sprite);

			int npass = MarkEffect.Begin();
			for (int i = 0; i != npass; ++i)
			{
				MarkEffect.BeginPass(0);

				foreach (var e in poss)
				{
					MarkEffect.SetValue<Vector4>("vPosition", new Vector4(e.X, e.Y, 0, 1));
					MarkEffect.CommitChanges();

					m_renderer.Device.DrawPrimitives(PrimitiveType.TriangleStrip, 0, 2);
				}
				MarkEffect.EndPass();
			}

			MarkEffect.End();
		}

		#endregion
	}


	//////////////////////////////////////////////////////////////////////////


	class DensityRenderMode : IRenderMode
	{
		#region IRenderMode Members

		public static DensityRenderMode Instance;

		const int SPLAT_ANGLES = 9;

		public string Name { get { return "Density map"; } }

		IMiniMapRenderer m_renderer;

		VertexBuffer Splat;
		Effect MapEffect;
		Effect SplattingEffect;
		Effect DensityEffect;
		Texture DensityMap;
		Texture Palette;

		public bool IsInitialized { get; set; }

		public DensityRenderMode()
		{
			Instance = this;
		}

		public void Init(IMiniMapRenderer renderer)
		{
			m_renderer = renderer;
			string resources = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "Resources");

			MapEffect = Effect.FromFile(m_renderer.Device, Path.Combine(resources, "MapEffect.fx"), ShaderFlags.None);
			SplattingEffect = Effect.FromFile(m_renderer.Device, Path.Combine(resources, "SplatEffect.fx"), ShaderFlags.None);
			DensityEffect = Effect.FromFile(m_renderer.Device, Path.Combine(resources, "DensityEffect.fx"), ShaderFlags.None);

			Palette = Texture.FromFile(m_renderer.Device, Path.Combine(resources, "Palette.dds"));
			DensityMap = CreateDensityRT(m_renderer.Device, m_renderer.DeviceParams.BackBufferWidth, m_renderer.DeviceParams.BackBufferHeight);

			Splat = MiniMapRenderer.CreatePolygon(m_renderer.Device, SPLAT_ANGLES, 12.0f);

			IsInitialized = true;
		}

		public void Draw()
		{
			var backBuf = m_renderer.Device.GetRenderTarget(0);
			m_renderer.Device.Clear(ClearFlags.Target, Color.Black, 1.0f, 0);
			m_renderer.Device.SetRenderTarget(0, DensityMap.GetSurfaceLevel(0));
			m_renderer.Device.Clear(ClearFlags.Target, new Color4(0, 0, 0, 0), 1.0f, 0);

			m_renderer.Device.SetRenderState(RenderState.ZFunc, Compare.Never);

			if (m_renderer.Map != null)
			{
				m_renderer.Device.BeginScene();

				var batch = m_renderer.Map.PrepareFilteredBatch();

				DrawSplats();
				m_renderer.Device.SetRenderTarget(0, backBuf);
				DrawMap();
				MapDensity();

				m_renderer.Device.EndScene();
			}

			m_renderer.Device.Present();
		}

		public Result OnLostDevice()
		{
			SplattingEffect.OnLostDevice();
			DensityEffect.OnLostDevice();
			DensityMap.Dispose();
			DensityMap = null;
			return ResultCode.Success;
		}

		public Result OnResetDevice()
		{
			SplattingEffect.OnResetDevice();
			DensityEffect.OnResetDevice();
			DensityMap = CreateDensityRT(m_renderer.Device, m_renderer.DeviceParams.BackBufferWidth, m_renderer.DeviceParams.BackBufferHeight);
			return ResultCode.Success;
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Impl

		private Texture CreateDensityRT(Device device, int width, int height)
		{
			return new Texture(device, width, height, 1, Usage.RenderTarget, Format.A8R8G8B8, Pool.Default);
		}

		public static bool UseTotalDensity = false;

		private void DrawSplats()
		{
			var global_batch = m_renderer.Map.PrepareFullBatch();
			var filter_batch = m_renderer.Map.PrepareFilteredBatch();

			m_renderer.Device.SetStreamSource(0, Splat, 0, SplatVertex.Size);
			m_renderer.Device.VertexDeclaration = SplatVertex.GetDeclaration(m_renderer.Device);
			SplattingEffect.Technique = "tech_main";
			SplattingEffect.SetValue<Matrix>("mWVP", m_renderer.ViewProj);
			float maxDensity = UseTotalDensity ? (float)global_batch.MaxDensity : (float)filter_batch.MaxDensity;
			SplattingEffect.SetValue<float>("fMaxDensity", maxDensity);

			int npass = SplattingEffect.Begin();
			for (int i = 0; i != npass; ++i)
			{
				SplattingEffect.BeginPass(0);

				foreach (var splat in filter_batch.DensityPoints)
				{
					SplattingEffect.SetValue<Vector2>("vPosition", splat.Key);
					SplattingEffect.SetValue<float>("fSplatDensity", (float)splat.Value);
					SplattingEffect.CommitChanges();

					m_renderer.Device.DrawPrimitives(PrimitiveType.TriangleFan, 0, SPLAT_ANGLES);
				}

				SplattingEffect.EndPass();
			}

			SplattingEffect.End();
		}

		private void DrawMap()
		{
			m_renderer.Device.SetStreamSource(0, m_renderer.Map.Vertices, 0, MapVertex.Size);
			m_renderer.Device.VertexDeclaration = MapVertex.GetDeclaration(m_renderer.Device);

			MapEffect.Technique = "tech_main";
			MapEffect.SetTexture("texture0", m_renderer.Map.Texture);
			MapEffect.SetValue<Matrix>("mWVP", m_renderer.ViewProj);

			int npass = MapEffect.Begin();
			for (int i = 0; i != npass; ++i)
			{
				MapEffect.BeginPass(0);
				m_renderer.Device.DrawPrimitives(PrimitiveType.TriangleStrip, 0, 2);
				MapEffect.EndPass();
			}

			MapEffect.End();
		}

		private void MapDensity()
		{
			var quad = m_renderer.CreateFullscreenQuad();
			m_renderer.Device.VertexDeclaration = QuadVertex.GetDeclaration(m_renderer.Device);
			m_renderer.Device.DrawUserPrimitives(PrimitiveType.TriangleStrip, 2, quad);

			DensityEffect.Technique = "tech_main";
			DensityEffect.SetTexture("texIntensity", DensityMap);
			DensityEffect.SetTexture("texPalette", Palette);
			DensityEffect.SetValue<Vector2>("sampleRatio", new Vector2(1.0f / (float)m_renderer.DeviceParams.BackBufferWidth, 1.0f / (float)m_renderer.DeviceParams.BackBufferHeight));
			DensityEffect.Begin();
			DensityEffect.BeginPass(0);

			m_renderer.Device.DrawPrimitives(PrimitiveType.TriangleStrip, 0, 2);

			DensityEffect.EndPass();
			DensityEffect.End();
		}

		#endregion
	}


	//////////////////////////////////////////////////////////////////////////
}
