using System;
using System.Xml.Serialization;
using System.IO;
using System.Globalization;
using System.Collections.Generic;

namespace StatsParse
{
	using TRawSessionRoot = stats_session;

	public static class CExtractSession
	{
		public static void Do(string filename, CSessionRoot res)
		{
			XmlSerializer serializer = new XmlSerializer(typeof(TRawSessionRoot));
			using (FileStream stream = new FileStream(filename, FileMode.Open, FileAccess.Read))
			{
				TRawSessionRoot rawDataRoot = (TRawSessionRoot)serializer.Deserialize(stream);
				res.MakeFrom(rawDataRoot);
			}
		}
	};

	public interface IActorLife
	{
		string Name { get; }
		uint EntityId { get; }
		int Team { get; }

		Dictionary<string, ITimelineValue[]> EventTimelines { get; }

		bool GetLifetime(ref int begin, ref int end);
	}

	public class CActorLife : IActorLife
	{
		protected string m_name;
		protected uint m_entityId;
		protected Dictionary<string, ITimelineValue[]> m_eventTimelines = new Dictionary<string, ITimelineValue[]>();

		protected CActorLife(string name, string entityId, timelines tlSection)
		{
			m_name = name;
			try
			{
				m_entityId = Convert.ToUInt32(entityId);
			}
			catch (System.FormatException)
			{
				System.Diagnostics.Debug.Assert(false);//System.Windows.Forms.MessageBox.Show("Invalid typecast");
				m_entityId = 0;
			}

			int tlineCnt = tlSection.timeline.Length;
			for (int tn = 0; tn < tlineCnt; ++tn)
			{
				timelinesTimelineVal[] inp = tlSection.timeline[tn].val;
				string nm = tlSection.timeline[tn].name;
				CSessionRoot.ExtractTimeline(inp, nm, this);
			}
		}

		public bool GetLifetime(ref int begin, ref int end)
		{
			ITimelineValue[] lifeTL;
			if (EventTimelines.TryGetValue("lifetime", out lifeTL))
			{
				if (lifeTL.Length != 2)
					return false;

				begin = lifeTL[0].GetTime();
				end = lifeTL[1].GetTime();
				return true;
			}
			return false;
		}

		public string Name
		{ 
			get
			{
				return m_name; 
			}
		}
		
		public uint EntityId
		{ 
			get
			{
				return m_entityId; 
			}
		}

		virtual public int Team
		{
			get
			{
				return 0;
			}
		}

		public Dictionary<string, ITimelineValue[]> EventTimelines
		{
			get
			{
				return m_eventTimelines;
			}
		}
	};

	public class CPlayerLife : CActorLife
	{
		private int m_team;
		private uint m_profileId;

		public CPlayerLife(string name, string team, string profileId, string entityId, timelines tlSection)
			: base(name, entityId, tlSection)
		{
			try
			{
				m_team = Convert.ToInt32(team);
				m_profileId = Convert.ToUInt32(profileId);
			}
			catch (System.FormatException)
			{
				System.Diagnostics.Debug.Assert(false);//System.Windows.Forms.MessageBox.Show("Invalid typecast");
				m_team = 0;
				m_profileId = 0;
			}
		}

		override public int Team
		{
			get
			{
				return m_team;
			}
		}
	};

	public class CAiActorLife : CActorLife
	{
		public CAiActorLife (string name, string entityId, timelines tlSection)
			: base(name, entityId, tlSection)
		{
		}
	};

	public class CTeamData
	{
		public string m_name;
		public int m_id;
		public ITimelineValue[] m_mapEventTimeline;
	};

	public class CRoundData
	{
		public int m_begin, m_end, m_winner;
		public CPlayerLife[] m_playerLifes;
		public CAiActorLife[] m_aiActorLifes;
	};

	public class CSessionRoot
	{
		public string m_mapName;
		public string m_mapPath;
		public CTeamData[] m_teamData;
		public CRoundData[] m_roundData;

		public void MakeFrom(TRawSessionRoot val)
		{
			m_mapName = val.map;
			m_mapPath = val.map_path;
			if ( m_mapPath == null )
				m_mapPath = "";

			MakeTeamData(val.team);
			MakeRoundData(val.round);
		}

		void MakeTeamData(stats_sessionTeam[] team)
		{
			if (team == null)
				return;
			int teamCnt = team.Length;
			m_teamData = new CTeamData[teamCnt];
			for (int i = 0; i < teamCnt; ++i)
			{
				m_teamData[i] = new CTeamData();

				m_teamData[i].m_name = "";
				m_teamData[i].m_id = i + 1;

				if (team[i].name != null)
					m_teamData[i].m_name = team[i].name;

				if (team[i].id != null)
				{
					try
					{
						m_teamData[i].m_id = Convert.ToInt32(team[i].id);
					}
					catch (System.FormatException)
					{
						System.Diagnostics.Debug.Assert(false);//System.Windows.Forms.MessageBox.Show("Invalid typecast");
						return;
					}
				}

				timelines section = team[i].timelines[0];
				if (section.timeline == null)
					continue;
				
				int tlineCnt = section.timeline.Length;

				for (int tn = 0; tn < tlineCnt; ++tn)
				{
					switch (section.timeline[tn].name)
					{
						case "event":
						{
							int eventCnt = section.timeline[tn].val.Length;
							m_teamData[i].m_mapEventTimeline = new CNamedTimelineValue[eventCnt];
							for (int j = 0; j < eventCnt; ++j)
							{
								timelinesTimelineVal from = section.timeline[tn].val[j];
								CNamedTimelineValue to = new CNamedTimelineValue();
								to.MakeFrom(from);
								m_teamData[i].m_mapEventTimeline[j] = to;
							}
						} break;

						//add another team's timelines here
					}
				}
				
			}
		}

		static ITimelineValue NewTimelineValue( string name )
		{
			switch (name)
			{
				case "position":			return new CPosTimelineValue();
				case "kill":					return new CKillTimelineValue();
				case "shot":					return new CShotTimelineValue();
				case "throw":					return new CThrowTimelineValue();
				case "hit":						return new CHitTimelineValue();
				case "melee":					return new CMeleeTimelineValue();
				case "melee_hit":			return new CMeleeHitTimelineValue();
				case "death":					return new CDeathTimelineValue();
				case "attachment":		return new CAttachmentTimelineValue();
				case "kick":					return new CNamedTimelineValue();
				case "explode":				return new CExplodeTimelineValue();
				case "lookdir":				break;																		// no event for this yet (avoid cluttering map)
		
				default:							return new CPrmTimelineValue();						// all other events are just position / time / 1 param
			}
			return null;			
		}

		static public bool ExtractTimeline(timelinesTimelineVal[] input, string name, IActorLife life)
		{
			int cnt = input.Length;
			ITimelineValue[] output = new ITimelineValue[cnt];
			for (int k = 0; k < cnt; ++k)
			{
				timelinesTimelineVal from = input[k];
				ITimelineValue to = NewTimelineValue( name );
				if (to != null)
				{
					to.MakeFrom(from);
					output[k] = to;
				}
				else
				{
					return false;
				}
			}

			life.EventTimelines[name] = output;
			return true;
		}

		void MakeRoundData(stats_sessionRound[] round)
		{
			if (round == null)
				return;

			int roundCnt = round.Length;
			m_roundData = new CRoundData[roundCnt];
			for (int i = 0; i < roundCnt; ++i)
			{
				m_roundData[i] = new CRoundData();
				
				try
				{
					m_roundData[i].m_begin = Convert.ToInt32(round[i].begin);
					m_roundData[i].m_end = Convert.ToInt32(round[i].end);
					m_roundData[i].m_winner = Convert.ToInt32(round[i].winner);
				}
				catch (System.FormatException)
				{
					System.Diagnostics.Debug.Assert(false);//System.Windows.Forms.MessageBox.Show("Invalid typecast");
					return;
				}

				if (round[i].player == null)
					continue;

				int plLifesCnt = round[i].player.Length;
				m_roundData[i].m_playerLifes = new CPlayerLife[plLifesCnt];
				for (int j = 0; j < plLifesCnt; ++j)
				{
					stats_sessionRoundPlayer srp = round[i].player[j];
					m_roundData[i].m_playerLifes[j] = new CPlayerLife( srp.name, srp.team, srp.profile_id, srp.entity_id, srp.timelines[0] );
				}

				if (round[i].ai_actor == null)
					continue;

				int aiLifesCnt = round[i].ai_actor.Length;
				m_roundData[i].m_aiActorLifes = new CAiActorLife[aiLifesCnt];
				for (int j = 0; j < aiLifesCnt; ++j)
				{
					stats_sessionRoundAi_actor sra = round[i].ai_actor[j];
					m_roundData[i].m_aiActorLifes[j] = new CAiActorLife(sra.name, sra.entity_id, sra.timelines[0]);
				}
			}
		}
	};
}