using System;
using System.Collections.Generic;
using System.Text;
using System.Globalization;
using System.Reflection;

namespace StatsParse
{
	using TRawTlineVal = timelinesTimelineVal;

	public static class CFloatingPointFormat
	{
		public static NumberFormatInfo Get()
		{
			NumberFormatInfo info = (NumberFormatInfo)CultureInfo.CurrentCulture.NumberFormat.Clone();
			info.NumberDecimalSeparator = ".";
			return info;
		}
	};

	public class Vec2
	{
		public float x, y;

		public Vec2(float x, float y)
		{
			this.x = x;
			this.y = y;
		}

		public Vec2()
		: this(0, 0)
		{
		}

		public static float Distance( Vec2 l, Vec2 r )
		{
			float dx = l.x - r.x;
			float dy = l.y - r.y;
			
			dx *= dx;
			dy *= dy;

			return (float)Math.Sqrt( dx + dy );
		}
	};

	public class Vec3 : Vec2
	{
		public float z;
		
		public Vec3(float x, float y, float z)
		: base(x, y)
		{
			this.z = z;
		}

		public Vec3()
		: this( 0, 0, 0)
		{
		}

		public Vec2 GetVec2()
		{
			return (Vec2)this;
		}
	};

	public interface ITimelineValue
	{
		void MakeFrom(TRawTlineVal from);
		int GetTime();
		Dictionary<string, object> GetFieldMap();
		IList<string> GetFieldNames();
		IList<string> GetFieldsTextRepresentation( Dictionary<uint, string> entityAliases );
	}

	public class CTimelineValueBase : ITimelineValue
	{
		public int m_time;
		public virtual void MakeFrom(TRawTlineVal from)
		{
			try
			{
				m_time = Convert.ToInt32(from.time);
			}
			catch (System.FormatException)
			{
				System.Diagnostics.Debug.Assert("Invalid typecast : CTimelineValueBase".Length == 0);
				return;
			}
		}

		public int GetTime()
		{
			return m_time;
		}

		public virtual Dictionary<string, object> GetFieldMap()
		{
			FieldInfo[] fInf = this.GetType().GetFields();

			Dictionary<string, object> fieldMap = new Dictionary<string, object>();
			foreach (FieldInfo fi in fInf)
			{
				string fieldName = fi.Name.Substring(2);
				if (fieldName != "time")
				{
					object fVal = fi.GetValue(this);
					if (fVal != null)
						fieldMap[fieldName] = fVal;
				}
			}

			return fieldMap;
		}

		public virtual IList<string> GetFieldNames()
		{
			Dictionary<string, object> fieldMap = GetFieldMap();
			IList<string> content = new List<string>();

			foreach (KeyValuePair<string, object> kvp in fieldMap)
				content.Add(kvp.Key);

			return content;
		}

		public virtual IList<string> GetFieldsTextRepresentation( Dictionary<uint, string> entityAliases )
		{
			Dictionary<string, object> fieldsToReplace = new Dictionary<string, object>();
			fieldsToReplace.Add( "killerId", null );
			fieldsToReplace.Add( "targetId", null );

			Dictionary<string, object> fieldMap = GetFieldMap();
			IList<string> content = new List<string>();

			foreach ( KeyValuePair<string, object> kvp in fieldMap )
			{
				string key = kvp.Key;
				string valStr = kvp.Value.ToString();
				
				if ( fieldsToReplace.ContainsKey( key ) )
				{
					uint val = (uint)kvp.Value;
					if ( entityAliases.ContainsKey( val ) )
						valStr = entityAliases[ val ];
				}

				content.Add( key + " : " + valStr );
			}
      
			return content;
		}
	};

	public class CPrmTimelineValue : CTimelineValueBase
	{
		public string m_prm;

		public override void MakeFrom(TRawTlineVal from)
		{
			base.MakeFrom(from);
			m_prm = from.prm;			
		}
	};

	public class CNamedTimelineValue : CTimelineValueBase
	{
		public string m_name;
		public override void MakeFrom(TRawTlineVal from)
		{
			base.MakeFrom(from);
			m_name = from.name;
		}
	};

	public class CPosTimelineValue : CTimelineValueBase
	{
		public Vec3 m_pos;
		public float m_ground;

		public override void MakeFrom(TRawTlineVal from)
		{
			base.MakeFrom(from);

			try
			{
				m_ground = Convert.ToSingle(from.param[0].g, CFloatingPointFormat.Get());

				m_pos = new Vec3();
				m_pos.x = Convert.ToSingle(from.param[0].x, CFloatingPointFormat.Get());
				m_pos.y = Convert.ToSingle(from.param[0].y, CFloatingPointFormat.Get());
				m_pos.z = Convert.ToSingle(from.param[0].z, CFloatingPointFormat.Get());
			}
			catch (System.FormatException)
			{
				System.Diagnostics.Debug.Assert("Invalid typecast : CPosTimelineValue".Length == 0);
				return;
			}
		}
	};

	public class CKillTimelineValue : CTimelineValueBase
	{
		public uint m_targetId;
		public uint m_projectileId;
		public string m_hitType;

		public override void MakeFrom(TRawTlineVal from)
		{
			base.MakeFrom(from);

			try
			{
				m_targetId = Convert.ToUInt32(from.param[0].target_id);
				m_projectileId = Convert.ToUInt32(from.param[0].projectile_id);
				m_hitType = from.param[0].hit_type;
			}
			catch (System.FormatException)
			{
				System.Diagnostics.Debug.Assert("Invalid typecast: CKillTimelineValue".Length == 0);
				return;
			}
		}
	};

	public class CShotTimelineValue : CTimelineValueBase
	{
		public uint m_projectileId;
		public uint m_pellets;
		public uint m_ammoLeft;
		public string m_ammoType;

		public override void MakeFrom(TRawTlineVal from)
		{
			base.MakeFrom(from);

			try
			{
				m_projectileId = Convert.ToUInt32(from.param[0].projectile_id);
				m_pellets = Convert.ToUInt32(from.param[0].pellets);
				m_ammoLeft = Convert.ToUInt32(from.param[0].ammo_left);
				m_ammoType = from.param[0].ammo_type;
			}
			catch (System.FormatException)
			{
				System.Diagnostics.Debug.Assert("Invalid typecast: CShotTimelineValue".Length == 0);
				return;
			}
		}
	};

	public class CThrowTimelineValue : CShotTimelineValue
	{
		public float m_lifeTime;
		public float m_timeLeft;
		public string m_weaponClass;

		public override void MakeFrom(TRawTlineVal from)
		{
			base.MakeFrom(from);

			try
			{
				m_lifeTime		= Convert.ToSingle(from.param[0].life_time,	CFloatingPointFormat.Get());
				m_timeLeft		= Convert.ToSingle(from.param[0].time_left,	CFloatingPointFormat.Get());
				m_weaponClass	= from.param[0].weapon_class;
			}
			catch (System.FormatException)
			{
				System.Diagnostics.Debug.Assert("Invalid typecast: CThrowTimelineValue".Length == 0);
				return;
			}
		}
	};

	public class CHitTimelineValue : CTimelineValueBase
	{
		public uint m_targetId;
		public uint m_projectileId;
		public uint m_ricochetCount;
		public float m_damage;
		public float m_hitToLookAngle;
		public string m_materialType;
		public string m_weaponClass;
		public string m_hitType;

		public override void MakeFrom(TRawTlineVal from)
		{
			base.MakeFrom(from);

			try
			{
				m_targetId = Convert.ToUInt32(from.param[0].target_id);
				m_projectileId = Convert.ToUInt32(from.param[0].projectile_id);
				m_ricochetCount = Convert.ToUInt32(from.param[0].ricochet_count);

				m_damage = Convert.ToSingle(from.param[0].damage, CFloatingPointFormat.Get());
				m_hitToLookAngle = Convert.ToSingle(from.param[0].hit_to_look_angle, CFloatingPointFormat.Get());

				m_materialType = from.param[0].material_type;
				m_weaponClass = from.param[0].weapon_class;
				m_hitType = from.param[0].hit_type;
			}
			catch (System.FormatException)
			{
				System.Diagnostics.Debug.Assert("Invalid typecast: CHitTimelineValue".Length == 0);
				return;
			}
		}
	};

	public class CMeleeTimelineValue : CTimelineValueBase
	{
		public uint m_meleeId;

		public override void MakeFrom(TRawTlineVal from)
		{
			base.MakeFrom(from);

			try
			{
				m_meleeId = Convert.ToUInt32(from.prm);
			}
			catch (System.FormatException)
			{
				System.Diagnostics.Debug.Assert("Invalid typecast: CMeleeTimelineValue".Length == 0);
				return;
			}
		}
	};

	public class CMeleeHitTimelineValue : CTimelineValueBase
	{
		public uint m_meleeId;
		public uint m_targetId;
		public string m_materialType;
		public string m_weaponClass;
		public float m_damage;

		public override void MakeFrom(TRawTlineVal from)
		{
			base.MakeFrom(from);

			try
			{
				m_meleeId = Convert.ToUInt32(from.param[0].melee_id);
				m_targetId = Convert.ToUInt32(from.param[0].target_id);
				m_materialType = from.param[0].material_type;
				m_weaponClass = from.param[0].weapon_class;
				m_damage = Convert.ToSingle(from.param[0].damage, CFloatingPointFormat.Get());
			}
			catch (System.FormatException)
			{
				System.Diagnostics.Debug.Assert("Invalid typecast: CMeleeTimelineValue".Length == 0);
				return;
			}
		}
	};

	public class CDeathTimelineValue : CTimelineValueBase
	{
		public uint m_projectileId;
		public uint m_killerId;
		public string m_hitType;

		public override void MakeFrom(TRawTlineVal from)
		{
			base.MakeFrom(from);

			try
			{
				m_projectileId = Convert.ToUInt32(from.param[0].projectile_id);
				m_killerId = Convert.ToUInt32(from.param[0].killer_id);
				m_hitType = from.param[0].hit_type;
			}
			catch (System.FormatException)
			{
				System.Diagnostics.Debug.Assert("Invalid typecast: CDeathTimelineValue".Length == 0);
				return;
			}
		}
	};

	public class CAttachmentTimelineValue : CTimelineValueBase
	{
		public string m_target;
		public string m_socket;
		public string m_name;

		public override void MakeFrom(TRawTlineVal from)
		{
			base.MakeFrom(from);

			m_target = from.param[0].target;
			m_socket = from.param[0].socket;
			m_name = from.param[0].name;
			
		}
	};

	public class CExplodeTimelineValue : CTimelineValueBase
	{
		public string m_name;
		public uint m_projectileId;
		public bool m_inHands;

		public override void MakeFrom(TRawTlineVal from)
		{
			base.MakeFrom(from);

			m_name = from.param[0].name;

			try
			{
				m_projectileId = Convert.ToUInt32(from.param[0].projectile_id);
				m_inHands = Convert.ToUInt32(from.param[0].in_hands) != 0;
			}
			catch (System.FormatException)
			{
				System.Diagnostics.Debug.Assert("Invalid typecast: CExplodeTimelineValue".Length == 0);
				return;
			}
		}
	};
}