﻿using System;
using System.Collections.Generic;
using System.Text;
using StatsDataSource.Core;
using StatsDataSource.Storage;
using StatsDataSource.Test;
using System.Threading;

namespace StatsStreaming.EngineComm
{
	public class EngineStreamPlugin : IStatsPlugin
	{
		//////////////////////////////////////////////////////////////////////////

		#region IStatsPlugin

		public string getName() { return "Engine communication plug-in"; }

		public int getUsedTagCount() { return 0; }

		public StatsCore Core;

		public IMainForm MainForm;

		public IPluginAction[] getActions() { return new IPluginAction[] { m_connectAction }; }

		private MessageParser m_parser;
		private MessageQueue m_msgQueue;
		private PipeClient m_pipeClient;

		public void Start(StatsCore Core, IMainForm MainForm, int tagBias)
		{
			this.Core = Core;
			this.MainForm = MainForm;

			m_connectAction = new DelegAction("&Actions/&Engine data exchange", Connect);

			m_msgQueue = new MessageQueue(Core.Repository);
			m_msgQueue.QueueUpdated += MessageQueueUpdated;

			m_parser = new MessageParser(m_msgQueue);

			m_pipeClient = new PipeClient();
			m_pipeClient.ConnectionEstablished += PipeConnectionEstablished;
			m_pipeClient.ConnectionTerminated += PipeConnectionTerminated;
			m_pipeClient.MessageReceived += PipeMessageReceived;

			if (Array.Find<string>(Core.CmdArgs, x => x == "--engine_connect") != null)
				m_connectAction.Execute();
		}

		public void HandleCoreMessage(ECoreMessage msg)
		{
			if (msg == ECoreMessage.Idle && !m_msgQueue.Empty)
			{
				m_msgQueue.DeliverAll();
				MainForm.SetStatus("Done");
			}
		}

		public void Stop()
		{
			if (m_pipeClient.IsConnected())
			{
				m_pipeClient.ConnectionEstablished -= PipeConnectionEstablished;
				m_pipeClient.ConnectionTerminated -= PipeConnectionTerminated;
				m_pipeClient.MessageReceived -= PipeMessageReceived;
				m_pipeClient.CloseConnection();
			}
		}

		DelegAction m_connectAction;

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Actions

		private void Connect(IPluginAction action)
		{
			try
			{
				if (!m_pipeClient.IsConnected())
				{
					m_pipeClient.OpenConnection("\\\\.\\pipe\\CryStatsPipe_" + Core.Build, 100 * 1024);
					MainForm.SetStatus("Waiting for connection...");
				}
				else
				{
					m_pipeClient.CloseConnection();
					MainForm.SetStatus("Connection closed");
				}
			}
			catch (Exception ex)
			{
				MainForm.SetStatus(ex.Message);
			}

			m_connectAction.setIsEnabled(m_pipeClient.IsConnected());
		}

		void Disconnect()
		{
		}

		void MessageQueueUpdated()
		{
			MainForm.SetStatus("Receiving data...");
		}

		#region Pipe messages
		void PipeConnectionEstablished()
		{
			MainForm.SetStatus("Engine stream connected");
		}

		void PipeConnectionTerminated()
		{
			MainForm.SetStatus("Engine connection terminated");
			m_connectAction.setIsEnabled(m_pipeClient.IsConnected());
		}

		public void PipeMessageReceived(string msg)
		{
			m_parser.ProcessMessage(msg);
		}
		#endregion

		#endregion

		//////////////////////////////////////////////////////////////////////////
	}
}
