﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using System.Xml;
using System.IO;
using StatsStreaming;
using StatsStreaming.Messages;
using StatsDataSource.ObjectModel;

namespace StatsStreaming
{
	public class MessageParser
	{
		private enum OpCode
		{
			eOC_Visit = 0,
			eOC_Leave = 1,

			eOC_Unknown = 255,
		}

		private MessageQueue m_queue;

		public MessageParser(MessageQueue queue)
		{
			m_queue = queue;
		}

		#region Parsing
		private OpCode ParseOpCode(string msg)
		{
			OpCode op = OpCode.eOC_Unknown;
			if (msg[0] == '{')
			{
				int endOp = msg.IndexOf('}') + 1;
				string soc = msg.Substring(1, endOp - 2);
				int ic;
				if (int.TryParse(soc, out ic))
				{
					op = (OpCode)ic;
				}
			}
			return op;
		}

		private bool TryGetLocator(string msg, out Locator loc)
		{
			int start = msg.IndexOf('[');
			int end = msg.IndexOf(']');

			if (start > 0 && end > 0)
			{
				string ls = msg.Substring(start, end - start + 1);
				loc = Locator.FromString(ls);
				return true;
			}

			loc = new Locator();
			return false;
		}

		private bool TryParsePair(string msg, out int id, out string name)
		{
			id = -1;
			name = null;

			string[] parts = msg.Split(new char[] { '[', ',', ']' });

			if (!int.TryParse(parts[1], out id))
				return false;

			name = parts[2];

			return true;
		}
		#endregion

		#region Message Processing
		public void ProcessMessage(string msg)
		{
			OpCode oc = ParseOpCode(msg);
			Locator l;

			switch (oc)
			{
				case OpCode.eOC_Visit:
					TryGetLocator(msg, out l);
					OnVisitNode(l);
					break;
				case OpCode.eOC_Leave:
					TryGetLocator(msg, out l);
					OnLeaveNode(l);
					break;
				default:
					OnDataMessage(msg);
					break;
			}
		}

		private void OnVisitNode(Locator locator)
		{
			m_queue.Enqueue(new NavigationMessage(NavigationMessage.EOperation.eOP_Push, locator));
		}

		private void OnLeaveNode(Locator locator)
		{
			m_queue.Enqueue(new NavigationMessage(NavigationMessage.EOperation.eOP_Pop, locator));
		}

		private void OnDataMessage(string msg)
		{
			XmlDocument doc = new XmlDocument();
			doc.LoadXml(msg);

			Debug.Assert(doc.ChildNodes.Count == 1);
			XmlNode root = doc.ChildNodes[0];

			ExtractStates(root);

			ExtractEvents(root);
		}
		#endregion

		#region Data parsing
		private void ExtractStates(XmlNode root)
		{
			StateMessage stateMsg = new StateMessage();
			ExtractStatesFromAttributes(root, stateMsg);
			ExtractStatesFromElements(root, stateMsg, null);

			if (stateMsg.States.Count > 0)
				m_queue.Enqueue(stateMsg);
		}

		private void ExtractStatesFromAttributes(XmlNode root, StateMessage stateMsg)
		{
			for (int i = 0; i != root.Attributes.Count; ++i)
			{
				XmlAttribute attr = root.Attributes[i];
				stateMsg.States.Add(new StateEntry(attr.Name, attr.Value));
			}
		}

		private void ExtractStatesFromElements(XmlNode node, StateMessage stateMsg, StateEntry state)
		{
			for (int i = 0; i != node.ChildNodes.Count; ++i)
			{
				XmlNode chld = node.ChildNodes[i];

				if (state == null && chld.Name == "timelines")
					continue;

				StateEntry newState = new StateEntry(chld.Name, null);

				if (state == null)
					stateMsg.States.Add(newState);
				else
					state.children.Add(newState);

				for (int j = 0; j != chld.Attributes.Count; ++j)
				{
					XmlAttribute attr = chld.Attributes[j];
					newState.children.Add(new StateEntry(attr.Name, attr.Value));
				}

				ExtractStatesFromElements(chld, stateMsg, newState);
			}
		}

		private void ExtractEvents(XmlNode root)
		{
			EventMessage msg = new EventMessage();

			for (int i = 0; i != root.ChildNodes.Count; ++i)
			{
				XmlNode chld = root.ChildNodes[i];
				if (chld.Name == "timelines")
				{
					for (int j = 0; j != chld.ChildNodes.Count; ++j)
					{
						XmlNode tl = chld.ChildNodes[j];
						Debug.Assert(tl.Name == "timeline");

						var group = new GroupEnry(tl.Attributes["name"].Value);
						msg.EventGroups.Add(group);

						ParseTimeline(group, tl);
					}
				}
			}

			if (msg.EventGroups.Count != 0)
				m_queue.Enqueue(msg);
		}

		private void ParseTimeline(GroupEnry group, XmlNode timeline)
		{
			for (int i = 0; i != timeline.ChildNodes.Count; ++i)
			{
				XmlNode val = timeline.ChildNodes[i];

				Int64 time = Int64.Parse(val.Attributes["time"].Value);

				List<KeyValuePair<string, string>> prms = new List<KeyValuePair<string, string>>();

				for (int prm = 0; prm != val.Attributes.Count; ++prm)
				{
					XmlAttribute attr = val.Attributes[prm];
					if (attr.Name != "time")
						prms.Add(new KeyValuePair<string, string>(attr.Name, attr.Value));
				}

				Debug.Assert(val.ChildNodes.Count < 2);

				if (val.ChildNodes.Count != 0)
				{
					for (int prm = 0; prm != val.ChildNodes[0].Attributes.Count; ++prm)
					{
						XmlAttribute attr = val.ChildNodes[0].Attributes[prm];
						prms.Add(new KeyValuePair<string, string>(attr.Name, attr.Value));
					}
				}

				group.events.Add(new EventEntry(time, prms.ToArray()));
			}
		}
		#endregion
	}
}
