﻿using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using Microsoft.Win32.SafeHandles;
using System.Threading;
using StatsStreaming;

namespace StatsDataSource.Test
{
	public class PipeClient : IDisposable
	{
		public delegate void ConnecionDeleg();
		public event ConnecionDeleg ConnectionEstablished;
		public event ConnecionDeleg ConnectionTerminated;

		public delegate void MessageReceivedDeleg(string msg);
		public event MessageReceivedDeleg MessageReceived;

		public void OpenConnection(string pipeName, int bufferSize)
		{
			if (IsConnected())
				throw new Exception("Already connected");

			m_pipename = pipeName;
			m_buffer = new byte[bufferSize];

			m_thread = new Thread(ThreadRoutine);
			m_thread.Start(this);
		}

		public void CloseConnection()
		{
			if (IsConnected())
				m_thread.Abort();
		}

		public bool IsConnected()
		{
			return m_thread != null && m_thread.IsAlive;
		}

		public void Dispose()
		{
			if (IsConnected())
				CloseConnection();
		}

		//////////////////////////////////////////////////////////////////////////////////

		private bool TryOpenPipe()
		{
			if (PipesWinAPI.WaitNamedPipe(m_pipename, 0xffffffff) == 0)
				return false;

			m_pipe = PipesWinAPI.CreateFile(
				 m_pipename,
				 (uint)PipesWinAPI.AccessMode.GENERIC_READ,
				 0,
				 IntPtr.Zero,
				 (uint)PipesWinAPI.CreationDisposition.OPEN_EXISTING,
				 0,
				 IntPtr.Zero);

			if (m_pipe.IsInvalid)
				return false;

			m_stream = new FileStream(m_pipe, FileAccess.Read, m_buffer.Length, false);

			return true;
		}


		public string doReceiveMessage()
		{
			int read = m_stream.Read(m_headerBuf, 0, m_headerBuf.Length);
			if (read != m_headerBuf.Length)
				return null;

			int msgLength = BitConverter.ToInt32(m_headerBuf, 0);
			StringBuilder msg = new StringBuilder(msgLength + 10);


			int bytesLeft = msgLength;

			while (true)
			{
				read = m_stream.Read(m_buffer, 0, Math.Min(m_buffer.Length, bytesLeft));
				if (read == 0)
					return null;

				bytesLeft -= read;

				msg.Append(Encoding.ASCII.GetString(m_buffer, 0, read));

				if (bytesLeft == 0)
					break;
			}

			return msg.ToString().Replace("\n", "").Replace("\0", "");
		}

		//////////////////////////////////////////////////////////////////////////////////

		private static void ThreadRoutine(object prm)
		{
			PipeClient cl = prm as PipeClient;

			while (!cl.TryOpenPipe())
				Thread.Sleep(500);

			if (cl.ConnectionEstablished != null)
				cl.ConnectionEstablished();

			try
			{
				for (; ; )
				{
					string msg = cl.doReceiveMessage();

					if (msg == null)
						break;

					if (cl.MessageReceived != null)
						cl.MessageReceived(msg);
				}
			}
			finally
			{
				cl.m_stream.Close();
				cl.m_stream = null;
				cl.m_pipe.Close();

				if (cl.ConnectionTerminated != null)
					cl.ConnectionTerminated();
			}
		}

		//////////////////////////////////////////////////////////////////////////////////

		private string m_pipename;
		private SafeFileHandle m_pipe;
		private FileStream m_stream;
		private byte[] m_headerBuf = new byte[4];
		private byte[] m_buffer;
		private Thread m_thread;
	}
}
