using System;
using StatsTool.Properties;
using System.Collections.Generic;
using System.Drawing;
using StatsParse;

namespace StatsTool
{
	static class CDrawDensity
	{
		static public void Init(Graphics g, int cellSize, int cellOffset)
		{
			m_data = new CDensityData();
			m_data.m_cellSize = cellSize;
			m_data.m_cellOffset = cellOffset;

			int colCnt = (int)Math.Ceiling(g.VisibleClipBounds.Width / m_data.m_cellSize);
			int rowCnt = (int)Math.Ceiling(g.VisibleClipBounds.Height / m_data.m_cellSize);

			m_data.m_pointsInSq = new long[rowCnt, colCnt];
			m_data.m_maxPointsInSq = 0;
		}

		static public void AddPositions(IList<CPlayerPos> positions)
		{
			IList<Vec2> newPos = new List<Vec2>();
			foreach (CPlayerPos pos in positions)
				newPos.Add( new Vec2(pos.X, pos.Y) );

			AddPositions(newPos);
		}

		static public void AddPositions(IList<Vec2> positions)
		{
			if (m_data.m_pointsInSq == null)
				return;

			foreach (Vec2 pos in positions)
			{
				int r = (int)pos.y / m_data.m_cellSize;
				int c = (int)pos.x / m_data.m_cellSize;

				if (r < 0 || r >= m_data.m_pointsInSq.GetLength(0))
					continue;
				if (c < 0 || c >= m_data.m_pointsInSq.GetLength(1))
					continue;

				++m_data.m_pointsInSq[r, c];
				m_data.m_maxPointsInSq = Math.Max(m_data.m_maxPointsInSq, m_data.m_pointsInSq[r, c]);
			}
		}

		static public void DoDraw(Graphics g, bool doFilterRange, long rangeFrom, long rangeTo)
		{
			if ( m_data.m_maxPointsInSq <= 0 )
				return;

			if ( !doFilterRange )
			{
				rangeFrom = 1;
				rangeTo = m_data.m_maxPointsInSq;
			}

			for (int r = 0; r < m_data.m_pointsInSq.GetLength(0); ++r)
				for (int c = 0; c < m_data.m_pointsInSq.GetLength(1); ++c)
				{
					long pt = m_data.m_pointsInSq[r, c];

					if (pt < rangeFrom)
						continue;

					double sc = (pt - rangeFrom) / (double)(rangeTo - rangeFrom);
					int clrMax = Color.White.R;
					int isc = (int)(clrMax * sc);

					int red = 0;
					int green = clrMax;

					int middle = clrMax / 2;
					if (isc < middle)
					{
						red = isc * 2;
						green = clrMax;
					}
					else
					{
						red = clrMax;
						green = 1 + clrMax - (isc - middle) * 2;
					}

					red = Math.Max(red, 0);
					red = Math.Min(red, clrMax);

					green = Math.Max(green, 0);
					green = Math.Min(green, clrMax);

					Color cl = Color.FromArgb(red, green, 0);
					Brush brush = new SolidBrush(cl);

					int sqFull = m_data.m_cellOffset;
					int sqOff = sqFull / 2;
					g.FillRectangle(brush,	sqOff + c * m_data.m_cellSize,
																	sqOff + r * m_data.m_cellSize,
																	m_data.m_cellSize - sqFull,
																	m_data.m_cellSize - sqFull );
				}
		}

		static public bool IsDensityMode
		{
			get
			{
				return Settings.Default.PathsDrawMode == 1 || Settings.Default.EventsDrawMode == 1;
			}
		}

		static public CDensityData DensityData
		{
			set
			{
				m_data = (CDensityData)value.Clone();
			}
		}

		static CDensityData m_data;
	}
}