using System;
using System.Collections.Generic;
using System.Text;
using System.Drawing.Drawing2D;
using System.Drawing;
using StatsTool.Properties;
using StatsParse;
using System.Windows.Forms;

namespace StatsTool
{
		public struct PathFilterFlags
		{
			private bool m_outdoor;
			private bool m_indoor;
			private bool m_specialMovesOnly;

			public PathFilterFlags(bool outdoor, bool indoor, bool specialMovesOnly)
			{
				m_outdoor = outdoor;
				m_indoor = indoor;
				m_specialMovesOnly = specialMovesOnly;
			}

			public bool Outdoor
			{
				get
				{
					return m_outdoor;
				}
				set
				{	
					m_outdoor = value;
				}
			}
		
			public bool Indoor
			{
				get
				{
					return m_indoor;
				}
				set
				{
					m_indoor = value;
				}
			}

			public bool SpecialMovesOnly 
			{
				get
				{
					return m_specialMovesOnly;
				}
				set
				{
					m_specialMovesOnly = value;
				}
			}
		};
		
		public struct SAttributeFilterData
		{
			public bool m_ignore;
			public EComparsionType m_comparsionType;
			public string m_fieldName;
			public object m_etalon;
		};

		public struct STimeFilterData
		{
			public float m_begin, m_end;
			public bool Meet( IEvent ev )
			{
				float tm = ev.Time;
				return tm >= m_begin && tm <= m_end;
			}
			public bool Meet(CPlayerPos pos)
			{
				float tm = pos.GetTimeInSeconds();
				return tm >= m_begin && tm <= m_end;
			}
		};

		public enum EComparsionType
		{
			Equal_to,
			Less_than,
			Greater_than,
			Not_equal_to,
			Less_equal_to,
			Greater_equal_to
		}

		delegate bool ComparsionDelegate(object l, object r);

		public static class CComparsionManager
		{
			static Dictionary<EComparsionType, ComparsionDelegate> delegates = new Dictionary<EComparsionType, ComparsionDelegate>();

			static bool IsEqual(object l, object r)
			{
				return l.Equals(r);
			}

			static bool TryCompareTo(object l, object r, out int res)
			{
				if (l is IComparable && r is IComparable)
				{
					IComparable cl = l as IComparable;
					IComparable cr = r as IComparable;
					res = cl.CompareTo(cr);
					return true; 
				}
				res = 0;
				return false;
			}

			static bool IsLess(object l, object r)
			{
				int res;
				if (TryCompareTo(l, r, out res))
					return res < 0;
				return false;
			}

			static bool IsGreater(object l, object r)
			{
				int res;
				if (TryCompareTo(l, r, out res))
					return res > 0;
				return false;
			}

			static bool IsNotEqual(object l, object r)
			{
				return !IsEqual(l, r);
			}

			static bool IsLessEqual(object l, object r)
			{
				return !IsGreater(l, r);
			}

			static bool IsGreaterEqual(object l, object r)
			{
				return !IsLess(l , r);
			}

			static void Init()
			{
				if (delegates.Count > 0)
					return;

				delegates.Add( EComparsionType.Equal_to,					IsEqual );
				delegates.Add( EComparsionType.Less_than,					IsLess );
				delegates.Add( EComparsionType.Greater_than,			IsGreater );
				delegates.Add( EComparsionType.Not_equal_to, 			IsNotEqual );
				delegates.Add( EComparsionType.Less_equal_to, 		IsLessEqual );
				delegates.Add( EComparsionType.Greater_equal_to,	IsGreaterEqual );
			}

			static public bool Do(object l, object r, EComparsionType cmp)
			{
				Init();
				return delegates[cmp](l, r);
			}
		}

		public static class CCompareAttributes
		{
			static public bool Do(SAttributeFilterData data, ITimelineValue val)
			{
				if (data.m_ignore)
					return true;

				Dictionary<string, object> fldMap = val.GetFieldMap();
				string fN = data.m_fieldName;
				object et = data.m_etalon;
				if (fN != null && et != null)
					if (fldMap.ContainsKey(fN))
						return CComparsionManager.Do(fldMap[fN], et, data.m_comparsionType);

				return false;
			}
		};


		public class CDrawPlayer
    {
				public static readonly string posTimelineName = "position";

        private uint m_id;
				private IActorLife m_srcLife;

        private List<CPlayerPos> m_positions = new List<CPlayerPos>();
				private Dictionary<string, IEvent[]> m_events = new Dictionary<string, IEvent[]>();
        private Color m_customColor;
        private bool m_draw = true;
        private CDrawTeam m_parent;
				private bool m_humanControlled;

				public Vec3 GetPosition( int time )
				{
					int cnt = m_positions.Count;
					if (cnt != 0)
					{
						foreach (CPlayerPos pos in m_positions)
						{
							if ( pos.Time >= time )
								return new Vec3( pos.X, pos.Y, pos.Z );
						}
					}

					if (m_positions.Count >= 1)
					{
						const int allowedTime = 5000;

						CPlayerPos pos = m_positions[0];
						int deltaTime = time - pos.Time;

						if ( deltaTime < 0 )
						{
							if ( deltaTime < -allowedTime )
								pos = null;
						}
						else
						{
							pos = m_positions[m_positions.Count-1];
							deltaTime = time - pos.Time;

							if ( deltaTime > allowedTime )
								pos = null;
						}
						
						if (pos != null)
							return new Vec3(pos.X, pos.Y, pos.Z);
					}

					return new Vec3();
				}

				public CDrawPlayer(IActorLife life, Color color, SharedData data, CDrawTeam parent, bool humanControlled )
				{
					m_customColor = color;
					m_id = life.EntityId;
					m_srcLife = life;
					m_parent = parent;
					m_humanControlled = humanControlled;

					if (life.EventTimelines != null )
					{
						if ( !life.EventTimelines.ContainsKey(posTimelineName) )
						{
							if ( !data.m_ignoreNoPosition )
							{
								DialogResult res = MessageBox.Show("No position timeline found for " + life.Name, "Bad data. Press cancel to skip", MessageBoxButtons.OKCancel);
								if ( res == DialogResult.Cancel )
									data.m_ignoreNoPosition = true;
							}
						}
						else
						{
							//need to fill positions first
							{
								ITimelineValue[] posVals = life.EventTimelines[posTimelineName];
								if (posVals != null)
								{
									int posCnt = posVals.Length;
									for (int i = 0; i < posCnt; ++i)
									{
										CPosTimelineValue val = posVals[i] as CPosTimelineValue;
										if (val != null)
										{
											Vec3 vec = val.m_pos;

											if (data.m_coordTransformData.ContainsKey(data.m_curMapName))
												data.m_coordTransformData[data.m_curMapName].Transform(vec);

											CPlayerPos pos = new CPlayerPos(vec.y, vec.x, vec.z, val.m_ground, val.m_time, Parent.Parent.Begin);
											m_positions.Add(pos);
										}
									}
								}
							}

							//now all other events, using GetPosition()
							foreach (KeyValuePair<string, ITimelineValue[]> kvp in life.EventTimelines)
							{
								if (kvp.Key != posTimelineName)
								{
									int evCnt = kvp.Value.Length;
									IEvent[] evs = new IEvent[evCnt];

									for (int i = 0; i < evCnt; ++i)
									{
										ITimelineValue val = kvp.Value[i];
										Vec3 pos = GetPosition(val.GetTime());
										CCrossDrawEvent ev = new CCrossDrawEvent(kvp.Key, pos.x, pos.y, val, this);
										evs[i] = ev;
									}
									m_events[kvp.Key] = evs;
								}
							}
						}
					}
				}

				public void DoDraw(Graphics g, SharedData data)
        {
					DoDraw(g, m_customColor, data);
        }

        public List<CPlayerPos> Positions
        {
					get { return m_positions; }
        }

				public Dictionary<string, IEvent[]> Events
        {
					get { return m_events; }
        }

				public void DoDraw(Graphics g, Color color, SharedData data)
				{
					switch (Settings.Default.PathsDrawMode)
					{
						case 0:
							{
								DrawPathsInner(g, color, data, m_srcLife.EventTimelines);
							} break;

						case 1:
							{
								AddPathsDensityData(data, m_srcLife.EventTimelines);
							} break;
					}

					switch (Settings.Default.EventsDrawMode)
					{
						case 0:
							{
								DrawEventsInner(g, data);
							} break;

						case 1:
							{
								AddEventsDensityData(data);
							} break;
					}
				}

				private void DrawPathsInner(Graphics g, Color color, SharedData data, Dictionary<string, ITimelineValue[]> eventTlines)
        {
            if(m_draw)
            {
								IList<GraphicsPath> paths = FormPaths(data, eventTlines);
                
                Pen p = new Pen(color);
                p.Width = 3;
                p.DashStyle = DashStyle.Solid;
								foreach (GraphicsPath gp in paths)
                {
                    g.DrawPath(p, gp);
                }
            }
        }

				private void AddPathsDensityData(SharedData data, Dictionary<string, ITimelineValue[]> eventTlines)
				{
					if (m_draw)
					{
						List<CPlayerPos> posFiltered = new List<CPlayerPos>();
						foreach ( CPlayerPos pos in m_positions )
							if (data.m_timeFilterData.Meet(pos))
								if (MeetFlags(pos, data.m_pathFilterFlags, eventTlines))
									posFiltered.Add(pos);

						CDrawDensity.AddPositions(posFiltered);
					}
				}

				private void AddEventsDensityData(SharedData data)
				{
					if (m_draw)
					{
						List<Vec2> evPos = AccamulateEventPositions(data);
						CDrawDensity.AddPositions(evPos);
					}
				}

				private List<Vec2> AccamulateEventPositions(SharedData data)
				{
					CategoryFilter filter = data.m_categoryFilters[EFilterType.EVENT];
					List<Vec2> evPos = new List<Vec2>();

					foreach (KeyValuePair<string, IEvent[]> kvp in m_events)
					{
						if (filter.GetFlag(kvp.Key))
						{
							IEvent[] evs = kvp.Value;
							foreach (IEvent ev in evs)
								if (data.m_timeFilterData.Meet(ev))
									if (CCompareAttributes.Do(data.m_attributeFilterData, ev.TimelineValue))
									{
										evPos.Add( ev.GetFinalPosition( data.m_useTargetPos ) );
										ev.Unhide();
									}
						}
					}
					
					return evPos;
				}

				private void DrawEventsInner(Graphics g, SharedData data)
        {
					foreach (KeyValuePair<string, IEvent[]> kvp in m_events)
					{
						IEvent[] evs = kvp.Value;
						if (data.m_categoryFilters[EFilterType.EVENT].GetFlag(kvp.Key))
							foreach (IEvent ev in evs)
								if ( data.m_timeFilterData.Meet(ev) )
									if ( CCompareAttributes.Do( data.m_attributeFilterData, ev.TimelineValue ) )
										ev.Draw(g, data.m_useTargetPos);
					}
        }

				public void RequestEventTypes(UniqueList<string> res, SharedData data)
				{
					foreach (KeyValuePair<string, IEvent[]> kvp in m_events)
						if (kvp.Value.GetLength(0) > 0)
							if (kvp.Value[0].InSelectedGroup(data.m_eventGroups))
								res.Add(kvp.Key);
				}

				private bool MeetFlags(CPlayerPos pos, PathFilterFlags flags, Dictionary<string, ITimelineValue[]> eventTlines)
				{
					if (!flags.Indoor && pos.Indoor == true)
						return false;
					if (!flags.Outdoor && pos.Outdoor == true)
						return false;
					if (flags.SpecialMovesOnly && !pos.InSpecialMove(eventTlines))
						return false;

					return true;
				}

				private List<GraphicsPath> FormPaths(SharedData data, Dictionary<string, ITimelineValue[]> eventTlines)
        {
					List<List<PointF>> paths = new List<List<PointF>>();

					List<PointF> subpath = new List<PointF>();
					bool makeNewSubpath = true;

					for (int i = 0; i < m_positions.Count - 1; ++i)
					{
						CPlayerPos curPos = m_positions[i];
						CPlayerPos nxtPos = m_positions[i + 1];

						bool curMeet = MeetFlags(curPos, data.m_pathFilterFlags, eventTlines) && data.m_timeFilterData.Meet(curPos);
						bool nxtMeet = MeetFlags(nxtPos, data.m_pathFilterFlags, eventTlines) && data.m_timeFilterData.Meet(nxtPos);

						if (curMeet && nxtMeet)
						{
							if (makeNewSubpath)
							{
								if (subpath.Count > 0)
									paths.Add(subpath);

								subpath = new List<PointF>();
								makeNewSubpath = false;

								subpath.Add(new PointF(curPos.X, curPos.Y));
							}

							subpath.Add( new PointF(nxtPos.X, nxtPos.Y) );
						}
						else
						{
							makeNewSubpath = true;
						}
					}//end of "for (int i..."

					if (subpath.Count > 0)
						paths.Add(subpath);

					return CreateGraphicPaths(paths);
        }

				private List<GraphicsPath> CreateGraphicPaths(List<List<PointF>> paths)
        {
					List<GraphicsPath> resPaths = new List<GraphicsPath>();

					foreach (List<PointF> subpath in paths)
					{
						GraphicsPath gp = new GraphicsPath();
						gp.StartFigure();

						int cnt = subpath.Count;

						if (cnt > 0)
						{
							gp.AddLines(subpath.ToArray());
							resPaths.Add(gp);
						}						
					}

					return resPaths;
        }

				public bool GetLifetime( ref int begin, ref int end)
				{
					return m_srcLife.GetLifetime(ref begin, ref end);
				}

        public uint Id
        {
            get
            {
                return m_id;
            }
        }

        public string Name
        {
            get
            {
							return m_srcLife.Name;
            }
        }

			public bool CanDraw
			{
				get
				{
					return m_draw;
				}
				set
				{
					m_draw = value;
				}
			}

        public Color Color
        {
            get
            {
                return m_customColor;
            }
            set
            {
                m_customColor = value;
            }
        }

        public CDrawTeam Parent
        {
            get
            {
                return m_parent;
            }
        }

				public bool HumanControlled
				{
					get
					{
						return m_humanControlled;
					}
				}

				public bool AiControlled
				{
					get
					{
						return !m_humanControlled;
					}
				}
    }
}
