using System;
using System.Collections.Generic;
using System.Text;
using System.Drawing;
using StatsTool.Properties;
using StatsParse;

namespace StatsTool
{
	public interface IEvent
  {
		Vec2 Pos { get; }
    float X { get; }
		float Y { get; }
    void Draw(Graphics g, bool useTargetPos);
		Vec2 GetFinalPosition(bool useTargetPos);
		void Hide();
		void Unhide();
		bool Hiden();
    string Type { get; }
		string Desc { get; }
		float Time { get; }
		IList<string> Fields{ get; }
		ITimelineValue TimelineValue { get; }
		CDrawPlayer Parent { get; }
		CDrawRound ParentRound { get; }

		bool InSelectedGroup(Dictionary<EEventGroup, bool> groups);
  }

	public abstract class CEvent : IEvent
  {
		public static string TARGET_ID_FIELD_NAME = "targetId";

		private string m_type;
		protected Vec2 m_pos;
    protected Color m_color;
		protected Color m_hiColor;
		private ITimelineValue m_tlineVal;
		private bool m_hiden;
		private int m_originTime;
		private CDrawPlayer m_parent;

		protected CEvent(string type, float x, float y, ITimelineValue tlineVal, CDrawPlayer parent)
    {
      m_type = type;
      m_pos = new Vec2(x,y);
      m_color = Color.FromName(Settings.Default.EventColor);
      m_hiColor = Color.FromName(Settings.Default.HighlightedEventColor);
			m_tlineVal = tlineVal;
			m_hiden = true;
			m_parent = parent;
			m_originTime = ParentRound.Begin;
    }
    
    public virtual void Draw(Graphics g, bool useTargetPos)
    {
        OnDraw(g, useTargetPos);
				m_hiden = false;
    }

		public void Hide()
		{
			m_hiden = true;
		}

		public void Unhide()
		{
			m_hiden = false;
		}

		public bool Hiden()
		{
			return m_hiden;
		}

		public bool InSelectedGroup(Dictionary<EEventGroup, bool> groups)
		{
			if (groups[EEventGroup.ALL])
				return true;

			switch (m_type)
			{
				case "kill":
				case "shot":
				case "hit":
				case "melee":
				case "melee_hit":
				case "activate":
				case "throw":
				case "explode":
				case "death":
				case "fallnplay":
				case "reload":
				case "weapon":
				case "weapon_params":
					return groups[EEventGroup.COMBAT];

				case "climb_alone":
				case "climb_coop":
				case "climb_assist":
				case "haste":
				case "weapon_stance":
				case "stance":
				case "tactical":
				case "push_solo":
				case "push_coop":
				case "slide":
				case "vault":
					return groups[EEventGroup.MOVEMENT];

				default:
					return groups[EEventGroup.OTHER];
			}
		}

    public virtual void OnDraw(Graphics g, bool useTargetPos)
    {
        throw new InvalidOperationException("Abstract method Event::Draw was called");
    }

		public Vec3 GetTargetPosition(ITimelineValue val)
		{
			Dictionary<string, object> fldMap = val.GetFieldMap();
			if (fldMap.ContainsKey( CEvent.TARGET_ID_FIELD_NAME ))
			{
				uint targetId = (uint)fldMap[ CEvent.TARGET_ID_FIELD_NAME ];
				foreach ( CDrawTeam tm in ParentRound.Teams )
					foreach ( CDrawPlayer pl in tm.Players )
						if ( pl.Id == targetId )
						{
							int ltBegin = 0, ltEnd = 0;
							int vtime = val.GetTime();
							if ( pl.GetLifetime(ref ltBegin, ref ltEnd) )
								if ( vtime >= ltBegin && vtime <= ltEnd )
									return pl.GetPosition( vtime );
						}
			}

			return new Vec3();
		}

		public Vec2 GetFinalPosition(bool useTargetPos)
		{
			if (useTargetPos)//TODO: room for optimization through caching here
				return GetTargetPosition(TimelineValue).GetVec2();

			return m_pos;
		}

		public Vec2 Pos
		{
			get { return m_pos; }
		}

    public float X
    {
        get { return m_pos.x; }
    }

    public float Y
    {
        get { return m_pos.y; }
    }

    public string Type
    {
			get { return m_type; }
    }

		public float Time
		{
			get 
			{
				return SharedData.GetTimeFromRoundStart(m_tlineVal.GetTime(), m_originTime);
			}
		}


		public IList<string> Fields
		{
			get { return m_tlineVal.GetFieldsTextRepresentation( ParentRound.GetEntityAliases() ); }
		}

		public string Desc
		{
			get 
			{
				TimeSpan timeSpan = TimeSpan.FromSeconds(Time);
				string res = "time: " + timeSpan.ToString().Split('.')[0];
				foreach (string str in Fields)
					res+= "\n" + str;

				return res;
			}
		}

    public Color Color
    {
        get { return m_color; }
    }

		public ITimelineValue TimelineValue
		{
			get { return m_tlineVal; } 
		}

		public CDrawPlayer Parent
		{
			get{ return m_parent; }
		}

		public CDrawRound ParentRound
		{
			get{ return Parent.Parent.Parent; }
		}
  }

	class CCrossDrawEvent : CEvent
	{
		public CCrossDrawEvent(string type, float x, float y, ITimelineValue tlineVal, CDrawPlayer parent)
			: base(type, x, y, tlineVal, parent)
		{
		}

		public override void OnDraw(Graphics g, bool useTargetPos)
		{
			Pen p = new Pen(m_color);
			p.Width = 2;
			int sz = 3;

			Vec2 fp = GetFinalPosition(useTargetPos);
			g.DrawLine(p, new Point((int)fp.x - sz, (int)fp.y - sz), new Point((int)fp.x + sz, (int)fp.y + sz));
			g.DrawLine(p, new Point((int)fp.x + sz, (int)fp.y - sz), new Point((int)fp.x - sz, (int)fp.y + sz));
		}
	}
}
