using System;
using System.Collections.Generic;
using System.Text;
using System.Xml.Serialization;
using System.IO;
using StatsTool.Properties;
using System.Drawing;
using System.Drawing.Imaging;
using System.Globalization;
using DdsFileType;
using StatsParse;

namespace StatsTool
{
	public class CMiniMapMan
	{
		private MetaData m_metadata;
		private RectangleF m_minimapRect;

		private string m_mapName;
		private string m_mapPath;
		private string m_appPath;

		private Image m_minimap;

		private static Dictionary<string, Image> m_minimaps = new Dictionary<string, Image>();

		#region Constructors

		public CMiniMapMan(string mapName, string mapPath, string appPath, SharedData sd )
		{
			m_mapName = mapName;
			m_mapPath = mapPath;
			m_appPath = appPath;
			LoadData(sd);
		}

		#endregion

		#region Protected functions

		protected string GetMapSubPath()
		{
			if (m_mapPath.Length > 0)
				return m_mapPath;
			else
				return Path.Combine("Levels", Path.Combine("K01", m_mapName));
		}

		protected void LoadData(SharedData sd)
		{
			if ( !Directory.Exists(Settings.Default.GamePath) )
			{
				System.Windows.Forms.MessageBox.Show("Incorrect game path: directory \"" + Settings.Default.GamePath + "\" not found");
				return;
			}
			
			string mapPath = Path.Combine(Settings.Default.GamePath,
																		Path.Combine("GameCrysis2",
																		Path.Combine(GetMapSubPath(), m_mapName + ".xml")));
			XmlSerializer serializer = new XmlSerializer(typeof(MetaData));
			try
			{
				if (File.Exists(mapPath))
				{
					using (FileStream stream = new FileStream(mapPath, FileMode.Open, FileAccess.Read))
					{
						m_metadata = (MetaData)serializer.Deserialize(stream);
					}

					LoadMiniMap();
					if (m_minimap != null && m_minimapRect != null)
					{
						CoordTransformData val = 
							new CoordTransformData(
																			m_minimapRect.Left,
																			m_minimapRect.Top, 
																			m_minimap.Width / m_minimapRect.Width,
																			m_minimap.Height / m_minimapRect.Height
																		);

						if (!sd.m_coordTransformData.ContainsKey(m_mapName))
							sd.m_coordTransformData.Add(m_mapName, val);
					}
				}
				else
				{
					System.Windows.Forms.MessageBox.Show("Cannot load map: file \"" + mapPath + "\" not found");
				}
			}
			catch (Exception e)
			{
				System.Windows.Forms.MessageBox.Show("Error" + e.ToString());
			}
		}

		protected void LoadMiniMap()
		{
			string mmpath = "";

			foreach (object o in m_metadata.Items)
			{
				if (o is MetaDataMiniMap)
				{
					MetaDataMiniMap mdmm = (o as MetaDataMiniMap);
					float startX = float.Parse(mdmm.startX, CFloatingPointFormat.Get());
					float endX = float.Parse(mdmm.endX, CFloatingPointFormat.Get());
					float startY = float.Parse(mdmm.startY, CFloatingPointFormat.Get());
					float endY = float.Parse(mdmm.endY, CFloatingPointFormat.Get());


					if (mdmm.Filename != null && mdmm.Filename != string.Empty)
					{
						mmpath = mdmm.Filename;
					}
					else if (mdmm.Filename0 != null & mdmm.Filename0 != string.Empty)
					{
						mmpath = mdmm.Filename0;
					}
					m_minimapRect = new RectangleF(startX, startY, endX - startX, endY - startY);
				}
			}
			if (mmpath == null || mmpath == string.Empty)
				return;

			mmpath = Path.Combine(Settings.Default.GamePath,
														Path.Combine("GameCrysis2",																
														Path.Combine(GetMapSubPath(), mmpath)));



			m_minimap = CreateMiniMap(mmpath);
		}

		protected Image LoadMiniMapTexture(string path)
		{
			Bitmap bmp;

			DateTime ftime = System.IO.File.GetLastWriteTime(path);

			string cachePath = Path.Combine(m_appPath, "MMCache");
			if (!Directory.Exists(cachePath))
			{
				Directory.CreateDirectory(cachePath);
			}
			string mmCacheName = Path.GetFileNameWithoutExtension(path) + ".jpg";
			string fullName = Path.Combine(cachePath, mmCacheName);
			if (File.Exists(fullName))
			{
				DateTime ctime = File.GetLastWriteTime(fullName);
				if(ctime > ftime)
					return Image.FromFile(fullName);
			}

			DdsFile ddsFile = new DdsFile();
			ddsFile.Load(new FileStream(path, FileMode.Open, FileAccess.Read));
			bmp = new Bitmap(ddsFile.GetWidth(), ddsFile.GetHeight());
			byte[] readPixelData = ddsFile.GetPixelData();

			for (int y = 0; y < ddsFile.GetHeight(); y++)
			{
				for (int x = 0; x < ddsFile.GetWidth(); x++)
				{
					int readPixelOffset = (y * ddsFile.GetWidth() * 4) + (x * 4);
					bmp.SetPixel(x, y, Color.FromArgb(readPixelData[readPixelOffset + 3],
																						readPixelData[readPixelOffset + 0],
																						readPixelData[readPixelOffset + 1],
																						readPixelData[readPixelOffset + 2]));
				}
			}
			bmp.Save(fullName, ImageFormat.Jpeg);
			return bmp;
		}

		protected Image CreateMiniMap(string path)
		{
			if (m_minimaps.ContainsKey(path))
			{
				return m_minimaps[path];
			}
			else
			{
				Image bmp = LoadMiniMapTexture(path);
				m_minimaps[path] = bmp;
				return bmp;
			}
		}
		#endregion

		#region Public properties

		public Image MiniMap
		{
			get
			{
				return m_minimap;
			}
		}

		#endregion
	}
}
