﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using StatsDataSource.Storage;
using StatsDataSource.ObjectModel;
using System.Collections;
using StatsDataSource.Core;
using StatsTool2.Filters;
using StatsDataSource.Filters;

namespace StatsTool2
{
	public partial class ElementView : UserControl, IFilteredStatsListener
	{
		//////////////////////////////////////////////////////////////////////////

		private Dictionary<string, ListViewItem> m_name2item = new Dictionary<string, ListViewItem>();
		private Dictionary<Locator, ListViewItem> m_loc2item = new Dictionary<Locator, ListViewItem>();
		private ListViewSorter m_lvSorter = new ListViewSorter();

		//////////////////////////////////////////////////////////////////////////

		public ElementView()
		{
			InitializeComponent();
			lvElements.ListViewItemSorter = m_lvSorter;
		}

		private void ElementView_Load(object sender, EventArgs e)
		{
			if (Program.StatsRepository != null)
				Program.StatsRepository.RegisterFilteredListener(this);
		}

		//////////////////////////////////////////////////////////////////////////

		#region Service
		private string[] GetItemName(GameElement el, out string uid)
		{
			uid = null;

			string[] names = new string[3];
			names[0] = el.Name;
			names[1] = "";
			names[2] = el.Parent.Desc.Name;

			if (el.Desc.KeyStates.Count == 0)
				return names;

			StringBuilder bld = new StringBuilder(el.Desc.KeyStates.Count * 40);
			foreach (var pk in el.Desc.KeyStates)
			{
				StatDesc desc = Program.StatsRepository.Registry.GetStateDesc(pk);
				if (desc == null)
					continue;

				GameState s;
				if (el.States.TryGetValue(desc.ID, out s))
				{
					if (!string.IsNullOrEmpty(s.Value))
					{
						if (bld.Length > 0)
							bld.Append(", ");

						bld.Append(s.Desc.Name);
						bld.Append(": ");
						bld.Append(s.Value);
					}
				}
			}

			names[1] = bld.ToString();
			uid = names[0] + names[1] + names[2];
			return names;
		}

		private string GetItemUID(ListViewItem it)
		{
			return it.SubItems[0].Text + it.SubItems[1].Text + it.SubItems[2].Text;
		}

		private void CreateListItem(GameElement el)
		{
			string uid;
			var name = GetItemName(el, out uid);

			ListViewItem it;
			if (!m_name2item.TryGetValue(uid, out it))
			{
				it = new ListViewItem(name);
				it.Tag = new List<GameElement>();
				lvElements.Items.Add(it);
				m_name2item.Add(uid, it);
			}

			var elems = (List<GameElement>)it.Tag;
			elems.Add(el);
			m_loc2item.Add(el.Locator, it);
		}

		private void RemoveListItem(GameElement el)
		{
			var it = m_loc2item[el.Locator];
			m_loc2item.Remove(el.Locator);

			var elems = (List<GameElement>)it.Tag;
			elems.Remove(el);

			if (elems.Count == 0)
			{
				string uid;
				GetItemName(el, out uid);
				m_name2item.Remove(uid);
				it.Remove();
			}
		}

		private void ClearList()
		{
			m_name2item.Clear();
			m_loc2item.Clear();
			lvElements.Items.Clear();
			lvElements.Groups.Clear();
		}

		//////////////////////////////////////////////////////////////////////////

		void UpdateView()
		{
			lvElements.BeginUpdate();
			ClearList();
			Program.StatsRepository.AcceptVisitorFiltered(new ElementUpdateVisitor(this));
			lvElements.EndUpdate();
		}

		//////////////////////////////////////////////////////////////////////////

		void UpdateItem(GameElement el, ListViewItem it)
		{
			string oldUid = GetItemUID(it);
			var elems = (List<GameElement>)it.Tag;
			elems.Remove(el);
			m_loc2item.Remove(el.Locator);

			if (elems.Count == 0)
			{
				m_name2item.Remove(oldUid);
				it.Remove();
			}

			CreateListItem(el);
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region IStatsListener

		private bool inModificationBatch = false;

		public void ModificationBatchStarted()
		{
			inModificationBatch = true;
		}

		public void ModificationBatchEnded(IncrementalUpdate update)
		{
			inModificationBatch = false;

			if (update == null)
			{
				UpdateView();
			}
			else
			{
				foreach (var n in update.AddedNodes)
					RepositoryNodeAdded(n);

				foreach (var s in update.AddedStates)
					if (!update.AddedNodes.Contains(s.Node))
						RepositoryStateAdded(s);
			}
		}

		public void RepositoryNodeAdded(GameNode node)
		{
			if (inModificationBatch)
			return;

			if(node.FilterState == EFilterState.Failed)
				return;

			if (!node.Locator.IsScope())
				CreateListItem((GameElement)node);
		}

		public void RepositoryStateAdded(GameState state)
		{
			if (inModificationBatch)
				return;

			if (!state.Node.Locator.IsScope())
			{
				GameElement el = (GameElement)state.Node;

				ListViewItem it;
				if (m_loc2item.TryGetValue(state.Node.Locator, out it))
				{
					if (el.FilterState == EFilterState.Failed)
						RemoveListItem(el);
					else if (el.Desc.KeyStates.Contains(state.Desc.Name))
						UpdateItem(el, it);
				}
				else if (el.FilterState != EFilterState.Failed)
				{
					CreateListItem(el);
				}
			}
		}

		public void RepositoryEventGroupAdded(GameEventGroup group)
		{
		}

		public void RepositoryEventAdded(GameEvent evnt)
		{
		}

		public void RepositoryFilterChanged(CompoundFilter active, EFilterUpdateType type)
		{
			if (type != EFilterUpdateType.Fast_Event && type != EFilterUpdateType.Regular_Event)
				UpdateView();
		}
		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region List view

		private void lvElements_ColumnClick(object sender, ColumnClickEventArgs e)
		{
			if (e.Column == m_lvSorter.SortColumn)
			{
				if (m_lvSorter.Order == SortOrder.Ascending)
					m_lvSorter.Order = SortOrder.Descending;
				else
					m_lvSorter.Order = SortOrder.Ascending;
			}
			else
			{
				m_lvSorter.SortColumn = e.Column;
				m_lvSorter.Order = SortOrder.Ascending;
			}

			lvElements.Sort();
		}

		private void lvElements_ItemDrag(object sender, ItemDragEventArgs e)
		{
			var item = e.Item as ListViewItem;
			if(item != null)
				DoDragDrop(((List<GameElement>)item.Tag)[0], DragDropEffects.Copy);
		}

		private void lvElements_SelectedIndexChanged(object sender, EventArgs e)
		{
			UpdateSelection();
		}

		private void UpdateSelection()
		{
			if (!updateSelectionTimer.Enabled)
				updateSelectionTimer.Start();
		}

		private void updateSelectionTimer_Tick(object sender, EventArgs e)
		{
			var l = new List<GameElement>();

			if (lvElements.SelectedItems.Count != 0)
				for (int i = 0; i != lvElements.SelectedItems.Count; ++i)
					l.AddRange((List<GameElement>)lvElements.SelectedItems[i].Tag);

			Program.StatsCore.CurrentElements = l;

			updateSelectionTimer.Stop();
		}

		#endregion

	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	class ElementUpdateVisitor : IStatsVisitor
	{
		private ElementView m_view;

		public ElementUpdateVisitor(ElementView view)
		{
			m_view = view;
		}

		public bool VisitScope(GameScope scope) { return true; }
		public void LeaveScope(GameScope scope) { }

		public bool VisitElement(GameElement elem) 
		{ 
			m_view.RepositoryNodeAdded(elem); 
			return false; 
		}
		public void LeaveElement(GameElement elem) { }

		public bool VisitState(GameState state) { return false; }
		public void LeaveState(GameState state) { }

		public bool StartEventGroups(GameNode node) { return false; }
		public bool VisitEventGroup(GameEventGroup evntGroup) { return false; }
		public void LeaveEventGroup(GameEventGroup evntGroup) { }
		public void EndEventGroups(GameNode node) { }

		public void VisitEvent(GameEvent evnt) { }
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	class ListViewSorter : IComparer
	{
		public int SortColumn { get; set; }
		public SortOrder Order { get; set; }
		private CaseInsensitiveComparer ObjectCompare;

		public ListViewSorter()
		{
			SortColumn = 0;
			Order = SortOrder.None;
			ObjectCompare = new CaseInsensitiveComparer();
		}

		public int Compare(object x, object y)
		{
			int compareResult;
			ListViewItem listviewX, listviewY;
			listviewX = (ListViewItem)x;
			listviewY = (ListViewItem)y;

			compareResult = ObjectCompare.Compare(listviewX.SubItems[SortColumn].Text, listviewY.SubItems[SortColumn].Text);

			if (Order == SortOrder.Ascending)
				return compareResult;
			else if (Order == SortOrder.Descending)
				return (-compareResult);
			else
				return 0;
		}
	}

}
