﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using StatsDataSource.Storage;
using StatsDataSource.ObjectModel;
using StatsDataSource.Core;
using StatsDataSource.Filters;

namespace StatsTool2
{
	public partial class EventView : UserControl, IFilteredStatsListener
	{
		private Dictionary<int, TreeNode> m_groupid2node = new Dictionary<int, TreeNode>();

		public EventView()
		{
			InitializeComponent();
		}

		private void EventView_Load(object sender, EventArgs e)
		{
			if (Program.StatsRepository != null)
			{
				Program.StatsRepository.RegisterFilteredListener(this);
				Program.StatsCore.NodesChanged += StatsCore_NodesChanged;
			}
		}

		//////////////////////////////////////////////////////////////////////////

		#region Service
		private string GetEventName(GameEvent evnt)
		{
			StringBuilder bld = new StringBuilder(evnt.Parameters.Count * 30 + 20);
			bld.Append('[');
			bld.AppendFormat("{0:00}:{1:00}:{2:00}", evnt.Time.Hours, evnt.Time.Minutes, evnt.Time.Seconds);
			bld.Append(']');

			foreach (var prm in evnt.Parameters)
			{
				bld.Append(' ');
				bld.Append(prm.Key);
				bld.Append(" = ");
				bld.Append(prm.Value);
				bld.Append(';');
			}

			return bld.ToString();
		}

		private TreeNode AddGroup(GameEventGroup group)
		{
			TreeNode node;
			if (!m_groupid2node.TryGetValue(group.Desc.ID, out node))
			{
				node = new TreeNode(group.Desc.Name);
				tvEvents.Nodes.Add(node);
				m_groupid2node.Add(group.Desc.ID, node);
				node.Tag = new List<GameEventGroup>();
			}

			var groups = (List<GameEventGroup>)node.Tag;
			groups.Add(group);

			if (node.Nodes.Count == 0 && group.Events.Count > 0)
				node.Nodes.Add("dummy");

			return node;
		}

		private void ClearTree()
		{
			tvEvents.Nodes.Clear();
			m_groupid2node.Clear();
		}

		//////////////////////////////////////////////////////////////////////////

		void StatsCore_NodesChanged(StatsCore sender, List<GameNode> nodes)
		{
			tvEvents.BeginUpdate();

			ClearTree();

			if (nodes.Count > 0)
			{
				EventUpdateVisitor v = new EventUpdateVisitor(this, nodes);
				Program.StatsRepository.AcceptVisitorFiltered(v);
			}

			tvEvents.EndUpdate();
		}

		private void tvEvents_ItemDrag(object sender, ItemDragEventArgs e)
		{
			var item = e.Item as TreeNode;
			if (item != null && item.Tag != null)
			{
				if(item.Tag is List<GameEventGroup>)
					DoDragDrop(((List<GameEventGroup>)item.Tag)[0], DragDropEffects.Copy);
				else
					DoDragDrop(((GameEvent)item.Tag), DragDropEffects.Copy);
			}
		}

		private void tvEvents_BeforeExpand(object sender, TreeViewCancelEventArgs e)
		{
			if (e.Node.Nodes.Count != 1 || e.Node.Nodes[0].Text != "dummy")
				return;

			var groups = (List<GameEventGroup>)e.Node.Tag;
			e.Node.Nodes.Clear();

			List<GameEvent> combined = new List<GameEvent>();
			
			foreach (var g in groups)
				combined.AddRange(g.Events);

			combined.Sort(new GameEventGroup.EventTimeComparer());

			tvEvents.BeginUpdate();

			foreach (var evnt in combined)
			{
				if (evnt.FilterState != EFilterState.Failed)
				{
					var item = new TreeNode(GetEventName(evnt));
					item.Tag = evnt;
					e.Node.Nodes.Add(item);
				}
			}

			tvEvents.EndUpdate();
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region IStatsListener

		private bool inModificationBatch = false;

		public void ModificationBatchStarted()
		{
			inModificationBatch = true;
		}

		public void ModificationBatchEnded(IncrementalUpdate update)
		{
			inModificationBatch = false;

			if (update == null)
			{
				StatsCore_NodesChanged(Program.StatsCore, Program.StatsCore.CurrentNodes);
			}
			else
			{
				foreach (var g in update.AddedEventGroups)
					RepositoryEventGroupAdded(g);

				foreach (var e in update.AddedEvents)
					RepositoryEventAdded(e);
			}
		}

		public void RepositoryNodeAdded(GameNode node)
		{
		}

		public void RepositoryStateAdded(GameState state)
		{
		}

		public void RepositoryEventGroupAdded(GameEventGroup group)
		{
			if (inModificationBatch)
				return;

			if (group.FilterState != EFilterState.Failed && Program.StatsCore.CurrentNodes.Contains(group.Node))
				AddGroup(group);
		}

		public void RepositoryEventAdded(GameEvent evnt)
		{
			if (inModificationBatch || evnt.FilterState == EFilterState.Failed)
				return;

			TreeNode node;
			if (m_groupid2node.TryGetValue(evnt.Group.Desc.ID, out node))
			{
				if (node.Nodes.Count != 1 || node.Nodes[0].Text != "dummy")
				{	
					var item = new TreeNode(GetEventName(evnt));
					item.Tag = evnt;
					node.Nodes.Add(item);
				}
			}
		}

		public void RepositoryFilterChanged(CompoundFilter active, EFilterUpdateType type)
		{
			if (type != EFilterUpdateType.Fast_Event)
				StatsCore_NodesChanged(Program.StatsCore, Program.StatsCore.CurrentNodes);
		}
		#endregion

		//////////////////////////////////////////////////////////////////////////
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	class EventUpdateVisitor : IStatsVisitor
	{
		private EventView m_view;
		private List<GameNode> m_nodes;

		public EventUpdateVisitor(EventView view, List<GameNode> nodes)
		{
			m_view = view;
			m_nodes = nodes;
		}

		public bool VisitScope(GameScope scope) { return true; }
		public void LeaveScope(GameScope scope) { }

		public bool VisitElement(GameElement elem) { return true; }
		public void LeaveElement(GameElement elem) { }

		public bool VisitState(GameState state) { return false; }
		public void LeaveState(GameState state) { }

		public bool StartEventGroups(GameNode node) { return m_nodes.Contains(node); }
		public bool VisitEventGroup(GameEventGroup evntGroup) 
		{
			m_view.RepositoryEventGroupAdded(evntGroup);
			return false; 
		}
		public void LeaveEventGroup(GameEventGroup evntGroup) { }
		public void EndEventGroups(GameNode node) { }
		public void VisitEvent(GameEvent evnt) { }
	}
}
