﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using StatsDataSource.Storage;
using StatsDataSource.ObjectModel;
using Zzzz;
using StatsTool2.Filters;
using System.IO;
using System.Xml.Serialization;
using System.Reflection;
using System.Diagnostics;
using StatsDataSource.Filters;

namespace StatsTool2
{
	public partial class FilterControl : UserControl, IStatsListener
	{
		public FilterControl()
		{
			InitializeComponent();

			rangeBar.ToolTipCallback = new TimeToolTipFmt();
			TotalMaximum = 100;
			rangeBar.RangeMaximum = TotalMaximum;
		}

		private int TotalMaximum
		{
			get { return rangeBar.TotalMaximum; }
			set
			{
				rangeBar.TotalMaximum = value;
				TimeSpan t = TimeSpan.FromMilliseconds(rangeBar.TotalMaximum);
				lMaxTime.Text = string.Format("{0:00}:{1:00}:{2:00}", t.Hours, t.Minutes, t.Seconds);
			}
		}

		private string GetPresetsDir()
		{
			string path = Path.Combine( Program.Location, "Presets" );
			if(!Directory.Exists(path))
				Directory.CreateDirectory(path);
			return path;
		}

		static readonly string ACTIVE_SET_FILE = "_active_.xml";
		private string GetActiveSetFile()
		{
			return Path.Combine(GetPresetsDir(), ACTIVE_SET_FILE);
		}

		private void FilterControl_Load(object sender, EventArgs e)
		{
			if (Program.StatsRepository != null)
			{
				Program.StatsRepository.RegisterListener(this);
				Program.MainForm.FormClosing += MainForm_FormClosing;
				LoadActiveFilters(GetActiveSetFile());
				LoadPresets(GetPresetsDir());
			}
		}

		void MainForm_FormClosing(object sender, FormClosingEventArgs e)
		{
			SaveActiveFilters(GetActiveSetFile());
		}

		//////////////////////////////////////////////////////////////////////////

		#region IStatsListener

		public void ModificationBatchStarted()
		{
		}

		public void ModificationBatchEnded(IncrementalUpdate update)
		{
		}

		public void RepositoryNodeAdded(GameNode node)
		{
		}

		public void RepositoryStateAdded(GameState state)
		{
		}

		public void RepositoryEventGroupAdded(GameEventGroup group)
		{
		}

		public void RepositoryEventAdded(GameEvent evnt)
		{
			int max = Math.Max((int)evnt.TimeMillisecs, rangeBar.TotalMaximum);
			if (rangeBar.TotalMaximum != max)
			{
				bool move = rangeBar.TotalMaximum == rangeBar.RangeMaximum;
				TotalMaximum = max;

				if (move)
					rangeBar.RangeMaximum = max;
			}
		}
		#endregion

		#region Slider

		private void rangeBar_RangeChanging(object sender, EventArgs e)
		{
			Int64 oldMin = Program.FilterManager.EventTimeFilter.MinTime;
			Int64 oldMax = Program.FilterManager.EventTimeFilter.MaxTime;

			Program.FilterManager.EventTimeFilter.MinTime = rangeBar.RangeMinimum;
			Program.FilterManager.EventTimeFilter.MaxTime = rangeBar.RangeMaximum;

			if ((Control.ModifierKeys & Keys.Shift) == Keys.Shift)
				return;

			if(oldMin != Int64.MinValue && oldMax != Int64.MaxValue)
				Program.StatsRepository.UpdateFilter(
					EFilterUpdateType.Fast_Event, 
					new EventTimeUpdater(oldMin, oldMax,
						Program.FilterManager.EventTimeFilter.MinTime,
						Program.FilterManager.EventTimeFilter.MaxTime,
						Program.StatsRepository.GetActiveFilter()));
		}

		private void rangeBar_RangeChanged(object sender, EventArgs e)
		{
			Program.FilterManager.EventTimeFilter.MinTime = rangeBar.RangeMinimum;
			Program.FilterManager.EventTimeFilter.MaxTime = rangeBar.RangeMaximum;

			Program.StatsRepository.UpdateFilter(EFilterUpdateType.Regular_Event, null);
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Filter controls

		private string ChooseNewFilterName()
		{
			int c = 1;
			string name = "Filter1";
			for (int i = 0; i != tvFilters.Nodes.Count; ++i)
			{
				if (tvFilters.Nodes[i].Text == name)
				{
					name = "Filter" + (++c).ToString();
					i = 0;
				}
			}
			return name;
		}

		private TreeNode CreateNewFilter(FilterBase f)
		{
			Debug.Assert(f is CompoundANDFilter);

			var n = new TreeNode(f.Name);
			n.Tag = f;
			n.Checked = f.Enabled;

			Program.FilterManager.RootFilter.AddChild(f);

			tvFilters.Nodes.Add(n);
			tsbSave.Enabled = tvFilters.Nodes.Count > 0;

			return n;
		}

		private TreeNode CreateNewFilter()
		{			
			var f = new CompoundANDFilter();
			f.Name = ChooseNewFilterName();

			var n = CreateNewFilter(f);
			
			tvFilters.SelectedNode = n;
			n.BeginEdit();

			return n;
		}

		//////////////////////////////////////////////////////////////////////////

		private void btnNewFilter_Click(object sender, EventArgs e)
		{
			CreateNewFilter();
		}

		private void tvFilters_BeforeLabelEdit(object sender, NodeLabelEditEventArgs e)
		{
			e.CancelEdit = e.Node.Parent != null;
		}

		private bool m_updatingChecks = false;
		private void tvFilters_AfterCheck(object sender, TreeViewEventArgs e)
		{
			if(m_updatingChecks) return;

			m_updatingChecks = true;

			var f = e.Node.Tag as FilterBase;
			f.Enabled = e.Node.Checked;

			CheckNodesRec(e.Node, e.Node.Checked);

			Program.StatsRepository.UpdateFilter(EFilterUpdateType.Regular, null);

			m_updatingChecks = false;
		}

		private void CheckNodesRec(TreeNode n, bool check)
		{
			if (n.Checked != check)
				n.Checked = check ? ((FilterBase)n.Tag).Enabled : false;

			for (int i = 0; i != n.Nodes.Count; ++i)
				CheckNodesRec(n.Nodes[i], check);
		}

		//////////////////////////////////////////////////////////////////////////

		private void AddNewRule(TreeNode parent, FilterBase filter)
		{
			var parenFilter = parent.Tag as CompoundFilter;
			if (parenFilter.MaxChildren == parenFilter.SubFilters.Count)
				return;

			TreeNode ruleNode = new TreeNode(filter.Name);
			ruleNode.Tag = filter;
			ruleNode.Checked = filter.Enabled;

			parenFilter.AddChild(filter);

			parent.Nodes.Add(ruleNode);
			parent.Expand();

			Program.StatsRepository.UpdateFilter(EFilterUpdateType.Regular, null);
		}

		private void btnNewRule_Click(object sender, EventArgs e)
		{
			FilterWizard wizard = new FilterWizard();
			if (wizard.ShowDialog(this) == DialogResult.OK)
			{
				AddNewRule(tvFilters.SelectedNode, wizard.ResultingFilter);
			}
		}

		private void btnClone_Click(object sender, EventArgs e)
		{
			tvFilters.BeginUpdate();
			var node = CloneRec(tvFilters.SelectedNode, null);
			node.Text = tvFilters.SelectedNode.Text + " copy";
			tvFilters.EndUpdate();
		}

		private TreeNode CloneRec(TreeNode toClone, TreeNode parent)
		{
			TreeNode node = new TreeNode(toClone.Text);
			var newFlter = (FilterBase) ((FilterBase)toClone.Tag).Clone();
			node.Tag = newFlter;
			node.Checked = newFlter.Enabled;

			if (parent == null)
			{
				tvFilters.Nodes.Add(node);
				Program.FilterManager.RootFilter.AddChild(newFlter);
			}
			else
			{
				parent.Nodes.Add(node);
				((CompoundFilter)parent.Tag).AddChild(newFlter);
			}

			for (int i = 0; i != toClone.Nodes.Count; ++i)
				CloneRec(toClone.Nodes[i], node);

			node.Expand();
			return node;
		}

		private void btnRemove_Click(object sender, EventArgs e)
		{
			tvFilters.BeginUpdate();
			RemoveRec(tvFilters.SelectedNode);
			tvFilters.SelectedNode.Remove();
			tvFilters.EndUpdate();

			tsbSave.Enabled = tvFilters.Nodes.Count > 0;

			Program.StatsRepository.UpdateFilter(EFilterUpdateType.Regular, null);
		}

		private void RemoveRec(TreeNode n)
		{
			for (int i = 0; i != n.Nodes.Count; ++i)
				RemoveRec(n.Nodes[i]);

			var f = (FilterBase)n.Tag;
			f.Parent.RemoveChild(f);
		}

		private void btnEdit_Click(object sender, EventArgs e)
		{
			var filter = (FilterBase)tvFilters.SelectedNode.Tag;

			var wizard = new FilterWizard(filter);

			if (wizard.ShowDialog(this) == DialogResult.OK)
			{
				var prnt = filter.Parent;
				prnt.RemoveChild(filter);

				filter = wizard.ResultingFilter;
				prnt.AddChild(filter);

				tvFilters.SelectedNode.Tag = filter;
				tvFilters.SelectedNode.Text = filter.Name;
			}

			Program.StatsRepository.UpdateFilter(EFilterUpdateType.Regular, null);
		}

		#endregion

		#region Filter persistance

		#region Presets

		private string GeneratePresetName()
		{
			string name = "Preset ";
			string resName;
			int i = 0;

			while (File.Exists(resName = Path.Combine(GetPresetsDir(), name + i.ToString() + ".xml")))
				++i;

			return resName;
		}

		private void tsbSave_Click(object sender, EventArgs e)
		{
			string name = GeneratePresetName();
			SaveActiveFilters(name);
			var lvi = new ListViewItem(new string[] { Path.GetFileNameWithoutExtension(name) });
			lvPresets.Items.Add(lvi);
			lvi.BeginEdit();
		}

		private void LoadPresets(string dir)
		{
			string[] files = Directory.GetFiles(dir, "*.xml");
			foreach (var ps in files)
			{
				if (Path.GetFileName(ps) == ACTIVE_SET_FILE)
					continue;

				string n = Path.GetFileNameWithoutExtension(ps);
				lvPresets.Items.Add(new ListViewItem(new string[] { n }));
			}
		}

		#endregion

		#region Active filters

		private void SaveActiveFilters(string filename)
		{
			var cont = new FilterPersistanceContainer() { Filters = new List<CompoundFilter>() };

			for (int i = 0; i != tvFilters.Nodes.Count; ++i)
			{
				var node = tvFilters.Nodes[i];
				cont.Filters.Add((CompoundFilter)node.Tag);
			}

			SaveFiltersToFile(filename, cont);
		}

		private void LoadActiveFilters(string filename)
		{
			var container = LoadFiltersFromFile(filename);
			if (container != null)
			{
				tvFilters.SuspendLayout();
				RestoreFilters(container);
				tvFilters.ResumeLayout();
			}
		}

		#endregion

		#region Serialization

		private void SaveFiltersToFile(string file, FilterPersistanceContainer tree)
		{
			try
			{
				using (var fs = new FileStream(file, FileMode.Create))
				{
					var ser = new XmlSerializer(typeof(FilterPersistanceContainer));
					ser.Serialize(fs, tree);
				}
			}
			catch
			{
				MessageBox.Show(Program.MainForm, "Failed to save filter data.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
		}

		private FilterPersistanceContainer LoadFiltersFromFile(string file)
		{
			try
			{
				if (!File.Exists(file))
					return null;

				using (var fs = new FileStream(file, FileMode.Open))
				{
					var ser = new XmlSerializer(typeof(FilterPersistanceContainer));
					var container = (FilterPersistanceContainer)ser.Deserialize(fs);
					return container;
				}
			}
			catch
			{
				return null;
			}
		}

		#endregion

		#region Restoring

		private void RestoreFilters(FilterPersistanceContainer container)
		{
			foreach (var f in container.Filters)
			{
				var n = CreateNewFilter(f);
				RestoreFiltersRec(n, f);
			}
			Program.StatsRepository.UpdateFilter(EFilterUpdateType.Regular, null);
		}

		private void RestoreFiltersRec(TreeNode parent, CompoundFilter f)
		{
			foreach (var c in f.SubFilters)
			{
				c.Parent = f;
				
				var n = new TreeNode(c.Name);
				n.Tag = c;
				n.Checked = c.Enabled && parent.Checked;
				parent.Nodes.Add(n);

				if (c is CompoundFilter)
					RestoreFiltersRec(n, (CompoundFilter)c);
			}
		}

		#endregion

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Presets

		private string GetPresetName(ListViewItem it)
		{
			return Path.Combine(GetPresetsDir(), it.SubItems[0].Text + ".xml");
		}

		private void DeletePreset(ListViewItem ps)
		{
			try
			{
				string file = GetPresetName(ps);
				ps.Remove();
				File.Delete(file);
			}
			catch
			{ }
		}

		private void lvPresets_AfterLabelEdit(object sender, LabelEditEventArgs e)
		{
			if(string.IsNullOrEmpty(e.Label))
			{
				e.CancelEdit = true;
				return;
			}

			try
			{
				string srcName = lvPresets.Items[e.Item].SubItems[0].Text;
				srcName = Path.Combine(GetPresetsDir(), srcName + ".xml");
				string destName = Path.Combine(GetPresetsDir(), e.Label + ".xml");
				File.Move(srcName, destName);
			}
			catch
			{
				e.CancelEdit = true;
			}
		}

		private void lvPresets_KeyUp(object sender, KeyEventArgs e)
		{
			if (e.KeyCode == Keys.F2 && lvPresets.SelectedItems.Count > 0)
				lvPresets.SelectedItems[0].BeginEdit();
			else if (e.KeyCode == Keys.Delete)
				foreach (var it in lvPresets.SelectedItems)
					DeletePreset((ListViewItem)it);
			else if (e.KeyCode == Keys.Enter)
				lvPresets_DoubleClick(null, null);
		}

		private void lvPresets_DoubleClick(object sender, EventArgs e)
		{
			if(lvPresets.SelectedItems.Count > 0)
			{
				var cont = LoadFiltersFromFile(GetPresetName(lvPresets.SelectedItems[0]));
				RestoreFilters(cont);
			}
		}

		private void lvPresets_ItemDrag(object sender, ItemDragEventArgs e)
		{
			DoDragDrop(e.Item, DragDropEffects.Copy);
		}

		#endregion

		#region Other events

		private void tvFilters_KeyUp(object sender, KeyEventArgs e)
		{
			if (tvFilters.SelectedNode != null && e.KeyCode == Keys.F2)
			{
				tvFilters.SelectedNode.BeginEdit();
			}
			else if (tvFilters.SelectedNode != null && e.KeyCode == Keys.Delete)
			{
				btnRemove_Click(null, null);
				tvFilters_AfterSelect(null, null);
			}
		}

		private void tvFilters_AfterSelect(object sender, TreeViewEventArgs e)
		{
			tsbNewRule.Enabled = tvFilters.SelectedNode != null && tvFilters.SelectedNode.Tag is CompoundFilter && !((CompoundFilter)tvFilters.SelectedNode.Tag).IsFull();
			tsbCopy.Enabled = tvFilters.SelectedNode != null && tvFilters.SelectedNode.Parent == null;
			tsbRemove.Enabled = tvFilters.SelectedNode != null;
			tsbEdit.Enabled = tvFilters.SelectedNode != null && tvFilters.SelectedNode.Tag is TerminalFilter;
			tsbANDGroup.Enabled = tsbNewRule.Enabled;
			tsbORGroup.Enabled = tsbNewRule.Enabled;
			tsbNOTGroup.Enabled = tsbNewRule.Enabled;
		}

		private void cmFilters_Opening(object sender, CancelEventArgs e)
		{
			cmbNewFilter.Enabled = true;
			cmbCopyFilter.Enabled = tsbCopy.Enabled;
			cmbNewRule.Enabled = tsbNewRule.Enabled;
			cmbEditRule.Enabled = tsbEdit.Enabled;
			cmbANDGroup.Enabled = tsbANDGroup.Enabled;
			cmbORGroup.Enabled = tsbORGroup.Enabled;
			cmbNOTGroup.Enabled = tsbNOTGroup.Enabled;
			cmbRemove.Enabled = tsbRemove.Enabled;
		}

		private void btnNewItem_Click(object sender, EventArgs e)
		{
			if (tvFilters.SelectedNode == null)
				btnNewFilter_Click(null, null);
			else
				btnNewRule_Click(null, null);
		}

		private void tvFilters_NodeMouseDoubleClick(object sender, TreeNodeMouseClickEventArgs e)
		{
			if (tsbEdit.Enabled)
				btnEdit_Click(null, null);
		}

		private void tvFilters_AfterLabelEdit(object sender, NodeLabelEditEventArgs e)
		{
			if (string.IsNullOrEmpty(e.Label))
				e.CancelEdit = true;
			else
				((CompoundFilter)e.Node.Tag).Name = e.Label;
		}

		private void btnANDGroup_Click(object sender, EventArgs e)
		{
			AddNewRule(tvFilters.SelectedNode, new CompoundANDFilter());
		}

		private void btnORGroup_Click(object sender, EventArgs e)
		{
			AddNewRule(tvFilters.SelectedNode, new CompoundORFilter());
		}

		private void cmbNOTGroup_Click(object sender, EventArgs e)
		{
			AddNewRule(tvFilters.SelectedNode, new CompoundNOTFilter());
		}

		#endregion

		#region Drag and Drop

		private void tvFilters_ItemDrag(object sender, ItemDragEventArgs e)
		{
			if(e.Item is TreeNode && ((TreeNode)e.Item).Parent != null)
				DoDragDrop(e.Item, DragDropEffects.Move);
		}

		private void tvFilters_DragEnter(object sender, DragEventArgs e)
		{
			e.Effect = DragDropEffects.Move | DragDropEffects.Copy;
		}

		private void tvFilters_DragDrop(object sender, DragEventArgs e)
		{
			Point pt = tvFilters.PointToClient(new Point(e.X, e.Y));
			TreeNode DestinationNode = tvFilters.GetNodeAt(pt);

			CompoundFilter DestinationFilter = null;
			if (DestinationNode != null)
				DestinationFilter = DestinationNode.Tag as CompoundFilter;

			if (DestinationFilter != null && e.Data.GetDataPresent(typeof(TreeNode)))
			{
				var DraggedNode = (TreeNode)e.Data.GetData(typeof(TreeNode));
				var DraggedFilter = DraggedNode.Tag as FilterBase;

				if (DraggedFilter == DestinationFilter)
					return;

				DraggedFilter.Parent.RemoveChild(DraggedFilter);
				DestinationFilter.AddChild(DraggedFilter);
				DraggedNode.Remove();
				DestinationNode.Nodes.Add(DraggedNode);
				DestinationNode.Expand();

				Program.StatsRepository.UpdateFilter(EFilterUpdateType.Regular, null);
			}
			if (e.Data.GetDataPresent(typeof(ListViewItem)))
			{
				var item = (ListViewItem)e.Data.GetData(typeof(ListViewItem));
				if (item.ListView == lvPresets)
				{
					var cont = LoadFiltersFromFile(GetPresetName(item));
					RestoreFilters(cont);
				}
			}
			else if(e.Data.GetDataPresent(typeof(GameScope)))
			{
				var scope = (GameScope)e.Data.GetData(typeof(GameScope));
				
				if(DestinationFilter == null)
					DestinationNode = CreateNewFilter();

				FilterBase f;
				if(FilterWizard.CreateFilter(out f,
																		EAffectedType.Scope, scope.Desc.Name, true,
																		null, false, ECompareOperation.Unknown, null, null))
				{
					AddNewRule(DestinationNode, f);
				}
			}
			else if (e.Data.GetDataPresent(typeof(GameElement)))
			{
				var elem = (GameElement)e.Data.GetData(typeof(GameElement));

				if (DestinationFilter == null)
					DestinationNode = CreateNewFilter();

				FilterBase f;
				if (FilterWizard.CreateFilter(out f,
																		EAffectedType.Element, elem.Desc.Name, true,
																		null, false, ECompareOperation.Unknown, null, null))
				{
					AddNewRule(DestinationNode, f);
				}
			}
			else if (e.Data.GetDataPresent(typeof(GameState)))
			{
				var state = (GameState)e.Data.GetData(typeof(GameState));

				if (DestinationFilter == null)
					DestinationNode = CreateNewFilter();

				FilterBase f;
				if (FilterWizard.CreateFilter(out f,
																		state.Node.Locator.IsScope() ? EAffectedType.Scope : EAffectedType.Element, 
																		null, false,
																		GetStatePath(state), true,
																		state.Value == null ? ECompareOperation.Unknown : ECompareOperation.Equal, "==", state.Value))
				{
					AddNewRule(DestinationNode, f);
				}
			}
			else if (e.Data.GetDataPresent(typeof(GameEvent)))
			{
				var evnt = (GameEvent)e.Data.GetData(typeof(GameEvent));

				if (DestinationFilter == null)
					DestinationNode = CreateNewFilter();

				FilterBase f;
				if (FilterWizard.CreateFilter(out f,
																		EAffectedType.Event,
																		evnt.Group.Desc.Name, true,
																		null, false, ECompareOperation.Unknown, null, null))
				{
					AddNewRule(DestinationNode, f);
				}
			}
			else if (e.Data.GetDataPresent(typeof(GameEventGroup)))
			{
				var group = (GameEventGroup)e.Data.GetData(typeof(GameEventGroup));

				if (DestinationFilter == null)
					DestinationNode = CreateNewFilter();

				FilterBase f;
				if (FilterWizard.CreateFilter(out f,
																		EAffectedType.Event,
																		group.Desc.Name, true,
																		null, false, ECompareOperation.Unknown, null, null))
				{
					AddNewRule(DestinationNode, f);
				}
			}
		}

		private string GetStatePath(GameState state)
		{
			string path = state.Desc.Name;
			state = state.Parent;

			while (state != null)
			{
				path = state.Desc.Name + "." + path;
				state = state.Parent;
			}

			return path;
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	class TimeToolTipFmt : ToolTipFormatter
	{
		public override string GetValue(double val)
		{
			TimeSpan t = TimeSpan.FromMilliseconds((Int64)val);
			return string.Format("{0:00}:{1:00}:{2:00}", t.Hours, t.Minutes, t.Seconds);
		}
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	[Serializable]
	[XmlRoot("filters_data")]
	[ XmlInclude(typeof(CompoundORFilter))
	, XmlInclude(typeof(CompoundANDFilter))
	, XmlInclude(typeof(CompoundNOTFilter))
	, XmlInclude(typeof(EventFilter))
	, XmlInclude(typeof(NodeFilter)) ]
	public class FilterPersistanceContainer
	{
		[XmlArray("root_filters")]
		[XmlArrayItem("filter")]
		public List<CompoundFilter> Filters { get; set; }
	}

	////////////////////////////////////////////////////////////
	////////////////////////////////////////////////////////////

	class EventTimeUpdater : IStatsVisitor
	{
		Int64 m_oldMin;
		Int64 m_oldMax;
		Int64 m_newMin;
		Int64 m_newMax;
		IStatsFilter m_active;

		public EventTimeUpdater(Int64 oldMin, Int64 oldMax, Int64 newMin, Int64 newMax, IStatsFilter active)
		{
			m_oldMin = oldMin;
			m_oldMax = oldMax;
			m_newMin = newMin;
			m_newMax = newMax;
			m_active = active;
		}

		public bool VisitScope(GameScope scope) { return scope.FilterState != EFilterState.Failed; }
		public void LeaveScope(GameScope scope) { }

		public bool VisitElement(GameElement elem) { return elem.FilterState != EFilterState.Failed; }
		public void LeaveElement(GameElement elem) { }

		public bool VisitState(GameState state) { return false; }
		public void LeaveState(GameState state) { }

		public bool StartEventGroups(GameNode node) { return true; }

		public bool VisitEventGroup(GameEventGroup evntGroup)
		{
			if(evntGroup.FilterState == EFilterState.Failed || evntGroup.Events.Count == 0) 
				return false;

			Int64 eMin = evntGroup.Events[0].TimeMillisecs;
			Int64 eMax = evntGroup.Events[evntGroup.Events.Count - 1].TimeMillisecs;

			// Test if we can disable whole collection at once
			if (!RangesIntersecting(m_newMin, m_newMax, eMin, eMax))
			{
				if (RangesIntersecting(m_oldMin, m_oldMax, eMin, eMax))
					FailRange(evntGroup.Events, 0, evntGroup.Events.Count);
				
				return false;
			}

			int omin = evntGroup.LowerBound(m_oldMin);
			int omax = evntGroup.UpperBound(m_oldMax) - 1;
			int nmin = evntGroup.LowerBound(m_newMin);
			int nmax = evntGroup.UpperBound(m_newMax) - 1;

			if (nmin > omin)
				FailRange(evntGroup.Events, omin, nmin + 1);
			else if (nmin < omin)
				TestRange(evntGroup.Events, nmin, Math.Min(omin, nmax) + 1);

			if (nmax < omax)
				FailRange(evntGroup.Events, nmax, omax + 1);
			else if (nmax > omax)
				TestRange(evntGroup.Events, Math.Max(omax, nmin), nmax + 1);

			return false;
		}

		public void LeaveEventGroup(GameEventGroup evntGroup) { }
		public void EndEventGroups(GameNode node) { }
		public void VisitEvent(GameEvent evnt) { }


		private bool RangesIntersecting(Int64 min1, Int64 max1, Int64 min2, Int64 max2)
		{
			return !(max1 < min2 || min1 > max2);
		}

		private void FailRange(List<GameEvent> events, int start, int end)
		{
			for (int i = start; i != end; ++i)
				events[i].FilterState = EFilterState.Failed;
		}

		private void TestRange(List<GameEvent> events, int start, int end)
		{
			for (int i = start; i != end; ++i)
			{
				var e = events[i];
				e.FilterState = m_active != null ? m_active.TestEvent(e) : EFilterState.Passed;
			}
		}
	}

	////////////////////////////////////////////////////////////
	////////////////////////////////////////////////////////////

}
