﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using StatsDataSource.Storage;
using StatsDataSource.ObjectModel;
using StatsTool2.Filters;
using StatsDataSource.Filters;

namespace StatsTool2
{
	public partial class FilterWizard : Form
	{
		///////////////////////////////////////////////////////////////////
		EAffectedType m_affectedType = EAffectedType.Unknown;
		public EAffectedType CurrentType
		{
			get { return m_affectedType; }
			set { if (value != m_affectedType) { m_affectedType = value; OnTypeChanged(); } }
		}

		private FilterBase m_filter;
		internal FilterBase ResultingFilter { get { return m_filter; } }

		///////////////////////////////////////////////////////////////////

		public FilterWizard()
		{
			InitializeComponent();
		}

		internal FilterWizard(FilterBase filter2edit) : this()
		{
			m_filter = filter2edit;

			var nf = filter2edit as NodeFilter;
			if (nf != null)
			{
				cbType.SelectedIndex = (int)nf.m_type;
				cbName.Text = nf.m_name;
				cbName_TextUpdate(null, null);
				cbRequireNameMatch.Checked = nf.m_matchName;
				cbStatePath.Text = nf.m_pathOrig;
				cbStatePath_TextUpdate(null, null);
				cbMissingIsFailure.Checked = nf.m_missingFails;
				cbStateOperation.SelectedIndex = nf.m_op != ECompareOperation.Unknown ? (int)nf.m_op : -1;
				cbStateValue.Text = nf.m_value;
				cbRequireNameMatch_CheckedChanged(null, null);
			}

			var ef = filter2edit as EventFilter;
			if (ef != null)
			{
				cbType.SelectedIndex = (int)EAffectedType.Event;
				cbName.Text = ef.m_name;
				cbName_TextUpdate(null, null);
				cbRequireNameMatch.Checked = ef.m_matchName;
				cbStatePath.Text = ef.m_param;
				cbStatePath_TextUpdate(null, null);
				cbMissingIsFailure.Checked = ef.m_missingFails;
				cbStateOperation.SelectedIndex = ef.m_op != ECompareOperation.Unknown ? (int)ef.m_op : -1;
				cbStateValue.Text = ef.m_value;
				cbRequireNameMatch_CheckedChanged(null, null);
			}
		}

		///////////////////////////////////////////////////////////////////

		#region Interaction

		void OnTypeChanged()
		{
			cbStatePath.Enabled = CurrentType != EAffectedType.Unknown;
			FillNameCollection();
			FillStateCollection();
		}

		#endregion

		///////////////////////////////////////////////////////////////////

		#region Collections

		void FillNameCollection()
		{
			cbName.BeginUpdate();
			cbName.Items.Clear();

			SortedList<string, string> names = new SortedList<string, string>();

			if(CurrentType == EAffectedType.Scope || CurrentType == EAffectedType.Node)
				foreach (var el in Program.StatsRepository.Registry.GetAllScopes())
					names[el.Name] = el.Name;

			if (CurrentType == EAffectedType.Element || CurrentType == EAffectedType.Node)
				foreach (var el in Program.StatsRepository.Registry.GetAllElements())
					names[el.Name] = el.Name;

			if (CurrentType == EAffectedType.Event)
				foreach (var el in Program.StatsRepository.Registry.GetAllEvents())
					names[el.Name] = el.Name;

			foreach (var n in names)
				cbName.Items.Add(n.Key);

			cbName.EndUpdate();
		}

		void FillStateCollection()
		{
			cbStatePath.BeginUpdate();

			cbStatePath.Items.Clear();

			CollectorBase c;

			if (CurrentType == EAffectedType.Event)
				c = new EventParamCollector(this, cbName.Text);
			else
				c = new StatePathCollector(this, string.IsNullOrEmpty(cbName.Text) ? null : cbName.Text);

			Program.StatsRepository.AcceptVisitor(c);
			var res = c.ComposeResult();

			foreach (var r in res)
				cbStatePath.Items.Add(r.Key);

			cbStatePath.EndUpdate();
		}

		void FillValueCollection()
		{
			cbStateValue.BeginUpdate();

			cbStateValue.Items.Clear();

			FilterBase tmpFilter = null; 

			if (CreateFilter(out tmpFilter,
												CurrentType,
												cbName.Text,
												cbRequireNameMatch.Enabled ? cbRequireNameMatch.Checked : false,
												cbStatePath.Enabled ? cbStatePath.Text : null,
												false,
												ECompareOperation.Unknown,
												cbStateOperation.Text,
												null))
			{
				CollectorBase c;

				if (CurrentType == EAffectedType.Event)
					c = new EventValueCollector(this, cbStatePath.Text);
				else
					c = new StateValueCollector(this, cbStatePath.Text);

				Program.StatsRepository.AcceptVisitor(c);
				var res = c.ComposeResult();

				foreach (var it in res)
					cbStateValue.Items.Add(it.Key);
			}

			cbStateValue.EndUpdate();
		}

		#endregion

		///////////////////////////////////////////////////////////////////

		#region Global events

		private void cbType_SelectedIndexChanged(object sender, EventArgs e)
		{
			CurrentType = (cbType.SelectedIndex < 0) ? EAffectedType.Unknown : (EAffectedType)cbType.SelectedIndex;
		}

		private void cbName_Validated(object sender, EventArgs e)
		{
			FillStateCollection();
		}

		private void cbName_TextUpdate(object sender, EventArgs e)
		{
			cbRequireNameMatch.Enabled = !string.IsNullOrEmpty(cbName.Text);
		}

		private void cbRequireNameMatch_CheckedChanged(object sender, EventArgs e)
		{
			btnOk.Enabled = (cbStateValue.Enabled && !string.IsNullOrEmpty(cbStateValue.Text))
				|| (cbMissingIsFailure.Enabled && cbMissingIsFailure.Checked)
				|| (cbRequireNameMatch.Enabled && cbRequireNameMatch.Checked);
		}

		#endregion

		///////////////////////////////////////////////////////////////////

		#region State filter

		private void cbStatePath_TextUpdate(object sender, EventArgs e)
		{
			cbStateOperation.Enabled = !string.IsNullOrEmpty(cbStatePath.Text);
			cbMissingIsFailure.Enabled = !string.IsNullOrEmpty(cbStatePath.Text);
		}

		private void cbStatePath_Validated(object sender, EventArgs e)
		{
			cbStateOperation_SelectedIndexChanged(null, null);
		}

		private void cbStateOperation_SelectedIndexChanged(object sender, EventArgs e)
		{
			cbStateValue.Enabled = !string.IsNullOrEmpty(cbStateOperation.Text) && !string.IsNullOrEmpty(cbStatePath.Text);
			if(cbStateOperation.SelectedIndex >= 0)
				FillValueCollection();
		}

		private void cbStateOperation_EnabledChanged(object sender, EventArgs e)
		{
			cbStateOperation_SelectedIndexChanged(null, null);
		}

		private void cbStateValue_EnabledChanged(object sender, EventArgs e)
		{
			cbRequireNameMatch_CheckedChanged(null, null);
		}

		private void cbStateValue_TextUpdate(object sender, EventArgs e)
		{
			cbStateValue_EnabledChanged(null, null);
		}

		#endregion

		///////////////////////////////////////////////////////////////////

		#region Filter creation

		internal static bool CreateFilter(out FilterBase f,
															EAffectedType targetType,
															string name, bool matchName, 
															string statePath, bool missingFails, 
															ECompareOperation operation,
															string operation_str,
															string value)
		{
			f = null;

			if (targetType == EAffectedType.Scope || targetType == EAffectedType.Element || targetType == EAffectedType.Node)
			{
				f = new NodeFilter();
				((NodeFilter)f).Init(targetType, operation, name, matchName, statePath, missingFails, value);
			}
			else if (targetType == EAffectedType.Event)
			{
				f = new EventFilter();
				((EventFilter)f).Init(operation, name, matchName, statePath, missingFails, value);
			}

			f.Name = string.Format(
					"{0}{1}{2} {3}{4} {5} {6}",
					string.IsNullOrEmpty(name) ? targetType.ToString() : name,
					matchName ? "[!]" : "",
					string.IsNullOrEmpty(statePath) ? "" : ":",
					statePath,
					missingFails ? "[!]" : "",
					operation != ECompareOperation.Unknown ? operation_str : "",
					value ?? "").Trim();

			return true;
		}

		private void btnOk_Click(object sender, EventArgs e)
		{
			string value = cbStateValue.Enabled ? cbStateValue.Text : null;
			var op = cbStateOperation.SelectedIndex < 0 || !cbStateOperation.Enabled || string.IsNullOrEmpty(value)
					? ECompareOperation.Unknown : (ECompareOperation)cbStateOperation.SelectedIndex;

			CreateFilter(out m_filter,
				CurrentType,
				cbName.Text,
				cbRequireNameMatch.Enabled ? cbRequireNameMatch.Checked : false,
				cbStatePath.Enabled ? cbStatePath.Text : null,
				cbMissingIsFailure.Enabled ? cbMissingIsFailure.Checked : false,
				op,
				op != ECompareOperation.Unknown ? cbStateOperation.Text : null,
				op != ECompareOperation.Unknown ? value : null
				);
		}

		#endregion

		///////////////////////////////////////////////////////////////////
	}

	///////////////////////////////////////////////////////////////////
	///////////////////////////////////////////////////////////////////

	class CollectorBase : IStatsVisitor
	{
		protected FilterWizard m_wizard;
		protected SortedList<string, string> m_dict;

		public CollectorBase(FilterWizard wizard)
		{
			m_wizard = wizard;
			m_dict = new SortedList<string, string>();
		}

		public SortedList<string, string> ComposeResult()
		{
			return m_dict;
		}

		public virtual bool VisitScope(GameScope scope) { return true; }
		public virtual void LeaveScope(GameScope scope) { }
		public virtual bool VisitElement(GameElement elem) { return true; }
		public virtual void LeaveElement(GameElement elem) { }
		public virtual bool VisitState(GameState state) { return true; }
		public virtual void LeaveState(GameState state) { }
		public virtual bool StartEventGroups(GameNode node) { return true; }
		public virtual bool VisitEventGroup(GameEventGroup evntGroup) { return true; }
		public virtual void LeaveEventGroup(GameEventGroup evntGroup) { }
		public virtual void EndEventGroups(GameNode node) { }
		public virtual void VisitEvent(GameEvent evnt) { }
	}

	///////////////////////////////////////////////////////////////////
	///////////////////////////////////////////////////////////////////

	class StatePathCollector : CollectorBase
	{
		List<string> m_nameScope = new List<string>();
		string m_dependant;

		public StatePathCollector(FilterWizard wizard, string dependant)
			: base(wizard)
		{
			m_dependant = dependant;
		}

		private string GetName()
		{
			string name = "";
			bool first = true;

			foreach (var s in m_nameScope)
			{
				if (first) first = false;
				else name += '.';

				name += s;
			}

			return name;
		}

		public override bool VisitElement(GameElement elem) 
		{
			return m_wizard.CurrentType != EAffectedType.Scope 
				&& (string.IsNullOrEmpty(m_dependant) || m_dependant == elem.Name); 
		}

		public override bool VisitState(GameState state) 
		{
			if (m_wizard.CurrentType == EAffectedType.Element && state.Node.Locator.IsScope())
				return false;

			if (m_dependant == null || m_dependant == state.Node.Name)
			{
				m_nameScope.Add(state.Desc.Name);
				string name = GetName();
				m_dict[name] = name;
			}
			return true; 
		}

		public override void LeaveState(GameState state) 
		{
			if (m_nameScope.Count != 0)
				m_nameScope.RemoveAt(m_nameScope.Count - 1);
		}

		public override bool StartEventGroups(GameNode node) { return false; }
	}

	///////////////////////////////////////////////////////////////////
	///////////////////////////////////////////////////////////////////


	class StateValueCollector : CollectorBase
	{
		string[] m_path;

		public StateValueCollector(FilterWizard wizard, string path)
			: base(wizard)
		{
			m_path = path.Split(new char[] { '.' });
		}

		private bool CheckName(GameState state)
		{
			int i;
			for (i = m_path.Length - 1; i >= 0 && state != null; --i)
			{
				if (state.Desc.Name != m_path[i])
					return false;

				state = state.Parent;
			}
			return i == -1 && state == null;
		}

		public override bool VisitElement(GameElement elem) { return m_wizard.CurrentType != EAffectedType.Scope; }

		public override bool VisitState(GameState state)
		{
			if (!string.IsNullOrEmpty(state.Value) && CheckName(state))
				m_dict[state.Value] = state.Value;
			return true;
		}

		public override bool StartEventGroups(GameNode node) { return false; }
	}


	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////


	class EventParamCollector : CollectorBase
	{
		string m_group;

		public EventParamCollector(FilterWizard wizard, string group)
			: base(wizard)
		{
			m_group = group;
		}

		public override bool VisitState(GameState state) { return false; }

		public override bool VisitEventGroup(GameEventGroup evntGroup)  
		{ 
			return string.IsNullOrEmpty(m_group) || m_group == evntGroup.Desc.Name; 
		}

		public override void VisitEvent(GameEvent evnt) 
		{
			foreach (var p in evnt.Parameters)
				m_dict[p.Key] = p.Key;
		}
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////


	class EventValueCollector : CollectorBase
	{
		string m_param;

		public EventValueCollector(FilterWizard wizard, string param)
			: base(wizard)
		{
			m_param = param;
		}

		public override bool VisitState(GameState state) { return false; }

		public override void VisitEvent(GameEvent evnt)
		{
			foreach (var p in evnt.Parameters)
				if(p.Key == m_param)
					m_dict[p.Value] = p.Value;
		}
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
}
