﻿using System;
using System.Collections.Generic;
using System.Text;
using StatsDataSource.Storage;
using StatsDataSource.ObjectModel;
using System.Xml.Serialization;
using StatsDataSource.Filters;

namespace StatsTool2.Filters
{
	[Serializable]
	public class NodeFilter : TerminalFilter
	{
		[XmlAttribute("type")]
		public EAffectedType m_type;

		[XmlAttribute("operation")]
		public ECompareOperation m_op;

		[XmlAttribute("stat_name")]
		public string m_name;

		[XmlArray("path")]
		public string[] m_path;

		[XmlAttribute("path_original")]
		public string m_pathOrig;

		[XmlAttribute("value")]
		public string m_value;

		[XmlAttribute("match_name")]
		public bool m_matchName;

		[XmlAttribute("missing_fails")]
		public bool m_missingFails;

		public NodeFilter()
		{
			Enabled = true;
		}

		public void Init(EAffectedType type,
											ECompareOperation op,
											string name,
											bool match_name,
											string path,
											bool missingFails,
											string value)
		{
			m_type = type;
			m_op = op;
			m_name = name;
			m_pathOrig = path ?? "";
			m_path = m_pathOrig.Split(new char[] { '.' });
			m_value = value;
			m_matchName = match_name;
			m_missingFails = missingFails;
		}

		public override object Clone()
		{
			var ret = new NodeFilter();
			ret.Init(m_type, m_op, m_name, m_matchName, m_pathOrig, m_missingFails, m_value);
			ret.Name = Name;
			ret.Enabled = Enabled;
			return ret;
		}

		public override EFilterState TestScope(GameScope scope)
		{
			if (!Enabled) return EFilterState.Unaffected;
			bool nameEmpty = string.IsNullOrEmpty(m_name);

			if (m_type == EAffectedType.Scope || m_type == EAffectedType.Node)
			{
				if ((nameEmpty || scope.Desc.Name == m_name) && !string.IsNullOrEmpty(m_pathOrig))
					return TestAllStates(FindStates(scope));

				if (!nameEmpty && m_matchName)
					return (scope.Desc.Name == m_name) ? EFilterState.Passed : EFilterState.Failed;
			}
			return EFilterState.Unaffected;
		}

		public override EFilterState TestElement(GameElement element)
		{
			if (!Enabled) return EFilterState.Unaffected;
			bool nameEmpty = string.IsNullOrEmpty(m_name);

			if (m_type == EAffectedType.Element || m_type == EAffectedType.Node)
			{
				if ((nameEmpty || element.Desc.Name == m_name) && !string.IsNullOrEmpty(m_pathOrig))
					return TestAllStates(FindStates(element));

				if (!nameEmpty && m_matchName)
					return (element.Desc.Name == m_name) ? EFilterState.Passed : EFilterState.Failed;
			}
			return EFilterState.Unaffected;
		}

		private List<GameState> FindStates(GameNode node)
		{
			var states = new List<GameState>();

			foreach (var s in node.States)
				FindStatesRec(states, s.Value, 0);

			return states;
		}

		private void FindStatesRec(List<GameState> states, GameState curState, int pathDepth)
		{
			if (pathDepth == m_path.Length - 1)
			{
				if (curState.Desc.Name == m_path[pathDepth]) states.Add(curState);
				return;
			}

			if (curState.Desc.Name == m_path[pathDepth])
			{
				++pathDepth;

				foreach (var c in curState.Children)
					FindStatesRec(states, c, pathDepth);
			}
		}

		private EFilterState TestAllStates(List<GameState> states)
		{
			if (states.Count == 0)
				return m_missingFails ? EFilterState.Failed : EFilterState.Unaffected;

			foreach (var s in states)
				if (CompareTestForValue(s))
					return EFilterState.Passed;

			return EFilterState.Failed;
		}

		private bool CompareTestForValue(GameState state)
		{
			if (m_value == null)
				return true;

			if (string.IsNullOrEmpty(state.Value))
				return false;

			string val = state.Value;
			if (m_op != ECompareOperation.Contains)
				return FilterManager.CompareValuesRelative(val, m_value, m_op);
			else
				throw new NotImplementedException();
		}
	}
}
