﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Threading;
using StatsDataSource.Storage;
using StatsDataSource.ObjectModel;
using System.Xml;
using System.IO;
using StatsTool2.Filters;
using StatsDataSource.Core;
using System.Reflection;
using System.Diagnostics;

namespace StatsTool2
{
	public partial class MainForm : Form, IMainForm
	{
		//////////////////////////////////////////////////////////////////////////
		private List<ToolStripMenuItem> m_pluginMenus = new List<ToolStripMenuItem>();
		private Thread m_mainThread;

		public ScopeView ScopeView { get { return this.cScopeView; } }
		public ElementView ElementView { get { return this.cElementView; } }
		public StateView StateView { get { return this.cStateView; } }
		public EventView EventView { get { return this.cEventView; } }
		public FilterControl FilterControl { get { return this.filterControl1; } }
		//////////////////////////////////////////////////////////////////////////

		public MainForm()
		{
			Program.MainForm = this;
			m_mainThread = Thread.CurrentThread;
			var asm = Assembly.GetExecutingAssembly();
			var build = FileVersionInfo.GetVersionInfo(asm.Location).FileVersion;

			string formatFile = Path.Combine(Program.Location, "Data/StatsFormat.xml");
			string processingFile = Path.Combine(Program.Location, "Data/Processing.txt");

			List<string> errors;
			Program.StatsCore = new StatsCore(build, XmlReader.Create(formatFile), processingFile, null, out errors);
			Program.StatsRepository = Program.StatsCore.Repository;

			if (errors.Count != 0)
			{
				string er = "";
				foreach (var e in errors)
					er += e + '\n';

				MessageBox.Show(this, er, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
			}

			Program.FilterManager = new FilterManager(Program.StatsRepository);
			Application.Idle += Application_Idle;

			InitializeComponent();
		}

		void Application_Idle(object sender, EventArgs e)
		{
			if (Program.StatsCore != null)
				Program.StatsCore.OnIdle();
		}

		private void MainForm_Load(object sender, EventArgs e)
		{
			InitPlugins();
		}

		private void MainForm_Shown(object sender, EventArgs e)
		{
			ParseCmdArgs();
		}

		private void MainFormFormClosed(object sender, FormClosedEventArgs e)
		{
			Program.StatsCore.Stop();
		}

		//////////////////////////////////////////////////////////////////////////

		#region External

		private void cScopeView_Enter(object sender, EventArgs e) 
		{ 
			var l = new List<GameNode>();
			foreach (var s in Program.StatsCore.CurrentScopes)
				l.Add(s);
			Program.StatsCore.CurrentNodes = l;
		}

		private void cElementView_Enter(object sender, EventArgs e) 
		{
			var l = new List<GameNode>();
			foreach (var el in Program.StatsCore.CurrentElements)
				l.Add(el);
			Program.StatsCore.CurrentNodes = l;
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Menu

		private void openSessionToolStripMenuItem_Click(object sender, EventArgs e)
		{
			if (openFileDialog.ShowDialog(this) == DialogResult.OK)
			{
				Program.StatsRepository.StartModificationBatch(false);

				foreach (var f in openFileDialog.FileNames)
				{
					if (!File.Exists(f))
						continue;

					using (XmlReader r = XmlReader.Create(f))
					{
						Program.StatsRepository.LoadFromXML(r);
					}
				}

				Program.StatsRepository.EndModificationBatch();
			}
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Plugins

		private void InitPlugins()
		{
			Program.StatsCore.LoadPlugins();

			Program.StatsCore.Start(this, Program.CmdArgs);

			foreach (var p in Program.StatsCore.Plugins)
				foreach (var a in p.getActions())
					AddAction(a);
		}

		private ToolStripItem FindItem(ToolStripItemCollection col, string name)
		{
			for(int i = 0; i != col.Count; ++i)
			{
				if (col[i].Text == name)
					return col[i];
			}
			return null;
		}

		private void AddAction(IPluginAction action)
		{
			string[] path = action.getName().Split(new char[] { '/' } );

			ToolStripMenuItem it = null;

			foreach(var el in path)
			{
				ToolStripItem found;

				if (it == null)
					found = FindItem(menuStrip1.Items, el);
				else
					found = FindItem(it.DropDownItems, el);

				if (found == null)
				{
					var nIt = new ToolStripMenuItem(el);

					if (it == null) menuStrip1.Items.Add(nIt);
					else it.DropDownItems.Add(nIt);

					it = nIt;
				}
				else
					it = (ToolStripMenuItem)found;
			}

			it.Click += PluginAction_Click;
			it.Tag = action;
			m_pluginMenus.Add(it);
		}

		private void menuStrip1_ItemClicked(object sender, ToolStripItemClickedEventArgs e)
		{
			foreach (var it in m_pluginMenus)
			{
				var action = it.Tag as IPluginAction;
				if (action == null) 
					continue;

				it.Enabled = action.CanExecute();
				it.Checked = action.IsEnabled();
			}
		}

		void PluginAction_Click(object sender, EventArgs e)
		{
			var action = ((ToolStripMenuItem)sender).Tag as IPluginAction;
			
			if(action.CanExecute())
				action.Execute();
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Cmd args
		void ParseCmdArgs()
		{
			if (Program.CmdArgs == null) 
				return;

			List<string> to_open = new List<string>();

			for (int i = 0; i < Program.CmdArgs.Length; ++i)
			{
				string cmd = Program.CmdArgs[i];
				if(cmd == "--open" && i != Program.CmdArgs.Length - 1)
				{
					to_open.Add(Program.CmdArgs[i + 1]);
					++i;
				}
			}

			foreach (var f in to_open)
			{
				if (File.Exists(f))
					using (XmlReader r = XmlReader.Create(f))
						Program.StatsRepository.LoadFromXML(r);
			}
		}
		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region IMainForm

		public void SetStatus(string status)
		{
			if (!Thread.CurrentThread.Equals(m_mainThread))
			{
				this.BeginInvoke(
					new MethodInvoker(
						delegate() { SetStatus(status); }
						));
				return;
			}

			lStatus.Text = status;
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////
	}
}
