﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using StatsDataSource.Storage;
using StatsDataSource.ObjectModel;
using StatsTool2.Filters;
using StatsDataSource.Filters;

namespace StatsTool2
{
	public partial class ScopeView : UserControl, IFilteredStatsListener
	{
		//////////////////////////////////////////////////////////////////////////
		private Dictionary<Locator, TreeNode> m_loc2node = new Dictionary<Locator, TreeNode>();
		//////////////////////////////////////////////////////////////////////////

		public ScopeView()
		{
			InitializeComponent();
		}

		private void ScopeView_Load(object sender, EventArgs e)
		{
			if (Program.StatsRepository != null)
			{
				Program.StatsRepository.RegisterFilteredListener(this);
			}
		}

		//////////////////////////////////////////////////////////////////////////

		#region Service

		private string GetScopeName(GameScope scope)
		{
			StringBuilder bld = new StringBuilder(30 + scope.States.Count * 20);
			bld.Append(scope.Name);

			bool first = true;
			foreach (var pk in scope.Desc.KeyStates)
			{
				StatDesc desc = Program.StatsRepository.Registry.GetStateDesc(pk);
				if (desc == null)
					continue;

				GameState s;
				if (scope.States.TryGetValue(desc.ID, out s))
				{
					if (!string.IsNullOrEmpty(s.Value))
					{
						bld.Append(first ? " [" : ", ");
						first = false;

						bld.Append(s.Desc.Name);
						bld.Append(": ");
						bld.Append(s.Value);
					}
				}
			}

			if (!first)
				bld.Append("]");

			return bld.ToString();
		}

		private TreeNode CreateScopeNode(GameNode node)
		{
			var tn = new TreeNode(GetScopeName((GameScope)node));
			tn.Tag = node;
			m_loc2node.Add(node.Locator, tn);
			return tn;
		}

		private void RemoveScope(TreeNode node)
		{
			twScopes.BeginUpdate();
			RemoveScopeRec(node);
			twScopes.EndUpdate();
		}

		private void RemoveScopeRec(TreeNode node)
		{
			var s = (GameScope)node.Tag;
			m_loc2node.Remove(s.Locator);

			for (int i = 0; i != node.Nodes.Count; ++i)
				RemoveScopeRec(node.Nodes[i]);

			node.Remove();
		}

		private void ClearTree()
		{
			twScopes.Nodes.Clear();
			m_loc2node.Clear();
		}

		private void UpdateView()
		{
			twScopes.BeginUpdate();
			ClearTree();
			Program.StatsRepository.AcceptVisitorFiltered(new ScopeUpdateVisitor(this));
			twScopes.EndUpdate();
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region IStatsListener

		private bool inModificationBatch = false;

		public void ModificationBatchStarted()
		{
			inModificationBatch = true;
		}

		public void ModificationBatchEnded(IncrementalUpdate update)
		{
			inModificationBatch = false;
			if (update == null)
			{
				UpdateView();
			}
			else
			{
				foreach (var n in update.AddedNodes)
					RepositoryNodeAdded(n);

				foreach (var s in update.AddedStates)
					if (!update.AddedNodes.Contains(s.Node))
						RepositoryStateAdded(s);
			}
		}

		public void RepositoryNodeAdded(GameNode node)
		{
			if (inModificationBatch)
				return;

			if (node.FilterState == EFilterState.Failed)
				return;

			if (node.Locator.IsScope())
			{
				if (node.Parent != null)
				{
					TreeNode parent;
					if(m_loc2node.TryGetValue(node.Parent.Locator, out parent))
					{
						parent.Nodes.Add(CreateScopeNode(node));
						parent.Expand();
					}
				}
				else
				{
					twScopes.Nodes.Add(CreateScopeNode(node));
				}
			}
		}

		public void RepositoryStateAdded(GameState state)
		{
			if (inModificationBatch)
				return;

			TreeNode n;
			if (state.Node.Locator.IsScope())
			{
				EFilterState testScope = state.Node.FilterState;

				if (m_loc2node.TryGetValue(state.Node.Locator, out n))
				{
					if (testScope == EFilterState.Failed) RemoveScope(n);
					else n.Text = GetScopeName((GameScope)n.Tag);
				}
				else if(testScope != EFilterState.Failed)
				{
					RepositoryNodeAdded(state.Node);
				}
			}
		}

		public void RepositoryEventGroupAdded(GameEventGroup group)
		{
		}

		public void RepositoryEventAdded(GameEvent evnt)
		{
		}

		public void RepositoryFilterChanged(CompoundFilter active, EFilterUpdateType type)
		{
			if (type != EFilterUpdateType.Fast_Event && type != EFilterUpdateType.Regular_Event)
				UpdateView();
		}
		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region Tree

		private void twScopes_AfterSelect(object sender, TreeViewEventArgs e)
		{
			List<GameScope> newScopes = new List<GameScope>();
			
			if(e.Node != null)
				newScopes.Add((GameScope)e.Node.Tag);

			Program.StatsCore.CurrentScopes = newScopes;
		}

		private void tvScopes_ItemDrag(object sender, ItemDragEventArgs e)
		{
			var item = e.Item as TreeNode;
			if(item != null)
				DoDragDrop(item.Tag, DragDropEffects.Copy);
		}

		#endregion

		//////////////////////////////////////////////////////////////////////////
	}


	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	class ScopeUpdateVisitor : IStatsVisitor
	{
		private ScopeView m_view;

		public ScopeUpdateVisitor(ScopeView view)
		{
			m_view = view;
		}

		public bool VisitScope(GameScope scope)
		{
			m_view.RepositoryNodeAdded(scope);
			return true;
		}
		public void LeaveScope(GameScope scope) { }

		public bool VisitElement(GameElement elem) { return false; }
		public void LeaveElement(GameElement elem) { }
		public bool VisitState(GameState state) { return false; }
		public void LeaveState(GameState state) { }
		public bool StartEventGroups(GameNode node) { return false; }
		public bool VisitEventGroup(GameEventGroup evntGroup) { return false; }
		public void LeaveEventGroup(GameEventGroup evntGroup) { }
		public void EndEventGroups(GameNode node) { }
		public void VisitEvent(GameEvent evnt) { }
	}
}
