﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using StatsDataSource.Storage;
using StatsDataSource.ObjectModel;
using StatsDataSource.Core;
using StatsTool2.Filters;
using StatsDataSource.Filters;

namespace StatsTool2
{
	public partial class StateView : UserControl, IFilteredStatsListener
	{
		//////////////////////////////////////////////////////////////////////////
		private Dictionary<int, TreeNode> m_id2node = new Dictionary<int, TreeNode>();
		//////////////////////////////////////////////////////////////////////////

		public StateView()
		{
			InitializeComponent();
		}

		private void StateView_Load(object sender, EventArgs e)
		{
			if (Program.StatsRepository != null)
			{
				Program.StatsRepository.RegisterFilteredListener(this);
				Program.StatsCore.NodesChanged += StatsCore_NodesChanged;
			}
		}

		//////////////////////////////////////////////////////////////////////////

		#region Service
		private string GetStateName(List<GameState> states)
		{
			states.Sort(new StateValueComparer());

			var sb = new StringBuilder(100);
			sb.Append(states[0].Desc.Name);

			bool first = true;
			string prev = null;
			foreach (var s in states)
			{
				if(s.Value == null || s.Value == prev)
					continue;

				if (first)
				{
					sb.Append(" : ");
					first = false;
				}
				else
				{
					sb.Append(", ");
				}

				sb.Append(s.Value);
				prev = s.Value;

				if(sb.Length > 70)
				{
					sb.Append("...");
					break;
				}
			}

			return sb.ToString();
		}

		private TreeNode FindSimilarNode(TreeNodeCollection col, GameState state)
		{
			if (state.Parent == null)
			{
				for (int i = 0; i != col.Count; ++i)
				{
					var sc = (List<GameState>)col[i].Tag;
					if (sc[0].Desc == state.Desc)
						return col[i];
				}
			}
			else if (state.Parent != null)
			{
				int index = state.Parent.Children.FindIndex(x => x == state);
				if (col.Count <= index)
					return null;

				var target = (List<GameState>)col[index].Tag;
				if (target[0].Desc == state.Desc)
					return col[index];
			}
			return null;
		}

		private void AddState(GameState state)
		{
			var col = state.Parent == null ? tvStates.Nodes : m_id2node[state.Parent.UniqueID].Nodes;

			TreeNode n = FindSimilarNode(col, state);

			if (n == null)
			{
				n = new TreeNode("");
				col.Add(n);
				n.Tag = new List<GameState>();
			}

			var states = (List<GameState>)n.Tag;
			states.Add(state);
			m_id2node.Add(state.UniqueID, n);
			if(states.Count < 50)
				n.Text = GetStateName(states);
		}

		private void ClearTree()
		{
			tvStates.Nodes.Clear();
			m_id2node.Clear();
		}

		void StatsCore_NodesChanged(StatsCore sender, List<GameNode> nodes)
		{
			tvStates.BeginUpdate();
			ClearTree();

			if (nodes.Count > 0)
			{
				StateUpdateVisitor v = new StateUpdateVisitor(this);
				Program.StatsRepository.AcceptVisitorFiltered(v);
			}

			tvStates.EndUpdate();
		}

		private void tvStates_ItemDrag(object sender, ItemDragEventArgs e)
		{
			var item = e.Item as TreeNode;
			if(item != null)
				DoDragDrop(((List<GameState>)item.Tag)[0], DragDropEffects.Copy);
		}
		#endregion

		//////////////////////////////////////////////////////////////////////////

		#region IStatsListener

		private bool inModificationBatch = false;

		public void ModificationBatchStarted()
		{
			inModificationBatch = true;
		}

		public void ModificationBatchEnded(IncrementalUpdate update)
		{
			inModificationBatch = false;

			if (update == null)
			{
				StatsCore_NodesChanged(Program.StatsCore, Program.StatsCore.CurrentNodes);
			}
			else
			{
				foreach (var s in update.AddedStates)
					RepositoryStateAdded(s);
			}
		}

		public void RepositoryNodeAdded(GameNode node)
		{
		}

		public void RepositoryStateAdded(GameState state)
		{
			if (inModificationBatch)
				return;

			if (state.FilterState == EFilterState.Failed || !Program.StatsCore.CurrentNodes.Contains(state.Node))
				return;

			AddState(state);
		}

		public void RepositoryEventGroupAdded(GameEventGroup group)
		{
		}

		public void RepositoryEventAdded(GameEvent evnt)
		{
		}

		public void RepositoryFilterChanged(CompoundFilter active, EFilterUpdateType type)
		{
			if (type != EFilterUpdateType.Fast_Event && type != EFilterUpdateType.Regular_Event)
				StatsCore_NodesChanged(Program.StatsCore, Program.StatsCore.CurrentNodes);
		}
		#endregion

		//////////////////////////////////////////////////////////////////////////
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	class StateUpdateVisitor : IStatsVisitor
	{
		private StateView m_view;

		public StateUpdateVisitor(StateView view)
		{
			m_view = view;
		}

		public bool VisitScope(GameScope scope) { return true; }
		public void LeaveScope(GameScope scope) { }

		public bool VisitElement(GameElement elem) { return true; }
		public void LeaveElement(GameElement elem) { }

		public bool VisitState(GameState state) 
		{
			m_view.RepositoryStateAdded(state);
			return true; 
		}
		public void LeaveState(GameState state) { }

		public bool StartEventGroups(GameNode node) { return false; }
		public bool VisitEventGroup(GameEventGroup evntGroup) { return false; }
		public void LeaveEventGroup(GameEventGroup evntGroup) { }
		public void EndEventGroups(GameNode node) { }

		public void VisitEvent(GameEvent evnt) { }
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	class StateValueComparer : IComparer<GameState>
	{
		public int Compare(GameState x, GameState y)
		{
			return string.Compare(x.Value, y.Value);
		}
	}
}
