#include "stdafx.h"

#include <string>

using std::wstring;

#include <CryUnitInterfaces.h>
#include <TestSuite.h>
#include <TestInfo.h>

#include <list>
#include <vector>
#include <string>

#include <vcclr.h>

#include "CryUnitWrapper.h"
#include "TestListener.h"
#include "TestRunInfo.h"
#include "TestSuite.h"

using namespace System::Collections;
using namespace System::IO;
using namespace System::Collections::Generic;

namespace CryUnitWrapper 
{
	class CTestListenerWrapper;

	ref class TestListener;

	public ref class TestRunner
	{
	public:		
		TestRunner(String^ basePath, String^ dllName, bool enableSmokeTest)
			: m_hLibrary(NULL)
			, m_hSystemLibrary(NULL)
			, m_enableSmokeTest(enableSmokeTest)
		{
			m_hLibrary = LoadDll(dllName, basePath);
			EnumerateTestSuites();
		}

		~TestRunner()
		{
			if (m_hSystemLibrary)
			{
				::FreeLibrary(m_hSystemLibrary);
			}

			if (m_hLibrary)
			{
				::FreeLibrary(m_hLibrary);
			}
		}

		void RunTests(TestListener^ listener)
		{			
			if (m_TestEnumerator == nullptr)
				return;

			try
			{
				m_TestEnumerator->Initialise();

				int count = m_TestSuites->Count;
				for (int i = 0; i < m_TestSuites->Count; ++i)
				{
					TestSuite^ testSuite = (TestSuite^) m_TestSuites[i];

					System::Console::Write("{0} [{1}]: ", testSuite->Name, testSuite->Type);

					int oldTestRunCount = listener->TestsRun->Count;
					int oldTestPassedCount = listener->TestsPassed->Count;
					int oldTestFailedCount = listener->TestsFailed->Count;

					testSuite->Run(listener);

					int numberOfTestsRun = listener->TestsRun->Count - oldTestRunCount;
					int numberOfTestsPassed = listener->TestsPassed->Count - oldTestPassedCount;
					int numberOfTestsFailed = listener->TestsFailed->Count - oldTestFailedCount;

					System::Console::WriteLine("RUN [{0}:{1}:{2}]", numberOfTestsRun, numberOfTestsPassed, numberOfTestsFailed);
				}
			}
			finally
			{
				m_TestEnumerator->Shutdown();
			}
		}

	private:
		HMODULE LoadDll(String^ dllName, String^ basePath)
		{
			HMODULE library = NULL;

			String^ currentDir = System::Environment::CurrentDirectory;

			pin_ptr<const wchar_t> dllNameStr = PtrToStringChars(dllName);
			pin_ptr<const wchar_t> basePathStr = PtrToStringChars(basePath);			
			pin_ptr<const wchar_t> currentPathStr = PtrToStringChars(currentDir);

			std::wstring dllPath = 
				std::wstring(basePathStr) + std::wstring(L"\\") +
				std::wstring(dllNameStr) + 
				std::wstring(L".dll");

			String^ filePath = System::IO::Path::GetFullPath(gcnew String(dllPath.c_str()));
			String^ fileDir = System::IO::Path::GetDirectoryName(filePath);

			Console::WriteLine("Loading {0}", filePath);

			System::IO::Directory::SetCurrentDirectory(fileDir);
			if (File::Exists(filePath))
			{
				pin_ptr<const wchar_t> dllPathStr = PtrToStringChars(filePath);
				library = ::LoadLibrary(dllPathStr);
				if (library == 0)
				{
					throw gcnew System::SystemException(GetWindowsLastError() + " - " + gcnew String(dllPath.c_str()));
				}
			}

			System::IO::Directory::SetCurrentDirectory(currentDir);

			return library;
		}

		CryUnit::ITestSuite::Type GetSuiteType()
		{
			CryUnit::ITestSuite::Type res = CryUnit::ITestSuite::UNIT_TEST;
			if (m_enableSmokeTest)
				res = CryUnit::ITestSuite::SMOKE_TEST;
			return res;
		}

		void EnumerateTestSuites()
		{	
			CryUnit::GetTestEnumerator getTestEnumerator = (CryUnit::GetTestEnumerator) GetProcAddress(m_hLibrary, "GetTestEnumerator");
			if (getTestEnumerator == 0)
			{
				throw gcnew System::SystemException("Can't find GetTestEnumerator().");
			}

			CryUnit::ITestEnumerator* testEnumrator = getTestEnumerator(GetSuiteType());
			if (testEnumrator == 0)
			{
				throw gcnew System::SystemException("Can't find test enumerator.");
			}

			m_TestEnumerator = testEnumrator;

			int numberOfTestSuites = m_TestEnumerator->NumberOfTestSuites();

			std::vector<CryUnit::ITestSuite*> testSuites;
			testSuites.resize(numberOfTestSuites);			// Fran: very naughty thing to do

			m_TestEnumerator->EnumerateTestSuites(&testSuites[0]);

			m_TestSuites = gcnew ArrayList();

			for (int i = 0; i < numberOfTestSuites; ++i)
			{				
				TestSuite^ testSuite = gcnew TestSuite(testSuites[i]);

				m_TestSuites->Add(testSuite);
			}
		}

		String^ GetWindowsLastError()
		{
			DWORD lastError = ::GetLastError();
			LPTSTR buff;
			::FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM, NULL, lastError, 0, (LPTSTR)&buff, 0, NULL);
			String^ res = gcnew String(buff);
			::LocalFree(buff);
			return res;
		}

	private:
		CryUnit::ITestEnumerator* m_TestEnumerator;

		HMODULE m_hLibrary;
		HMODULE m_hSystemLibrary;

		ArrayList^ m_TestSuites;
		bool m_enableSmokeTest;
	};
}
