#include "StdAfx.h"
#include "Config.h"

#include <cctype>
#include <algorithm>

Config::OptionFlag::OptionFlag(const string& name)
	: m_name(LowerCase(name))
	, m_enabled(false)
{

}

Config::OptionFlag::OptionFlag(const string& name, const string& defaultValue)
	: m_name(LowerCase(name))
	, m_value(defaultValue)
	, m_enabled(false)
{

}


bool Config::OptionFlag::IsEnabled() const
{
	return m_enabled;
}

const string& Config::OptionFlag::GetValue() const
{
	return m_value;
}

bool Config::OptionFlag::Match(const string& originalArg)
{
	string downcaseArg = LowerCase(originalArg);

	bool matches = false;
	if (downcaseArg == m_name)
	{
		matches = true;
		m_enabled = true;
	}
	else
	{
		TryMatchValue(downcaseArg, originalArg, matches);
	}
	return matches;
}

void Config::OptionFlag::TryMatchValue(const string& downcaseArg, const string& originalArg, bool& matches) 
{
	if (downcaseArg.find_first_not_of(m_name) == m_name.size())
	{
		string::size_type first = downcaseArg.find(":");
		if (first != string::npos)
		{
			m_value = string(originalArg, first + 1, downcaseArg.size() - first);
			m_enabled = true;
			matches = true;
		}
	}
}

string Config::OptionFlag::LowerCase(const string& toDowncase) const
{
	string downcased = toDowncase;
	std::transform(toDowncase.begin(), toDowncase.end(), downcased.begin(), std::tolower);
	return downcased;
}

Config::Config(int argc, char** argv)
	: m_reportFlag("report", "Xml-Reports")
	, m_smokeFlag("smoke")
	, m_forceFlag("force")
	, m_dllName("CryGameCrysis2")
{
	m_optionFlags.push_back(&m_reportFlag);
	m_optionFlags.push_back(&m_smokeFlag);
	m_optionFlags.push_back(&m_forceFlag);

	if (argc >= 2)
	{
		ParseCommandLine(argc, argv);
	}
}

void Config::ParseCommandLine(int argc, char** argv)
{
	int argIndex = 1;
	bool found = false;
	do
	{
		found = MatchOption(argv[argIndex]);
		if (found)
		{
			++argIndex;
		}
	}
	while (found);

	if (argIndex < argc)
		m_dllName = argv[argIndex];
}

bool Config::MatchOption(const string& arg)
{
	for (size_t index = 0; index < m_optionFlags.size(); ++index)
	{
		OptionFlag* optionFlag = m_optionFlags[index];
		if (optionFlag->Match(arg))
			return true;
	}
	return false;
}

bool Config::IsSmokeTest() const
{
	return m_smokeFlag.IsEnabled();
}

bool Config::IsReportEnabled() const
{
	return m_reportFlag.IsEnabled();
}

const string& Config::GetReportDirectory() const 
{
	return m_reportFlag.GetValue();
}

bool Config::IsForceEnabled() const
{
	return m_forceFlag.IsEnabled();
}

const string Config::GetDllName() const
{
	return m_dllName + ".dll";
}
