#include "StdAfx.h"
#include "TestRunnerObserver.h"
#include "TestSuiteLoader.h"
#include "XmlTestWriter.h"

//////////////////////////////////////////////////////////////////////////

TestRunInfo::TestRunInfo()
	: FileLineNumber(0)
{

}

TestRunInfo::TestRunInfo(const CryUnit::STestInfo& testInfo)
	: Name(testInfo.Name)
	, FileName(testInfo.FileName)
	, FileLineNumber(testInfo.FileLineNumber)
{

}

//////////////////////////////////////////////////////////////////////////

CallStackElement::CallStackElement(const CryUnit::ICallStackElement& callStackElement)
	: FunctionName(callStackElement.GetFunctionName())
	, ModuleName(callStackElement.GetModuleName())
	, FileName(callStackElement.GetFileName())
	, FileLineNumber(callStackElement.GetFileLineNumber())
{

}

//////////////////////////////////////////////////////////////////////////

TestFailedInfo::TestFailedInfo()
	: FileLineNumber(0)
{

}

TestFailedInfo::TestFailedInfo(const CryUnit::SFailureInfo& failureInfo)
	: Condition(failureInfo.Condition)
	, Message(failureInfo.Message)
	, FileName(failureInfo.FileName)
	, FileLineNumber(failureInfo.FileLineNumber)
{
	for (int i = 0; i < failureInfo.CallStack.GetSize(); ++i)
	{
		CallStack.push_back(failureInfo.CallStack.GetElementByIndex(i));
	}
}

//////////////////////////////////////////////////////////////////////////

TestSuiteContext::TestSuiteContext()
	: m_startTime(std::clock())
{

}

TestSuiteContext::TestSuiteContext(const string& suiteName)
	: m_suiteName(suiteName)
	, m_startTime(std::clock())
{
	
}

const string& TestSuiteContext::GetSuiteName() const
{
	return m_suiteName;
}

void TestSuiteContext::AddTestPassed(const TestRunInfo& testRunInfo)
{
	m_testPassedCollection.push_back(testRunInfo);
}

size_t TestSuiteContext::GetNumberOfTestPassed() const
{
	return m_testPassedCollection.size();
}
	
const TestRunInfo& TestSuiteContext::GetTestPassedAt(size_t index) const
{
	return m_testPassedCollection.at(index);
}

void TestSuiteContext::AddTestFailed(const TestRunInfo& testRunInfo, const TestFailedInfo& testFailedInfo)
{
	m_testFailedCollection.push_back(std::make_pair(testRunInfo, testFailedInfo));
}

size_t TestSuiteContext::GetNumberOfTestFailed() const
{
	return m_testFailedCollection.size();
}

const std::pair<TestRunInfo, TestFailedInfo>& TestSuiteContext::GetTestFailedAt(size_t index) const
{
	return m_testFailedCollection.at(index);
}

void TestSuiteContext::AddTestSkipped(const TestRunInfo& testRunInfo)
{
	m_testSkippedCollection.push_back(testRunInfo);
}

size_t TestSuiteContext::GetNumberOfTestSkipped() const
{
	return m_testSkippedCollection.size();
}

const TestRunInfo& TestSuiteContext::GetTestSkippedAt(size_t index) const
{
	return m_testSkippedCollection.at(index);
}

float TestSuiteContext::ComputeElapsedSeconds() const
{
	return float(std::clock() - m_startTime)/CLOCKS_PER_SEC;
}

//////////////////////////////////////////////////////////////////////////

TestRunnerObserver::TestRunnerObserver(TestWriter& testWriter)
	: m_testWriter(testWriter)
	, m_exitCode(0)
	, m_testRunCount(0)
	, m_testPassedCount(0)
	, m_startTime(std::clock())
{

}

int TestRunnerObserver::GetExitCode()
{
	return m_exitCode;
}

void TestRunnerObserver::TestSuiteStart(const string& suiteName)
{
	m_suiteContext = TestSuiteContext(suiteName);
}

void TestRunnerObserver::TestSuiteEnd(const string& suiteName)
{
	size_t testFailed = m_suiteContext.GetNumberOfTestFailed();
	size_t testPassed = m_suiteContext.GetNumberOfTestPassed();
	printf("%s [%lu:%lu:%lu] [%f]\n", m_suiteContext.GetSuiteName().c_str(), testPassed + testFailed, testPassed, testFailed, m_suiteContext.ComputeElapsedSeconds());
	m_testWriter.TestSuiteEnd(m_suiteContext);
}

void TestRunnerObserver::StartTestRun(const TestRunInfo& testInfo)
{
	m_runningTest = testInfo;
}

void TestRunnerObserver::TestRun(const TestRunInfo& testInfo)
{
	++m_testRunCount;
}

void TestRunnerObserver::TestPassed(const TestRunInfo& testInfo)
{
	++m_testPassedCount;
	m_suiteContext.AddTestPassed(testInfo);
}

void TestRunnerObserver::TestFailed(const TestRunInfo& testInfo, const TestFailedInfo& testFailedInfo)
{
	m_exitCode = 1;
	m_suiteContext.AddTestFailed(testInfo, testFailedInfo);
	m_testFailedCollection.push_back(std::make_pair(testInfo, testFailedInfo));
}

void TestRunnerObserver::TestSkipped(const TestRunInfo& testInfo)
{
	m_suiteContext.AddTestSkipped(testInfo);
	m_testSkippedCollection.push_back(testInfo);
}

void TestRunnerObserver::InitDll(const string& dllName, const string& binariesDirectory)
{
	printf("==================== Running tests ============================ \n");
	printf("Module: %s\n", dllName.c_str());
	printf("Binaries folder: %s\n", binariesDirectory.c_str());
	printf("=============================================================== \n");
	printf("Test Suites:\n");
}

void TestRunnerObserver::InitDllFailed(const TestSuiteLoaderException& exception)
{
	m_exitCode = 1;
	printf("TestSuiteLoaderException What [%s] DLL [%s] BinPath [%s]\n", exception.what(), exception.GetDllName().c_str(), exception.GetBinariesDirectory().c_str());
}

void TestRunnerObserver::EndRunTest()
{
	printf("=============================================================== \n");
	printf("Test Results:\n");
	printf("Run: %lu\n", m_testRunCount);
	printf("Passed: %lu\n", m_testPassedCount);
	printf("Failed: %lu\n", m_testFailedCollection.size());
	printf("ElapsedTime: %f\n", float(std::clock() - m_startTime)/CLOCKS_PER_SEC);
	printf("=============================================================== \n");
	LogSkippedTest();
	LogFailedTest();
}

void TestRunnerObserver::VisualStudioOutputInfo(const string& fileName, int fileLineNumber, const string& message)
{
	printf("%s(%lu) : %s\n", fileName.c_str(), fileLineNumber, message.c_str());
}

void TestRunnerObserver::LogFailedTest()
{
	if (m_testFailedCollection.empty())
		return;

	printf("\n");
	printf("\n\n\n\nList of failing tests:\n");

	for (TestFailureInfoCollection::const_iterator it = m_testFailedCollection.begin(); it != m_testFailedCollection.end(); ++it)
	{
		const TestRunInfo& testRunInfo = it->first;
		const TestFailedInfo& testFailedInfo = it->second;
		VisualStudioOutputInfo(testFailedInfo.FileName, testFailedInfo.FileLineNumber, string("Test Failed ") + testRunInfo.Name);
		string msg = testFailedInfo.Message;
		if (!testFailedInfo.Condition.empty())
		{
			msg += ": " + testFailedInfo.Condition;
		}

		VisualStudioOutputInfo(testFailedInfo.FileName, testFailedInfo.FileLineNumber, msg);
		LogCallStack(testFailedInfo);
		printf("\n");
	}

	printf("====================\n");
}

void TestRunnerObserver::LogSkippedTest()
{
	if (m_testSkippedCollection.empty())
		return;

	printf("Skipped: %lu\n", m_testSkippedCollection.size());
	for (TestRunInfoCollection::const_iterator it = m_testSkippedCollection.begin(); it != m_testSkippedCollection.end(); ++it)
	{
		VisualStudioOutputInfo(it->FileName, it->FileLineNumber, it->Name);
	}
	printf("=============================================================== \n");
}

void TestRunnerObserver::LogCallStack(const TestFailedInfo& testFailedInfo)
{
	if (!testFailedInfo.CallStack.empty())
	{
		printf("StackTrace:\n");
		for (size_t i = 0; i < testFailedInfo.CallStack.size(); i++)
		{
			const CallStackElement& callStackElement = testFailedInfo.CallStack.at(i);
			VisualStudioOutputInfo(callStackElement.FileName, callStackElement.FileLineNumber, callStackElement.FunctionName);
		}
	}
}

//////////////////////////////////////////////////////////////////////////
