#pragma once

#ifndef TEST_RUNNER_OBSERVER_H
#define TEST_RUNNER_OBSERVER_H

#include <vector>
#include <utility>
#include <ctime>

//////////////////////////////////////////////////////////////////////////

class TestWriter;
class TestSuiteLoaderException;

//////////////////////////////////////////////////////////////////////////

class TestRunInfo
{
public:
	TestRunInfo();
	TestRunInfo(const CryUnit::STestInfo& testInfo);

	string Name;
	string FileName;
	size_t FileLineNumber;
};

//////////////////////////////////////////////////////////////////////////

class CallStackElement
{
public:
	CallStackElement(const CryUnit::ICallStackElement& callStackElement);

	string FunctionName;
	string ModuleName;
	string FileName;
	int FileLineNumber;
};

//////////////////////////////////////////////////////////////////////////

class TestFailedInfo
{
public:
	TestFailedInfo();
	TestFailedInfo(const CryUnit::SFailureInfo& failureInfo);

	string Condition;
	string Message;
	string FileName;
	size_t FileLineNumber;
	std::vector<CallStackElement> CallStack;
};

//////////////////////////////////////////////////////////////////////////

typedef std::vector<TestRunInfo> TestRunInfoCollection;
typedef std::vector< std::pair<TestRunInfo, TestFailedInfo> > TestFailureInfoCollection;

class TestSuiteContext
{
public:
	TestSuiteContext();
	TestSuiteContext(const string& suiteName);

	const string& GetSuiteName() const;

	void AddTestPassed(const TestRunInfo& testRunInfo);
	size_t GetNumberOfTestPassed() const;
	const TestRunInfo& GetTestPassedAt(size_t index) const;

	void AddTestFailed(const TestRunInfo& testRunInfo, const TestFailedInfo& testFailedInfo);
	size_t GetNumberOfTestFailed() const;
	const std::pair<TestRunInfo, TestFailedInfo>& GetTestFailedAt(size_t index) const;

	void AddTestSkipped(const TestRunInfo& testRunInfo);
	size_t GetNumberOfTestSkipped() const;
	const TestRunInfo& GetTestSkippedAt(size_t index) const;

	float ComputeElapsedSeconds() const;

private:
	string m_suiteName;

	TestRunInfoCollection m_testPassedCollection;
	TestFailureInfoCollection m_testFailedCollection;
	TestRunInfoCollection m_testSkippedCollection;
	size_t m_testRunCount;
	size_t m_testPassedCount;
	clock_t m_startTime;
};

//////////////////////////////////////////////////////////////////////////

class TestRunnerObserver
{

public:
	TestRunnerObserver(TestWriter& testWriter);

	void Reset();

	int GetExitCode();

	void TestSuiteStart(const string& suiteName);
	void TestSuiteEnd(const string& suiteName);
	void StartTestRun(const TestRunInfo& testInfo);
	void TestRun(const TestRunInfo& testInfo);
	void TestPassed(const TestRunInfo& testInfo);
	void TestFailed(const TestRunInfo& testInfo, const TestFailedInfo& testFailedInfo);
	void TestSkipped(const TestRunInfo& testInfo);
	void InitDll(const string& dllName, const string& binariesDirectory);
	void InitDllFailed(const TestSuiteLoaderException& exception);
	void EndRunTest();

private:
	void VisualStudioOutputInfo(const string& fileName, int fileLineNumber, const string& message);
	void LogFailedTest();
	void LogSkippedTest();
	void LogCallStack(const TestFailedInfo& testFailedInfo);
	
	TestWriter& m_testWriter;
	int m_exitCode;

	TestSuiteContext m_suiteContext;
	TestRunInfo m_runningTest;

	TestRunInfoCollection m_testSkippedCollection;
	TestFailureInfoCollection m_testFailedCollection;

	int m_testRunCount;
	int m_testPassedCount;
	clock_t m_startTime;
};

//////////////////////////////////////////////////////////////////////////

#endif
