#include "StdAfx.h"
#include "TestSuiteLoader.h"
#include <CryPath.h>
#include <CryLibrary.h>

TestSuiteLoaderException::TestSuiteLoaderException(const string& what, const string& dllName, const string& binariesDirectory)
	: std::exception(what)
	, m_dllName(dllName)
	, m_binariesDirectory(binariesDirectory)
{

}

const string& TestSuiteLoaderException::GetDllName() const
{
	return m_dllName; 
}

const string& TestSuiteLoaderException::GetBinariesDirectory() const
{
	return m_binariesDirectory;
}

TestSuiteLoader::TestSuiteLoader()
	: m_rootDirectory(GetRootDirectory())
	, m_gameModule(NULL)
	, m_testEnumerator(NULL)
{

}

string TestSuiteLoader::GetRootDirectory() const
{
	char modulePath[256];
	GetModuleFileName(GetModuleHandle(0), modulePath, sizeof(modulePath));
	string rootDirectory = PathUtil::ToUnixPath(modulePath);

	string::size_type rootDirectoryPos = rootDirectory.rfind("/Bin");
	if (rootDirectoryPos != string::npos)
	{
		rootDirectory = rootDirectory.substr(0, rootDirectoryPos);
	}

	return rootDirectory;
}

TestSuiteLoader::~TestSuiteLoader()
{
	if (m_testEnumerator)
	{
		m_testEnumerator->Shutdown();
		m_testEnumerator = NULL;
	}

	if (m_gameModule)
	{
		CryFreeLibrary(m_gameModule);
		m_gameModule = NULL;
	}
}

string TestSuiteLoader::GetBinariesDirectory() const
{
#ifdef WIN64
	const char* binaries = "Bin64";
#else
	const char* binaries = "Bin32";
#endif

	return PathUtil::Make(m_rootDirectory, binaries);
}

void TestSuiteLoader::Init(const string& dllName, CryUnit::ITestSuite::Type suiteType)
{
	string binariesDirectory = GetBinariesDirectory();

	SetCurrentDirectory(binariesDirectory.c_str());
	m_gameModule = CryLoadLibrary(dllName.c_str());

	SetCurrentDirectory(m_rootDirectory.c_str());
	if (m_gameModule)
	{
		InitTestEnumerator(suiteType);
	}

	ReportError(dllName, binariesDirectory);
}

void TestSuiteLoader::InitTestEnumerator(CryUnit::ITestSuite::Type suiteType)
{
	CryUnit::GetTestEnumerator getTestEnumerator = reinterpret_cast<CryUnit::GetTestEnumerator>(CryGetProcAddress(m_gameModule, "GetTestEnumerator"));
	if (getTestEnumerator)
	{
		m_testEnumerator = getTestEnumerator(suiteType);
		if (m_testEnumerator)
		{
			m_testEnumerator->Initialise();

			int numberOfTestSuites = m_testEnumerator->NumberOfTestSuites();
			m_testSuites.resize(numberOfTestSuites);

			m_testEnumerator->EnumerateTestSuites(&m_testSuites[0]);
		}
	}
}

void TestSuiteLoader::ReportError(const string& dllName, const string& binariesDirectory) const
{
	if (m_rootDirectory.empty())
	{
		throw TestSuiteLoaderException("Path not valid", dllName, binariesDirectory);
	}

	if (!m_gameModule)
	{
		throw TestSuiteLoaderException("DLL not loaded", dllName, binariesDirectory);
	}

	if (!m_testEnumerator)
	{
		throw TestSuiteLoaderException("GetTestEnumerator not found in DLL", dllName, binariesDirectory);
	}
}

size_t TestSuiteLoader::GetNumberOfSuites() const
{
	return m_testSuites.size();
}

CryUnit::ITestSuite* TestSuiteLoader::GetSuiteAt(size_t index) const
{
	return m_testSuites[index];
}
