#include "StdAfx.h"
#include "XmlTestWriter.h"
#include "XmlWriter.h"
#include "TestRunnerObserver.h"

#include <CryPath.h>

TestWriter::~TestWriter()
{

}

void TestWriter::TestSuiteEnd(const TestSuiteContext& suiteContext)
{

}

XmlTestWriter::XmlTestWriter(const string& xmlPath)
	: m_xmlPath(xmlPath)
{

}

void XmlTestWriter::TestSuiteEnd(const TestSuiteContext& suiteContext)
{
	XmlWriter writer;

	writer.WriteStartElement("testsuite");
	WriteSuiteAttributes(writer, suiteContext);

	WriteTestCases(writer, suiteContext);

	writer.WriteEndElement();

	string reportFileName = string().Format("TEST-%s.xml", suiteContext.GetSuiteName());
	CCryFile file(PathUtil::Make(m_xmlPath, reportFileName), "w");

	const string xmlString = writer.ToString();	
	file.Write(xmlString.c_str(), xmlString.size());

	file.Flush();
}

void XmlTestWriter::WriteSuiteAttributes(XmlWriter& writer, const TestSuiteContext& suiteContext)
{
	int passedCount = suiteContext.GetNumberOfTestPassed();
	int failedCount = suiteContext.GetNumberOfTestFailed();
	int skippedCount = suiteContext.GetNumberOfTestSkipped();

	writer.WriteAttribute("errors", 0);
	writer.WriteAttribute("skipped", skippedCount);
	writer.WriteAttribute("tests", passedCount + failedCount + skippedCount);
	writer.WriteAttribute("time", 0);
	writer.WriteAttribute("failures", failedCount);
	writer.WriteAttribute("name", suiteContext.GetSuiteName());
}

void XmlTestWriter::WriteTestCases(XmlWriter& writer, const TestSuiteContext& suiteContext)
{
	for (size_t i = 0; i < suiteContext.GetNumberOfTestPassed(); ++i)
		TestCasePassed(writer, suiteContext.GetTestPassedAt(i));

	for (size_t i = 0; i < suiteContext.GetNumberOfTestSkipped(); ++i)
		TestCaseSkipped(writer, suiteContext.GetTestSkippedAt(i));

	for (size_t i = 0; i < suiteContext.GetNumberOfTestFailed(); ++i)
		TestCaseFailed(writer, suiteContext.GetTestFailedAt(i).first, suiteContext.GetTestFailedAt(i).second);
}

void XmlTestWriter::TestCasePassed(XmlWriter& writer, const TestRunInfo& testRunInfo)
{
	BeginTestCase(writer, testRunInfo);
	EndTestCase(writer);
}

void XmlTestWriter::TestCaseSkipped(XmlWriter& writer, const TestRunInfo& testRunInfo)
{
	BeginTestCase(writer, testRunInfo);

	writer.WriteStartElement("skipped");
	writer.WriteEndElement();

	EndTestCase(writer);
}

void XmlTestWriter::BeginTestCase(XmlWriter& writer, const TestRunInfo& testRunInfo)
{
	writer.WriteStartElement("testcase");
	writer.WriteAttribute("time", 0);
	writer.WriteAttribute("name", testRunInfo.Name);
}

void XmlTestWriter::EndTestCase(XmlWriter& writer)
{
	writer.WriteEndElement();
}

string XmlTestWriter::GetCallStackString(const TestFailedInfo& testFailedInfo) const
{
	string res;
	if (!testFailedInfo.CallStack.empty())
	{
		for (size_t i = 0; i < testFailedInfo.CallStack.size(); i++)
		{
			const CallStackElement& callStackElement = testFailedInfo.CallStack.at(i);
			res += string().Format("\tat %s(%s:%d)\r\n", callStackElement.FunctionName, callStackElement.FileName, callStackElement.FileLineNumber);
		}
	}
	return res;
}

void XmlTestWriter::TestCaseFailed(XmlWriter& writer, const TestRunInfo& testRunInfo, const TestFailedInfo& testFailedInfo)
{
	BeginTestCase(writer, testRunInfo);

	writer.WriteStartElement("failure");
	writer.WriteAttribute("type", "Assert");
	writer.WriteAttribute("message", testFailedInfo.Message);
	string messageInfo;
	messageInfo.Format("%s:(%lu) - {%s}", testFailedInfo.FileName, testFailedInfo.FileLineNumber, testFailedInfo.Condition);
	messageInfo += GetCallStackString(testFailedInfo);
	writer.WriteCData(messageInfo);
	writer.WriteEndElement();

	EndTestCase(writer);
}
