using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using CryUnitWrapper;

namespace TestRunner
{
    class Config
    {
        class OptionFlag
        {
            public OptionFlag(string name)
            {
                m_name = name;
                m_regexParam = new Regex(string.Format("{0}:(.*)", m_name), RegexOptions.IgnoreCase);
                m_value = null;
                m_enabled = false;
            }

            public bool Match(string arg)
            {
                bool matched = false;
                if (string.Compare(m_name, arg, true) == 0)
                {
                    matched = true;
                    m_enabled = true;
                }
                else
                {
                    Match param = m_regexParam.Match(arg);
                    if (param.Success)
                    {
                        matched = true;
                        m_enabled = true;
                        m_value = param.Groups[1].Value;
                    }
                }
                return matched;
            }

            public bool Enabled
            {
                get { return m_enabled; }
                set { m_enabled = value; }
            }

            public string Value
            {
                get { return m_value; }
            }

            private string m_name;
            private Regex m_regexParam;
            private string m_value;
            private bool m_enabled;
        }

        private string m_baseDir = null;
        private ICollection<string> dllNames = new List<string>();
        ICollection<OptionFlag> m_optionFlags;
        private OptionFlag m_reportFlag;
        private OptionFlag m_smokeFlag;
        private OptionFlag m_forceFlag;

        public Config(string[] args)
        {
            InitOptionFlags();

            if (args.Length >= 2)
            {
                ParseCommandLine(args);
            }
        }

        public Config(string baseDir, string dllName, bool runSmokeTests)
        {
            InitOptionFlags();
            m_baseDir = baseDir;
            dllNames = new String[] { dllName };
        }

        private void InitOptionFlags()
        {
            m_optionFlags = new List<OptionFlag>();

            m_reportFlag = new OptionFlag("REPORT");
            m_optionFlags.Add(m_reportFlag);

            m_smokeFlag = new OptionFlag("SMOKE");
            m_optionFlags.Add(m_smokeFlag);
            
            m_forceFlag = new OptionFlag("FORCE");
            m_optionFlags.Add(m_forceFlag);
        }

        public bool EnableSmokeTests
        {
            get { return m_smokeFlag.Enabled; }
        }

        public bool EnableXmlReport
        {
            get { return m_reportFlag.Enabled; }
        }

        public string ReportFile
        {
            get { return m_reportFlag.Value; }
        }

        public bool ForceRun
        {
            get { return m_forceFlag.Enabled; }
        }

        public string BaseDir
        {
            get { return m_baseDir; }
        }

        public ICollection<string> DllNames
        {
            get { return dllNames; }
        }

        public bool IsOK
        {
            get { return m_baseDir != null && dllNames.Count > 0; }
        }

        private void ParseCommandLine(string[] args)
        {
            int argIndex = 0;
            bool found = false;
            do
            {
                found = MatchOption(args[argIndex]);
                if (found)
                {
                    ++argIndex;
                }
            }
            while (found);

            ParseBaseDirAndDllNames(args, argIndex);
        }

        private bool MatchOption(string arg)
        {
            foreach (OptionFlag optionFlag in m_optionFlags)
            {
                if (optionFlag.Match(arg))
                    return true;
            }
            return false;
        }

        private void ParseBaseDirAndDllNames(string[] args, int argIndex)
        {
            m_baseDir = args[argIndex];

            for (int i = argIndex + 1; i < args.Length; ++i)
            {
                dllNames.Add(args[i]);
            }
        }
    }
}
