using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

namespace TestRunner
{
    class ConsoleRunner
    {
        public static void Run(Config config)
        {
            if (!config.IsOK)
            {
                Environment.ExitCode = 1;
                return;
            }

            String env = Environment.GetEnvironmentVariable("CRY_NO_TESTING");
            if (env != null && !config.ForceRun)
            {
                Environment.ExitCode = 0;
                return;
            }

            RunnerObserver runnerObserver = new RunnerObserver(CreateTestWriter(config));
            Runner runner = new Runner(config, runnerObserver);

            RunAllTestsAsync(runner, runnerObserver);
            Environment.ExitCode = runnerObserver.ExitCode;
        }

        public static void LogEnvironment(string[] args)
        {
            Console.WriteLine("Current directory: {0}", System.Environment.CurrentDirectory);
            for (int i = 0; i < args.Length; ++i)
            {
                Console.Write(" {0}", args[i]);
            }
            Console.WriteLine();
        }

        private static TestWriter CreateTestWriter(Config config)
        {
            TestWriter testWriter = null;
            if (config.EnableXmlReport)
            {
                testWriter = new XmlTestWriter(GetReportDirectory(config));
            }
            else
            {
                testWriter = new TestWriter();
            }
            return testWriter;
        }

        private static string GetReportDirectory(Config config)
        {
            string xmlReportsDir;
            if (config.ReportFile != null && config.ReportFile.Length > 0)
            {
                xmlReportsDir = config.ReportFile;
            }
            else
            {
                xmlReportsDir = Path.Combine(config.BaseDir, "..\\Xml-Reports");
            }
            Directory.CreateDirectory(xmlReportsDir);
            return xmlReportsDir;
        }

        private delegate void RunTestsDelegate();

        private static void RunAllTestsAsync(Runner runner, RunnerObserver runnerObserver)
        {
            TimeSpan timeout = new TimeSpan(0, 5, 0);
            RunTestsDelegate runTests = runner.RunTests;
            IAsyncResult runTestsAsyncResult = runTests.BeginInvoke(null, null);

            if (runTestsAsyncResult.AsyncWaitHandle.WaitOne(timeout, false))
            {
                runTests.EndInvoke(runTestsAsyncResult);
            }
            else
            {
                runnerObserver.TimeoutElapsed();
            }
        }
    }
}
