using System;
using System.IO;
using System.Collections.Generic;
using CryUnitWrapper;

namespace TestRunner
{
    class TestRunTimedInfo : TestRunInfo
    {
        private TimeSpan m_elapsedTime;

        public TestRunTimedInfo(TestRunInfo testInfo)
			: base(testInfo.Name, testInfo.Fixture, testInfo.FileName, testInfo.FileLineNumber)
        {
            m_elapsedTime = TimeSpan.Zero;
        }

        public void ComputeElapsedTime(DateTime startTime)
        {
            m_elapsedTime = DateTime.Now.Subtract(startTime);
        }

        public TimeSpan ElapsedTime 
        {
            get { return m_elapsedTime; }
        }
    }

    class TestFailureInfo : TestRunTimedInfo
    {
        FailureInfo m_failureInfo;

        public TestFailureInfo(TestRunInfo testInfo, FailureInfo failureInfo)
            : base(testInfo)
        {
            m_failureInfo = failureInfo;
        }

        public string Condition
        {
            get { return m_failureInfo.Condition; }
        }

        public string Message
        {
            get { return m_failureInfo.Message; }
        }

        public string ErrorFileName
        {
            get { return m_failureInfo.FileName; }
        }

        public int ErrorLineNumber
        {
            get { return m_failureInfo.FileLineNumber; }
        }

        public CallStackElement[] CallStack
        {
            get { return m_failureInfo.CallStack; }
        }
   }

    class SuiteContext
    {
        private string m_suiteName;
        private ICollection<TestRunTimedInfo> m_testPassedCollection;
        private ICollection<TestFailureInfo> m_testFailedCollection;
        private ICollection<TestRunTimedInfo> m_testSkippedCollection;
        private DateTime m_startTime;

        public SuiteContext(string suiteName)
        {
            m_suiteName = suiteName;
            m_testPassedCollection = new List<TestRunTimedInfo>();
            m_testFailedCollection = new List<TestFailureInfo>();
            m_testSkippedCollection = new List<TestRunTimedInfo>();
            m_startTime = DateTime.Now;
        }

        public string SuiteName
        {
            get { return m_suiteName; }
        }

        public ICollection<TestFailureInfo> TestFailedCollection
        {
            get { return m_testFailedCollection; }
        }

        public ICollection<TestRunTimedInfo> TestPassedCollection
        {
            get { return m_testPassedCollection; }
        }

        public ICollection<TestRunTimedInfo> TestSkippedCollection
        {
            get { return m_testSkippedCollection; }
        }

        public TimeSpan ElapsedTime
        {
            get { return DateTime.Now.Subtract(m_startTime); }
        }
    }

    class RunnerObserver
    {
        private TestWriter m_testWriter;
        private int m_exitCode;

        private SuiteContext m_suiteContext;
        private TestRunInfo m_runningTest;
        private ICollection<TestRunInfo> m_testSkippedCollection;
        private ICollection<TestFailureInfo> m_testFailedCollection;
        private int m_testRunCount;
        private int m_testPassedCount;
        private DateTime m_startRunTestInDllTime;
        private DateTime m_runningTestStartTime;

        public RunnerObserver(TestWriter writer)
        {
            m_testWriter = writer;
            m_exitCode = 0;

            m_suiteContext = null;
            m_runningTest = null;
            m_startRunTestInDllTime = DateTime.Now;
            m_runningTestStartTime = DateTime.Now;
            Reset();
        }

        private void Reset()
        {
            m_testSkippedCollection = new List<TestRunInfo>();
            m_testFailedCollection = new List<TestFailureInfo>();
            m_testRunCount = 0;
            m_testPassedCount = 0;
        }

        public int ExitCode
        {
            get { return m_exitCode;  }
        }

        public void FatalError(Exception exception)
        {
            m_exitCode = 1;
            Console.WriteLine("FatalError: {0}", exception.Message);
            Console.WriteLine("StackTrace: {0}", exception.StackTrace);
            if (m_runningTest != null)
            {
                Console.WriteLine("RunningTest: {0}", m_runningTest.Name);
                VisualStudioOutputInfo(m_runningTest.FileName, m_runningTest.FileLineNumber, exception.Message);

                if (m_suiteContext != null)
                {
                    m_suiteContext.TestFailedCollection.Add(new TestFailureInfo(m_runningTest, new FailureInfo("FatalError", exception.Message, m_runningTest.FileName, m_runningTest.FileLineNumber, new CallStackElement[0])));
                    m_testWriter.TestSuiteEnd(m_suiteContext);
                }
            }
        }

         private void VisualStudioOutputInfo(string fileName, int fileLineNumber, string message)
        {
            Console.WriteLine("{0}({1}) : {2}", fileName, fileLineNumber, message);
        }

        public void TestSuiteStart(string suiteName)
        {
            m_suiteContext = new SuiteContext(suiteName);
        }

        public void TestSuiteEnd(string suiteName)
        {
            int testFailed = m_suiteContext.TestFailedCollection.Count;
            int testPassed = m_suiteContext.TestPassedCollection.Count;
            Console.WriteLine("{0} [{1}:{2}:{3}] [{4}.{5:000}s]", m_suiteContext.SuiteName, testPassed + testFailed, testPassed, testFailed, m_suiteContext.ElapsedTime.Seconds, m_suiteContext.ElapsedTime.Milliseconds);
            m_testWriter.TestSuiteEnd(m_suiteContext);
        }

        public void StartTestRun(TestRunInfo testInfo)
        {
            m_runningTestStartTime = DateTime.Now;
            m_runningTest = testInfo;
        }

        public void TestRun(TestRunInfo testInfo)
        {
            ++m_testRunCount;            
        }

        public void TestPassed(TestRunInfo testInfo)
        {
            ++m_testPassedCount;
            TestRunTimedInfo testRunTimedInfo = new TestRunTimedInfo(testInfo);
            testRunTimedInfo.ComputeElapsedTime(m_runningTestStartTime);

            m_suiteContext.TestPassedCollection.Add(testRunTimedInfo);
        }

        public void TestFailed(TestRunInfo testInfo, FailureInfo failureInfo)
        {
            m_exitCode = 1;
            TestFailureInfo testFailureInfo = new TestFailureInfo(testInfo, failureInfo);
            testFailureInfo.ComputeElapsedTime(m_runningTestStartTime);

            m_suiteContext.TestFailedCollection.Add(testFailureInfo);
            m_testFailedCollection.Add(testFailureInfo);
        }

        public void TestSkipped(TestRunInfo testInfo)
        {
            TestRunTimedInfo testRunTimedInfo = new TestRunTimedInfo(testInfo);
            m_suiteContext.TestSkippedCollection.Add(testRunTimedInfo);
            m_testSkippedCollection.Add(testRunTimedInfo);
        }

        public void TimeoutElapsed()
        {
            Console.WriteLine("Timeout elapsed - maybe a loop in some test?");
        }

        public void StartRunTestInDll(string baseDir, string dllName)
        {
            m_startRunTestInDllTime = DateTime.Now;
            Console.WriteLine("==================== Running tests ============================ ");
            Console.WriteLine("Module: {0}", dllName);
            Console.WriteLine("System folder: {0}", Path.GetFullPath(baseDir));
            Console.WriteLine("=============================================================== ");
            Console.WriteLine("Test Suites:");
        }

        public void EndRunTestInDll()
        {
            TimeSpan elapsedTime = DateTime.Now.Subtract(m_startRunTestInDllTime);

            Console.WriteLine("=============================================================== ");
            Console.WriteLine("Test Results:");
            Console.WriteLine("Run: {0}", m_testRunCount);
            Console.WriteLine("Passed: {0}", m_testPassedCount);
            Console.WriteLine("Failed: {0}", m_testFailedCollection.Count);
            Console.WriteLine("ElapsedTime: {0:0.00}s", elapsedTime.TotalSeconds);
            Console.WriteLine("=============================================================== ");
            LogSkippedTest();
            LogFailedTest();

            Reset();
        }

        private void LogSkippedTest()
        {
            if (m_testSkippedCollection.Count == 0)
                return;

            Console.WriteLine("Skipped: {0}", m_testSkippedCollection.Count);
            foreach (TestRunInfo testRunInfo in m_testSkippedCollection)
            {
                VisualStudioOutputInfo(testRunInfo.FileName, testRunInfo.FileLineNumber, testRunInfo.Name);
            }
            Console.WriteLine("=============================================================== ");
        }
        
        private void LogCallStack(TestFailureInfo testFailureInfo)
        {
            CallStackElement[] callStack = testFailureInfo.CallStack;
            if (callStack.Length > 0)
            {
                Console.WriteLine("StackTrace:");
                for (int i = 0; i < callStack.Length; i++)
                {
                    VisualStudioOutputInfo(callStack[i].FileName, callStack[i].FileLineNumber, callStack[i].FunctionName);
                }
           }
        }

        private void LogFailedTest()
        {
            if (m_testFailedCollection.Count == 0)
                return;
            
            Console.WriteLine("");
            Console.WriteLine("\n\n\n\nList of failing tests:");

            foreach (TestFailureInfo testInfo in m_testFailedCollection)
            {
                VisualStudioOutputInfo(testInfo.FileName, testInfo.FileLineNumber, string.Format("Test Failed {0}", testInfo.Name));
                string msg = testInfo.Message;
                if (testInfo.Condition.Length > 0)
                {
                    msg += ": " + testInfo.Condition;
                }

                VisualStudioOutputInfo(testInfo.ErrorFileName, testInfo.ErrorLineNumber, msg);
                LogCallStack(testInfo);
                Console.WriteLine("");
            }

            Console.WriteLine("====================");
        }
    }
}
