using System;
using System.Collections.Generic;
using System.IO;
using System.Xml;
using System.Text;
using CryUnitWrapper;

namespace TestRunner
{
    class TestWriter
    {
        public virtual void TestSuiteEnd(SuiteContext suiteContext) 
        {

        }
    }

    class XmlTestWriter : TestWriter
    {
        private String m_xmlPath;

        public XmlTestWriter(String xmlPath)
        {
            m_xmlPath = xmlPath;
        }

        public override void TestSuiteEnd(SuiteContext suiteContext)
        {
            string reportFileName = string.Format("TEST-{0}.xml", suiteContext.SuiteName);
            XmlTextWriter writer = new XmlTextWriter(Path.Combine(m_xmlPath, reportFileName), Encoding.Default);

            writer.WriteStartDocument();
            writer.WriteStartElement("testsuite");

            WriteSuiteAttributes(writer, suiteContext);
            WriteTestCases(writer, suiteContext);
            
            writer.WriteEndDocument();
            writer.Close();
        }

        private void WriteSuiteAttributes(XmlTextWriter writer, SuiteContext suiteContext)
        {
            int passedCount = suiteContext.TestPassedCollection.Count;
            int failedCount = suiteContext.TestFailedCollection.Count;
            int skippedCount = suiteContext.TestSkippedCollection.Count;

            writer.WriteAttributeString("errors", "0");
            writer.WriteAttributeString("skipped", Convert.ToString(skippedCount));
            writer.WriteAttributeString("tests", Convert.ToString(passedCount + failedCount + skippedCount));
            writer.WriteAttributeString("time", FormatElapsedTime(suiteContext.ElapsedTime));
            writer.WriteAttributeString("failures", Convert.ToString(failedCount));
            writer.WriteAttributeString("name", suiteContext.SuiteName);
        }

        private static string FormatElapsedTime(TimeSpan elapsedTime)
        {
            return string.Format("{0:0.00}", elapsedTime.TotalSeconds);
        }

        private void WriteTestCases(XmlTextWriter writer, SuiteContext suiteContext)
        {
            foreach (TestRunTimedInfo testInfo in suiteContext.TestPassedCollection)
                TestCasePassed(writer, testInfo);

            foreach (TestRunTimedInfo testInfo in suiteContext.TestSkippedCollection)
                TestCaseSkipped(writer, testInfo);

            foreach (TestFailureInfo testInfo in suiteContext.TestFailedCollection)
                TestCaseFailed(writer, testInfo);
        }

        private void TestCasePassed(XmlTextWriter writer, TestRunTimedInfo testInfo)
		{
            BeginTestCase(writer, testInfo);
            EndTestCase(writer);
        }

        private void TestCaseSkipped(XmlTextWriter writer, TestRunTimedInfo testInfo)
		{
            BeginTestCase(writer, testInfo);

            writer.WriteStartElement("skipped");
            writer.WriteEndElement();

            EndTestCase(writer);
        }

        private string GetCallStackString(TestFailureInfo testFailureInfo)
        {
            StringBuilder stringBuilder = new StringBuilder();
            CallStackElement[] callStack = testFailureInfo.CallStack;
            if (callStack.Length > 0)
            {
                stringBuilder.Append("\r\nStackTrace:\r\n");
                for (int i = 0; i < callStack.Length; i++)
                {
                    stringBuilder.Append(string.Format("\tat {0}({1}:{2})\r\n", callStack[i].FunctionName, callStack[i].FileName, callStack[i].FileLineNumber));
                }
                
            }
            return stringBuilder.ToString();
        }

        private void TestCaseFailed(XmlTextWriter writer, TestFailureInfo testInfo)
		{
            BeginTestCase(writer, testInfo);

            writer.WriteStartElement("failure");
            writer.WriteAttributeString("type", "Assert");
            writer.WriteAttributeString("message", testInfo.Message);
            string messageInfo = string.Format("{0}:({1}) - {2}", testInfo.ErrorFileName, testInfo.ErrorLineNumber, testInfo.Condition);
            messageInfo += GetCallStackString(testInfo);
            writer.WriteCData(messageInfo);
            writer.WriteEndElement();

            EndTestCase(writer);
        }

        private void BeginTestCase(XmlTextWriter writer, TestRunTimedInfo testInfo)
        {
            writer.WriteStartElement("testcase");
            writer.WriteAttributeString("time", FormatElapsedTime(testInfo.ElapsedTime));
            writer.WriteAttributeString("name", testInfo.Name);
        }

        private void EndTestCase(XmlTextWriter writer)
        {
            writer.WriteEndElement();
        }
    }
}
