﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using CryUnitWrapper;
using System.IO;

namespace TestInterface
{
	public class TestShell : MarshalByRefObject
	{
		public delegate void OnLoadEventHandler(TestShellLoadEventArgs eventArgs);
		public delegate void OnSuiteInfoEventHandler(TestSuiteInfoEventArgs eventArgs);
		public delegate void OnTestRunInfoHandler(TestRunInfoEventArgs eventArgs);
		public delegate void OnTestFailedInfoHandler(TestFailedInfoEventArgs eventArgs);
		public delegate void OnAnimationTestingFlushHandler(AnimationTestingFlushEvengArgs eventArgs);
		public delegate void OnAnimationTestingRunInfoHandler(AnimationTestingRunInfoEventArgs eventArgs);
		public delegate void OnAnimationTestingFatalErrorHandler(AnimationTestingFatalErrorEventArgs eventArgs);

		public event OnLoadEventHandler OnLoad;
		public event OnSuiteInfoEventHandler OnSuiteStart;
		public event OnSuiteInfoEventHandler OnSuiteEnd;
		public event OnTestRunInfoHandler OnTestPassed;
		public event OnTestFailedInfoHandler OnTestFailed;
		public event OnTestRunInfoHandler OnTestSkipped;
		public event OnAnimationTestingFlushHandler OnAnimationTestingFlush;
		public event OnAnimationTestingRunInfoHandler OnAnimationTestingTestPassed;
		public event OnAnimationTestingRunInfoHandler OnAnimationTestingTestFailed;
		public event OnAnimationTestingFatalErrorHandler OnAnimationTestingFatalError;

		private CommonTestTree commonTestTree;

		public TestShell()
		{
			Console.WriteLine("TestShell ctor called");
		}

		public void LoadSmoke()
		{
			Load(true);
		}

		public void LoadUnit()
		{
			Load(false);
		}

		private void Load(bool loadSmokeTest)
		{
			Console.WriteLine("Load called SmokeTest [{0}]", loadSmokeTest);
			try
			{
				TestSuiteLoader testSuiteLoader = new TestSuiteLoader(Config.GetBinaries(), Config.DllName, loadSmokeTest);

				commonTestTree = new TestTree(testSuiteLoader);
				OnLoad(new TestShellLoadEventArgs(commonTestTree, false));
			}
			catch (Exception ex)
			{
				Console.WriteLine(ex.GetType().Name);
				Console.WriteLine(ex.Message);
				Console.WriteLine(ex.StackTrace);
			}
		}

		public void LoadAnimationEngine(string suiteFileName)
		{
			try
			{
				AnimationTesting.Suite animationSuite = new AnimationTesting.Suite();
				animationSuite.Load(new StreamReader(suiteFileName));
				Console.WriteLine("Creating engine instance for animation");
				commonTestTree = new AnimationTestTree(new AnimationTesting.Engine(Config.Binaries), animationSuite);
				OnLoad(new TestShellLoadEventArgs(commonTestTree, true));
			}
			catch (Exception exception)
			{
				Console.WriteLine(exception.Message);
			}
		}

		public void RunAll()
		{
			Console.WriteLine("Operation RunAll called");
			commonTestTree.Run(new TestListener(this));
		}

		public void RunSuite(string name)
		{
			Console.WriteLine("RunSuite " + name);
			TreeNode suiteNode = commonTestTree.FindChild(name);
			if (suiteNode == null)
			{
				Console.WriteLine("Suite not found");
				return;
			}

			suiteNode.Run(new TestListener(this));
		}

		public void RunFixture(string name)
		{
			Console.WriteLine("RunFixture " + name);
			TreeNode fixtureNode = commonTestTree.FindFixture(name);
			if (fixtureNode == null)
			{
				Console.WriteLine("Fixture not found");
				return;
			}

			fixtureNode.Run(new TestListener(this));
		}

		public void RunTest(string testName, string fixtureName)
		{
			Console.WriteLine("RunTest " + testName);
			TreeNode fixtureNode = commonTestTree.FindFixture(fixtureName);
			if (fixtureNode == null)
			{
				Console.WriteLine("Fixture {0} not found", fixtureName);
				return;
			}
			TreeNode testNode = fixtureNode.FindChild(testName);
			if (testNode == null)
			{
				Console.WriteLine("Test not found", fixtureName);
				return;
			}

			testNode.Run(new TestListener(this));
		}

		public void Close()
		{
			Console.WriteLine("Operation Close called");
		}

		public void NotifySuiteEnd(string suiteName)
		{
			OnSuiteEnd(new TestSuiteInfoEventArgs(suiteName));
		}

		public void NotifySuiteStart(string suiteName)
		{
			OnSuiteStart(new TestSuiteInfoEventArgs(suiteName));
		}
		
		public void NotifyTestFailed(TestRunInfo testInfo, FailureInfo failureInfo)
		{
			OnTestFailed(new TestFailedInfoEventArgs(testInfo, failureInfo));
		}

		public void NotifyTestPassed(TestRunInfo testInfo)
		{
			OnTestPassed(new TestRunInfoEventArgs(testInfo));
		}

		public void NotifyTestSkipped(TestRunInfo testInfo)
		{
			OnTestSkipped(new TestRunInfoEventArgs(testInfo));
		}

		public void NotifyAnimationTestingTestPassed(AnimationTestingRunInfo runInfo)
		{
			OnAnimationTestingTestPassed(new AnimationTestingRunInfoEventArgs(runInfo));
		}

		public void NotifyAnimationTestingTestFailed(AnimationTestingRunInfo runInfo)
		{
			OnAnimationTestingTestFailed(new AnimationTestingRunInfoEventArgs(runInfo));
		}

		public void NotifyAnimationTestingFatalError(string message)
		{
			OnAnimationTestingFatalError(new AnimationTestingFatalErrorEventArgs(message));
		}

		public void NotifyAnimationTestingFlush(bool isOK)
		{
			OnAnimationTestingFlush(new AnimationTestingFlushEvengArgs(isOK));
		}
	}
}
