﻿using System;
using System.Collections.Generic;
using CryUnitWrapper;

namespace TestInterface
{
	public interface TestTreeVisitor
	{
		void Visit(SuiteNode visitor);
		void Visit(FixtureNode visitor);
		void Visit(TestNode visitor);
	}

	[Serializable]
	public class TreeNode
	{
		private Dictionary<string, TreeNode> children;
		private TreeNode parent;
		private string name;

		[NonSerialized]
		private object tag;

		public TreeNode(string name, TreeNode parent)
		{
			children = new Dictionary<string, TreeNode>();
			this.parent = parent;
			this.name = name;
		}

		public void AddChild(TreeNode treeNode)
		{
			children.Add(treeNode.Name, treeNode);
		}

		public TreeNode FindChild(string name)
		{
			TreeNode res;
			if (children.TryGetValue(name, out res))
			{
				return res;
			}
			return null;
		}

		public IDictionary<string, TreeNode> Children
		{
			get { return children; }
		}

		public object Tag
		{
			get { return tag; }
			set { tag = value; }
		}

		public string Name
		{
			get { return name; }
			set { name = value; }
		}

		public TreeNode Parent
		{
			get { return parent; }
		}

		public virtual void AcceptVisitor(TestTreeVisitor visitor)
		{

		}

		public virtual void Run(TestListener listener)
		{

		}

		public virtual int ComputeNumberOfTests()
		{
			int res = 0;
			foreach (TreeNode testNode in Children.Values)
			{
				res += testNode.ComputeNumberOfTests();
			}
			return res;
		}
	}

	[Serializable]
	public abstract class CommonTestTree : TreeNode
	{
		public CommonTestTree(string name, TreeNode parent)
			: base(name, parent)
		{

		}

		public abstract TreeNode FindFixture(string fixtureName);
	}

	[Serializable]
	public class SuiteNode : TreeNode
	{
		[NonSerialized]
		private TestSuite testSuite;

		public SuiteNode(string name, TestSuite testSuite, TreeNode parent)
			: base(name, parent)
		{
			this.testSuite = testSuite;
		}

		public override void AcceptVisitor(TestTreeVisitor visitor)
		{
			visitor.Visit(this);
		}

		public override void Run(TestListener listener)
		{
			testSuite.Run(listener);
		}
	}

	[Serializable]
	public class FixtureNode : TreeNode
	{
		public FixtureNode(string name, TreeNode parent)
			: base(name, parent)
		{

		}

		public override void AcceptVisitor(TestTreeVisitor visitor)
		{
			visitor.Visit(this);
		}

		public override void Run(TestListener listener)
		{
			foreach (TreeNode test in Children.Values)
			{
				test.Run(listener);
			}
		}
	}

	[Serializable]
	public class TestNode : TreeNode
	{
		[NonSerialized]
		private TestSuite testSuite;

		[NonSerialized]
		private TestRunInfo testRunInfo;

		public TestNode(string name, TestRunInfo testRunInfo, TestSuite testSuite, TreeNode parent)
			: base(name, parent)
		{
			this.testSuite = testSuite;
			this.testRunInfo = testRunInfo;
		}

		public override void AcceptVisitor(TestTreeVisitor visitor)
		{
			visitor.Visit(this);
		}

		public override void Run(TestListener listener)
		{
			testSuite.RunTestCase(testRunInfo, listener);
		}

		public override int ComputeNumberOfTests()
		{
			return 1;
		}
	}

	[Serializable]
	public class TestTree : CommonTestTree
	{
		private IDictionary<string, TreeNode> fixtures = new Dictionary<string, TreeNode>();

		public override TreeNode FindFixture(string fixtureName)
		{
			return fixtures[fixtureName];
		}

		private FixtureNode GetFixtureNode(TreeNode suiteNode, string fixtureName)
		{
			TreeNode fixtureNode = suiteNode.FindChild(fixtureName);
			if (fixtureNode == null)
			{
				fixtureNode = new FixtureNode(fixtureName, suiteNode);
				fixtures.Add(fixtureName, fixtureNode);
				suiteNode.AddChild(fixtureNode);
			}

			return (FixtureNode)fixtureNode;
		}

		private void Add(TestSuite testSuite)
		{
			SuiteNode suiteNode = new SuiteNode(testSuite.Name, testSuite, this);
			AddChild(suiteNode);

			foreach (TestRunInfo testRunInfo in testSuite.TestCases)
			{
				FixtureNode fixtureNode = GetFixtureNode(suiteNode, testRunInfo.Fixture);
				fixtureNode.AddChild(new TestNode(testRunInfo.Name, testRunInfo, testSuite, fixtureNode));
			}
		}

		public override void Run(TestListener listener)
		{
			foreach (TreeNode treeNode in Children.Values)
			{
				treeNode.Run(listener);
			}
		}

		public TestTree(TestSuiteLoader testSuiteLoader)
			: base("Root", null)
		{
			foreach (TestSuite suite in testSuiteLoader.SuiteList)
			{
				Add(suite);
			}
		}
	}
}
