﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Drawing;

using Forms = System.Windows.Forms;
using TestInterface;

namespace TestViewer
{
	public class RemoteTestExecutorListener : MarshalByRefObject
	{
		private CommonTestTree commonTestTree;
		private TestViewerForm form;
		private Forms.TreeView testTreeView;
		private ColoredProgressBar progressBar;

		public RemoteTestExecutorListener(TestViewerForm form, Forms.TreeView testTreeView, ColoredProgressBar progressBar)
		{
			this.form = form;
			this.testTreeView = testTreeView;
			this.progressBar = progressBar;
		}

		public int ComputeNumberOfTests()
		{
			return commonTestTree.ComputeNumberOfTests();
		}

		public int ComputeNumberOfTests(TreeNode treeNode)
		{
			return treeNode.ComputeNumberOfTests();
		}
	
		private void PopulateView(TreeNode treeNode, Forms.TreeNodeCollection viewNodes)
		{
			foreach (KeyValuePair<string, TreeNode> child in treeNode.Children)
			{
				Forms.TreeNode childViewNode = viewNodes.Add(child.Key);
				child.Value.Tag = childViewNode;
				childViewNode.Tag = child.Value;
				childViewNode.ForeColor = Color.Gray;
				PopulateView(child.Value, childViewNode.Nodes);
			}
		}

		public void OnLoadDone(TestShellLoadEventArgs eventArgs)
		{
			form.BeginInvoke((Forms.MethodInvoker)delegate() 
			{
				commonTestTree = eventArgs.TestTree;
				PopulateView(eventArgs.TestTree, testTreeView.Nodes);
				form.ButtonsEnabled = true;
				if (eventArgs.ShowAnimationTestConsole)
				{
					form.ShowAnimationTestConsole();
				}
			});
		}

		public void OnSuiteStart(TestSuiteInfoEventArgs eventArgs)
		{
		}

		public void OnSuiteEnd(TestSuiteInfoEventArgs eventArgs)
		{
		}

		public Color GetNewColor(Color controlColor, Color color)
		{
			if (color == Color.Green)
			{
				if (controlColor == Color.Yellow || controlColor == Color.Red)
				{
					return controlColor;
				}
			}

			if (color == Color.Yellow)
			{
				if (controlColor == Color.Red)
				{
					return controlColor;
				}
			}

			return color;
		}

		private void SetTestColorAsync(TreeNode treeNode, Color color)
		{
			form.BeginInvoke((Forms.MethodInvoker)delegate()
			{
				Forms.TreeNode testNode = (Forms.TreeNode)treeNode.Tag;
				Forms.TreeNode fixtureNode = (Forms.TreeNode)treeNode.Parent.Tag;
				Forms.TreeNode suiteNode = (Forms.TreeNode)treeNode.Parent.Parent.Tag;

				testNode.ForeColor = GetNewColor(testNode.ForeColor, color);
				fixtureNode.ForeColor = GetNewColor(fixtureNode.ForeColor, color);
				suiteNode.ForeColor = GetNewColor(suiteNode.ForeColor, color);
				progressBar.Color = GetNewColor(progressBar.Color, color);
				progressBar.Value++;
			});
		}

		private TreeNode FindTestNode(CryUnitWrapper.TestRunInfo testRunInfo)
		{
			return commonTestTree.FindFixture(testRunInfo.Fixture).FindChild(testRunInfo.Name);
		}

		public void OnTestPassed(TestRunInfoEventArgs eventArgs)
		{
			TreeNode testTreeNode = FindTestNode(eventArgs.TestRunInfo);
			SetTestColorAsync(testTreeNode, Color.Green);
		}

		public void OnTestSkipped(TestRunInfoEventArgs eventArgs)
		{
			TreeNode testTreeNode = FindTestNode(eventArgs.TestRunInfo);
			SetTestColorAsync(testTreeNode, Color.Yellow);
		}

		public void OnTestFailed(TestFailedInfoEventArgs eventArgs)
		{
			TreeNode testTreeNode = FindTestNode(eventArgs.TestRunInfo);
			SetTestColorAsync(testTreeNode, Color.Red);
		}

		public void OnAnimationTestingFlush(AnimationTestingFlushEvengArgs eventArgs)
		{
			form.BeginInvoke((Forms.MethodInvoker)delegate()
			{
				progressBar.Color = eventArgs.IsOK ? Color.Green : Color.Red;
				progressBar.Value = progressBar.MaxValue;
			});
		}

		private void FillAnimationTestingInfoWithRunInfo(AnimationTestingRunInfoEventArgs eventArgs, string msg)
		{
			form.BeginInvoke((Forms.MethodInvoker)delegate()
			{
				form.AnimationTestConsole.Info.Text += string.Format("{0} [{1}] Character [{2}] Animation [{3}]\r\n", msg, eventArgs.RunInfo.TestName, eventArgs.RunInfo.CharacterName, eventArgs.RunInfo.AnimationName);
			});
		}

		public void OnAnimationTestingTestFailed(AnimationTestingRunInfoEventArgs eventArgs)
		{
			FillAnimationTestingInfoWithRunInfo(eventArgs, "TestFailed");
		}

		public void OnAnimationTestingTestPassed(AnimationTestingRunInfoEventArgs eventArgs)
		{
			FillAnimationTestingInfoWithRunInfo(eventArgs, "TestPassed");
		}

		public void OnAnimationTestingFatalError(AnimationTestingFatalErrorEventArgs eventArgs)
		{
			form.BeginInvoke((Forms.MethodInvoker)delegate()
			{
				form.AnimationTestConsole.Info.Text += string.Format("FatalError: {0}\r\n", eventArgs.Message);
			});
		}
	}
}
