﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using Forms = System.Windows.Forms;

using TestInterface;
using System.Diagnostics;

namespace TestViewer
{
	public partial class TestViewerForm : Forms.Form
	{
		private RemoteTestExecutor remoteTestExecutor;
		private RemoteTestExecutorListener remoteTestExecutorListener;
		private AnimationTestConsole animationTestConsole;

		public TestViewerForm()
		{
			InitializeComponent();

			remoteTestExecutorListener = new RemoteTestExecutorListener(this, testTreeView, progressBar);
			remoteTestExecutor = new RemoteTestExecutor(remoteTestExecutorListener);
			remoteTestExecutor.OnStop += OnProcessTerminated;

			ButtonsEnabled = false;
		}

		public bool ButtonsEnabled
		{
			set 
			{
				buttonLoadUnit.Enabled = value;
				buttonLoadSmoke.Enabled = value;
				buttonLoadAnimation.Enabled = value;
				buttonRunAll.Enabled = value;
				buttonRunSelection.Enabled = value;
				buttonDetach.Enabled = value;

				if (animationTestConsole != null)
				{
					buttonRunSelection.Enabled = false;
				}
			}
		}

		private void TestViewerForm_Load(object sender, EventArgs e)
		{
			remoteTestExecutor.OnStart += LoadUnitOnProcessStart;
			remoteTestExecutor.Start();
		}
		
		private void OnProcessTerminated()
		{
			remoteTestExecutor.Start();
		}

		private void LoadUnitOnProcessStart()
		{
			remoteTestExecutor.LoadUnit();
			remoteTestExecutor.OnStart -= LoadUnitOnProcessStart;
		}

		private void LoadSmokeOnProcessStart()
		{
			remoteTestExecutor.LoadSmoke();
			remoteTestExecutor.OnStart -= LoadSmokeOnProcessStart;
		}

		private void LoadTestOnProcessStart(RemoteTestExecutorStartHandler testToLoad)
		{
			ButtonsEnabled = false;
			remoteTestExecutor.OnStart += testToLoad;

			if (remoteTestExecutor.IsRunning())
			{
				testTreeView.Nodes.Clear();
				if (animationTestConsole != null)
				{
					animationTestConsole.Close();
					animationTestConsole = null;
				}
				remoteTestExecutor.Stop();
			}
			else
			{
				remoteTestExecutor.Start();
			}
		}

		private void buttonLoadUnit_Click(object sender, EventArgs e)
		{
			LoadTestOnProcessStart(LoadUnitOnProcessStart);
		}

		private void buttonLoadSmoke_Click(object sender, EventArgs e)
		{
			LoadTestOnProcessStart(LoadSmokeOnProcessStart);
		}

		private void TestViewerForm_FormClosed(object sender, Forms.FormClosedEventArgs e)
		{
			if (!remoteTestExecutor.IsRunning())
			{
				return;
			}

			remoteTestExecutor.Stop();
		}

		private void MakeTreeViewGray(Forms.TreeNodeCollection treeNodeCollection)
		{
			foreach (Forms.TreeNode child in treeNodeCollection)
			{
				child.ForeColor = Color.Gray;
				MakeTreeViewGray(child.Nodes);
			}
		}

		private void ResetTestView()
		{
			MakeTreeViewGray(testTreeView.Nodes);
			progressBar.Value = 0;
			progressBar.Color = Color.Green;
		}

		private void RunFinishCallBack(IAsyncResult asyncResult)
		{
			BeginInvoke((Forms.MethodInvoker)delegate()
			{
				ButtonsEnabled = true;
			});
		}

		private delegate void RunAllDelegate();

		private void buttonRunAll_Click(object sender, EventArgs e)
		{
			if (!IsTestExecutorRunning())
			{
				return;
			}

			ResetTestView();
			ButtonsEnabled = false;

			progressBar.MaxValue = remoteTestExecutorListener.ComputeNumberOfTests();
			RunAllDelegate runAllDelegate = new RunAllDelegate(remoteTestExecutor.RunAll);
			runAllDelegate.BeginInvoke(RunFinishCallBack, null);
		}

		private delegate void RunSelectionDelegate(TestInterface.TreeNode testTreeNode);

		private void buttonRunSelection_Click(object sender, EventArgs e)
		{
			if (null == testTreeView.SelectedNode || !IsTestExecutorRunning())
			{
				return;
			}

			ResetTestView();
			ButtonsEnabled = false;

			TestInterface.TreeNode testTreeNode = (TestInterface.TreeNode) testTreeView.SelectedNode.Tag;
			progressBar.MaxValue = remoteTestExecutorListener.ComputeNumberOfTests(testTreeNode);
			RunSelectionDelegate runSelectionDelegate = new RunSelectionDelegate(remoteTestExecutor.Run);
			runSelectionDelegate.BeginInvoke(testTreeNode, RunFinishCallBack, null);
		}

		private bool IsTestExecutorRunning()
		{
			if (!remoteTestExecutor.IsRunning())
			{
				Forms.MessageBox.Show("No process running");
				return false;
			}

			return true;
		}

		private void OnProcessTerminatedBecauseDetached()
		{
			BeginInvoke((Forms.MethodInvoker)delegate()
			{
				buttonLoadUnit.Enabled = true;
				buttonLoadSmoke.Enabled = true;
				buttonLoadAnimation.Enabled = true;
			});

			remoteTestExecutor.OnStop -= OnProcessTerminatedBecauseDetached;
			remoteTestExecutor.OnStop += OnProcessTerminated;
		}

		private void buttonDetach_Click(object sender, EventArgs e)
		{
			ButtonsEnabled = false;
			testTreeView.Nodes.Clear();
			if (animationTestConsole != null)
			{
				animationTestConsole.Close();
				animationTestConsole = null;
			}
			remoteTestExecutor.OnStop -= OnProcessTerminated;
			remoteTestExecutor.OnStop += OnProcessTerminatedBecauseDetached;
			remoteTestExecutor.Stop();
		}

		class AnimationLoader
		{
			private string suiteFileName;
			private RemoteTestExecutor remoteTestExecutor;

			public AnimationLoader(RemoteTestExecutor remoteTestExecutor)
			{
				this.remoteTestExecutor = remoteTestExecutor;
			}

			public bool TryLoadSuite()
			{
				string cachedCurrentDirectory = Environment.CurrentDirectory;

				Forms.OpenFileDialog openFileDialog = new Forms.OpenFileDialog();
				Forms.DialogResult dialogResult = openFileDialog.ShowDialog();

				Environment.CurrentDirectory = cachedCurrentDirectory;

				if (dialogResult != Forms.DialogResult.OK)
				{
					return false;
				}

				suiteFileName = openFileDialog.FileName;

				try
				{
					new AnimationTesting.Suite().Load(new System.IO.StreamReader(suiteFileName));
				}
				catch (Exception exception)
				{
					Forms.MessageBox.Show(exception.Message, exception.GetType().ToString(), Forms.MessageBoxButtons.OK, Forms.MessageBoxIcon.Error);
					return false;
				}
				return true;
			}

			public void LoadEngine()
			{
				remoteTestExecutor.LoadAnimationEngine(suiteFileName);
				remoteTestExecutor.OnStart -= LoadEngine;
			}
		}

		private void buttonLoadAnimation_Click(object sender, EventArgs e)
		{
			AnimationLoader animationLoader = new AnimationLoader(remoteTestExecutor);
			if (!animationLoader.TryLoadSuite())
			{
				return;
			}

			LoadTestOnProcessStart(animationLoader.LoadEngine);
		}

		public void ShowAnimationTestConsole()
		{
			buttonRunSelection.Enabled = false;
			animationTestConsole = new AnimationTestConsole();
			animationTestConsole.Show();
		}

		public AnimationTestConsole AnimationTestConsole
		{
			get { return animationTestConsole; }
		}

		private void buttontXMLGenerator_Click(object sender, EventArgs e)
		{
			AnimationTestXMLGeneratorForm form = new AnimationTestXMLGeneratorForm();
			form.ShowDialog();
		}
	}
}
