//***************************************************************************************
//
// File supervisor: Softimage Rendering & Pipeline team
//
// (c) Copyright 2001-2005 Avid Technology, Inc. . All rights reserved.
//
//***************************************************************************************

/**************************************************************************************
THIS CODE IS PUBLISHED AS A SAMPLE ONLY AND IS PROVIDED "AS IS".
IN NO EVENT SHALL SOFTIMAGE, AVID TECHNOLOGY, INC. AND/OR THEIR RESPECTIVE
SUPPLIERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
CONNECTION WITH THE USE OR PERFORMANCE OF THIS CODE .

COPYRIGHT NOTICE. Copyright  1999-2005 Avid Technology Inc. . All rights reserved. 

SOFTIMAGE is a registered trademark of Avid Technology Inc. or its subsidiaries 
or divisions. Windows NT is a registered trademark of Microsoft Corp. All other
trademarks contained herein are the property of their respective owners. 
****************************************************************************************/
/*! \file FTKUtil.cpp
/*!
	implementation file for the FTK utility functions
*/

#include "stdafx.h"
#include "FTKUtil.h"
#include <xsi_fcurve.h>
#include <xsi_fcurvekey.h> 
#include "Modules/cnv_plottedanimation.h"
#include <SceneInfo.h> 

CSLVariantParameter		*FTKUSetShaderParameterValue(CSLXSIShader *in_pShader, char *in_pName, float in_fValue)
{
	CSLVariantParameter *l_pParameter = in_pShader->AddParameter();
	SI_TinyVariant	l_Value;

	l_pParameter->SetName(in_pName);
	l_Value.variantType = SI_VT_FLOAT;
	l_Value.fVal = in_fValue;
	l_pParameter->SetValue(&l_Value);

	return l_pParameter;
}


CSLVariantParameter		*FTKUSetShaderParameterValue(CSLXSIShader *in_pShader, char *in_pName, LONG in_lValue)
{
	CSLVariantParameter *l_pParameter = in_pShader->AddParameter();
	SI_TinyVariant	l_Value;

	l_pParameter->SetName(in_pName);
	l_Value.variantType = SI_VT_LONG;
	l_Value.lVal = in_lValue;
	l_pParameter->SetValue(&l_Value);

	return l_pParameter;
}

CSLVariantParameter		*FTKUSetShaderParameterValue(CSLXSIShader *in_pShader, char *in_pName, bool in_bValue)
{
	CSLVariantParameter *l_pParameter = in_pShader->AddParameter();
	SI_TinyVariant	l_Value;

	l_pParameter->SetName(in_pName);
	l_Value.variantType = SI_VT_BOOL;
	l_Value.boolVal = in_bValue ? 1 : 0;
	l_pParameter->SetValue(&l_Value);

	return l_pParameter;
}

CSLVariantParameter		*FTKUSetShaderParameterValue(CSLXSIShader *in_pShader, char *in_pName, char *in_pValue)
{
	CSLVariantParameter *l_pParameter = in_pShader->AddParameter();
	SI_TinyVariant	l_Value;

	l_pParameter->SetName(in_pName);
	l_Value.variantType = SI_VT_PCHAR;
	l_Value.p_cVal = in_pValue;
	l_pParameter->SetValue(&l_Value);

	return l_pParameter;
}


CSLVariantParameter		*FTKUSetParameterValue(CSLVariantParameter *in_pParameter, float in_fValue)
{
	SI_TinyVariant	l_Value;
	l_Value.variantType = SI_VT_FLOAT;
	l_Value.fVal = in_fValue;
	in_pParameter->SetValue(&l_Value);

	return in_pParameter;
}

CSLVariantParameter		*FTKUSetParameterValue(CSLVariantParameter *in_pParameter, LONG in_lValue)
{
	SI_TinyVariant	l_Value;
	l_Value.variantType = SI_VT_LONG;
	l_Value.lVal = in_lValue;
	in_pParameter->SetValue(&l_Value);

	return in_pParameter;
}

CSLVariantParameter		*FTKUSetParameterValue(CSLVariantParameter *in_pParameter, bool in_bValue)
{
	SI_TinyVariant	l_Value;
	l_Value.boolVal = in_bValue ? 1 : 0;
	in_pParameter->SetValue(&l_Value);

	return in_pParameter;
}

CSLVariantParameter		*FTKUSetParameterValue(CSLVariantParameter *in_pParameter, char *in_pValue)
{
	SI_TinyVariant	l_Value;
	l_Value.variantType = SI_VT_PCHAR;
	l_Value.p_cVal = in_pValue;
	in_pParameter->SetValue(&l_Value);

	return in_pParameter;
}

CSLShaderConnectionPoint 	*FTKUConnectShader(CSLXSIShader *in_pSource, CSLXSIShader *in_pDestination, char *in_pConnection)
{
	// allocate 2 empty params
	CdotXSIParam *l_pNewParam = new CdotXSIParam("", SI_VT_PCHAR);
	CdotXSIParam *l_pNewParam2 = new CdotXSIParam("", SI_VT_PCHAR);

	// add the params to the param list

	// insert right after the last parameter
	in_pDestination->Template()->Params().Add( l_pNewParam, -1 );
	in_pDestination->Template()->Params().Add( l_pNewParam2, -1 );
	SI_TinyVariant l_Value;
	l_Value.variantType = SI_VT_PCHAR;
	l_Value.p_cVal = "SHADER";
	in_pDestination->Template()->Params().Item(in_pDestination->Template()->Params().GetCount()-1)->SetValue(l_Value);
	
	// allocate a new variant parameter
	SI_Int l_nIndex = (2 * in_pDestination->GetConnectionPointCount()) + (in_pDestination->GetParameterCount()) + 4;
	CSLShaderConnectionPoint *l_pNew = 0;
	_SI_NEW(l_pNew, CSLShaderConnectionPoint(in_pDestination->Template(), l_nIndex ) );

	l_pNew->SetShader(in_pSource);
	l_pNew->SetName(in_pConnection);
	// add it to the list of connection points of the material
	return in_pDestination->ConnectConnectionPoint(l_pNew);
}

CSLShaderConnectionPoint 	*FTKUConnectImage(CSLImage *in_pSource, CSLXSIShader *in_pDestination, char *in_pConnection)
{
	// allocate 2 empty params
	CdotXSIParam *l_pNewParam = new CdotXSIParam("", SI_VT_PCHAR);
	CdotXSIParam *l_pNewParam2 = new CdotXSIParam("", SI_VT_PCHAR);

	// add the params to the param list

	// insert right after the last parameter
	in_pDestination->Template()->Params().Add( l_pNewParam, -1 );
	in_pDestination->Template()->Params().Add( l_pNewParam2, -1 );
	
	SI_TinyVariant l_Value;
	l_Value.variantType = SI_VT_PCHAR;
	l_Value.p_cVal = "IMAGE";
	in_pDestination->Template()->Params().Item(in_pDestination->Template()->Params().GetCount()-1)->SetValue(l_Value);

	// allocate a new variant parameter
	SI_Int l_nIndex = (2 * in_pDestination->GetConnectionPointCount()) + (in_pDestination->GetParameterCount()) + 4;
	CSLShaderConnectionPoint *l_pNew = 0;
	_SI_NEW(l_pNew, CSLShaderConnectionPoint(in_pDestination->Template(), l_nIndex ) );

	l_pNew->SetImage(in_pSource->Name().GetText());
	l_pNew->SetName(in_pConnection);
	// add it to the list of connection points of the material
	return in_pDestination->ConnectConnectionPoint(l_pNew);

}

CSLConnectionPoint 	*FTKUConnectShader(CSLXSIShader *in_pSource, CSLXSIMaterial *in_pDestination, char *in_pConnection)
{	
	CSLConnectionPoint *l_pConnection;
	l_pConnection = in_pDestination->AddConnectionPoint();
	l_pConnection->SetName(in_pConnection);
	l_pConnection->SetShader(in_pSource);
	return l_pConnection;
}

void	FTKUGetUsedTexCoord(CSLModel *in_pModel, CSLXSIMaterial *in_pMaterial, int *out_pNbTexCoord, int **out_pTexCoordIndices, bool **out_pTexCoordRepeated)
{
	int loop, count = in_pMaterial->GetShaderCount();


	int *l_pTexCoordIndices = NULL;
	bool *l_pTexCoordRepeated = NULL;
	int l_NbTexCoord = 0;

	for(loop = 0; loop < count ; loop++)
	{
		CSLXSIShader *l_pShader = in_pMaterial->GetShaderList()[loop];
		int loop2, count2 = l_pShader->GetInstanceDataCount();
		bool l_bFound = FALSE;

		for(loop2 = 0; loop2 < count2; loop2++)
		{
			CSLShaderInstanceData *l_pInstanceData = l_pShader->GetInstanceDataList()[loop2];
			if(l_pInstanceData->GetReference() == in_pModel)
			{
				int loop3, count3 = l_pInstanceData->GetParameterCount();
				for(loop3 = 0; loop3 < count3; loop3++)
				{
					CSLVariantParameter *l_pParameter = l_pInstanceData->GetParameterList()[loop3];
					if(strcmp(l_pParameter->GetName(), "tspace_id") == 0)
					{
						char *l_pTextureProjectionName = l_pParameter->GetValue()->p_cVal;
						CSLShape_35 *l_pShape = (CSLShape_35 *) ((CSLMesh *) in_pModel->Primitive())->Shape();
						int loop4, count4 = l_pShape->GetUVCoordArrayCount();
						for(loop4 = 0; loop4 < count4; loop4++)
						{
							CSLUVCoordArray *l_pTexCoord = l_pShape->UVCoordArrays()[loop4];
							if(strcmp(l_pTexCoord->GetTextureProjection(), l_pTextureProjectionName) == 0)
							{
								l_NbTexCoord ++;
								l_pTexCoordIndices = (int *) realloc(l_pTexCoordIndices, sizeof(int) * l_NbTexCoord);
								l_pTexCoordIndices[l_NbTexCoord-1] = loop4;
								l_bFound = TRUE;
							}
						}
					}
				}
			}
		}
	
		if(!l_bFound)
		{
			int loop3, count3 = l_pShader->GetParameterCount();
			for(loop3 = 0; loop3 < count3; loop3++)
			{
				CSLVariantParameter *l_pParameter = l_pShader->GetParameterList()[loop3];
				if(strcmp(l_pParameter->GetName(), "tspace_id") == 0)
				{
					char *l_pTextureProjectionName = l_pParameter->GetValue()->p_cVal;
					CSLShape_35 *l_pShape = (CSLShape_35 *) ((CSLMesh *) in_pModel->Primitive())->Shape();
					int loop4, count4 = l_pShape->GetUVCoordArrayCount();
					for(loop4 = 0; loop4 < count4; loop4++)
					{
						CSLUVCoordArray *l_pTexCoord = l_pShape->UVCoordArrays()[loop4];
						if(strcmp(l_pTexCoord->GetTextureProjection(), l_pTextureProjectionName) == 0)
						{
							l_NbTexCoord ++;
							l_pTexCoordIndices = (int *) realloc(l_pTexCoordIndices, sizeof(int) * l_NbTexCoord);
							l_pTexCoordIndices[l_NbTexCoord-1] = loop4;
						}
					}
				}
			}
		}
	}

	// compute the repeated flags
	if(l_NbTexCoord)
	{
		l_pTexCoordRepeated = (bool *) calloc(sizeof(bool), l_NbTexCoord);
		for(loop = 0; loop < l_NbTexCoord; loop++)
		{
			int loop2; 
			
			if(loop > 1)
			{
				for(loop2 = 0; loop2 < (loop - 1); loop2++)
				{
					if(l_pTexCoordIndices[loop2] == l_pTexCoordIndices[loop])
					{
						l_pTexCoordRepeated[loop] = TRUE;
						break;
					}
				}
			}
		}
	}


	*out_pTexCoordIndices = l_pTexCoordIndices;
	*out_pTexCoordRepeated = l_pTexCoordRepeated;
	*out_pNbTexCoord = l_NbTexCoord;

}

void	FTKUSetShaderTextureSpaceID(CSLModel *in_pModel, CSLXSIMaterial *in_pMaterial, char *in_pTextureProjectionName, int in_nShaderIndex)
{
	int loop, count = in_pMaterial->GetShaderCount(), index = 0;
	for(loop = 0; loop < count; loop++)
	{
		CSLXSIShader *l_pShader = in_pMaterial->GetShaderList()[loop];

		int loop2, count2 = l_pShader->GetParameterCount();
		for(loop2 = 0; loop2 < count2; loop2++)
		{
			CSLVariantParameter *l_pParameter = l_pShader->GetParameterList()[loop2];
			if (strcmp(l_pParameter->GetName(), "tspace_id") == 0)
			{
				if (index == in_nShaderIndex)
				{
					CSLShaderInstanceData *l_pInstanceData = l_pShader->AddInstanceData();
					l_pInstanceData->SetReference(in_pModel);
					CSLVariantParameter *l_pInstanceDataParameter = l_pInstanceData->AddParameter();
					l_pInstanceDataParameter->SetName("tspace_id");
					SI_TinyVariant l_Value;
					l_Value.variantType = SI_VT_PCHAR;
					l_Value.p_cVal = &(in_pTextureProjectionName[1]);
					l_pInstanceDataParameter->SetValue(&l_Value);


					SI_TinyVariant *l_pValue = l_pParameter->GetValue();
					if(strcmp(l_pValue->p_cVal, "") == 0)
					{	
						l_pParameter->SetValue(&l_Value);
					}
				}

				index ++;
			}
		}
	}
}


CSLShaderConnectionPoint	*FTKUGetConnection(CSLXSIShader *in_pShader, char *in_pName)
{
	int loop, count = in_pShader->GetConnectionPointCount();
	for(loop = 0; loop < count; loop++)
	{
		if(strcmp(in_pName, in_pShader->GetConnectionPointList()[loop]->GetName()) == 0)
		{	
			return in_pShader->GetConnectionPointList()[loop];
		}
	}

	return NULL;
}

CSLXSIShader* FTKUGetPreviousRTS(CSLXSIShader* in_pShader)
{
	int loop;

	for (loop = 0; loop < in_pShader->GetConnectionPointCount(); loop++)
	{
		if (strcmp(in_pShader->GetConnectionPointList()[loop]->GetName(), "previous") == 0)
			break;
	}

	if (loop >= in_pShader->GetConnectionPointCount())
		return NULL;
	else
		return in_pShader->GetConnectionPointList()[loop]->GetShader();
}

void FTKUParameterAnimationFromXSI
	(
		CSLTemplate* in_pFTKTemplate,
		const CRefArray &in_XSIParameterList,
		CSLFCurve::EFCurveType in_eFCurveType,
		CAnimationPlotter *in_pAnimationPlotter,
		const CString &in_ParamName
	)
{
	Parameter l_Parameter = in_XSIParameterList.GetItem(in_ParamName);

	if(l_Parameter.IsValid())
		FTKUParameterAnimationFromXSI( in_pFTKTemplate, in_eFCurveType,	l_Parameter, in_pAnimationPlotter, in_ParamName );
}

void FTKUParameterAnimationFromXSI
	(
		CSLTemplate* in_pFTKTemplate,
		CSLFCurve::EFCurveType in_eFCurveType,
		const Parameter &in_Parameter,
		CAnimationPlotter *in_pAnimationPlotter,
		const CString &in_ParamName
	)
{
	CRef l_CRef = in_Parameter.GetSource();
	CString l_ParamName;

	// check if this parameter is animated
	if (l_CRef.IsValid())
	{
		FCurve l_FCurve = l_CRef;
		
		// if the parameter name isn't provided, extract it
		if (in_ParamName.IsEmpty())
		{
			l_CRef = in_Parameter.GetParent();

			siClassID l_ClassID = l_CRef.GetClassID();

			CString l_Tmp = in_Parameter.GetFullName();
			const char * l_pTmp = l_Tmp.GetAsciiString();

			l_Tmp = in_Parameter.GetName();
			l_pTmp = l_Tmp.GetAsciiString();

			if (l_CRef.IsA(siCompoundParameterID) || l_CRef.IsA(siParameterID))
			{
				Parameter l_Parent = l_CRef;

				l_ParamName = l_Parent.GetScriptName();
				l_ParamName += L".";
				l_ParamName += in_Parameter.GetScriptName();
			}
			else
			{
				l_ParamName = in_Parameter.GetScriptName();
			}
		}
		else
		{
			l_ParamName = in_ParamName;
		}

		// if the animation plotter is not NULL then we want to
		// register this fcurve to be plotted later
		if(in_pAnimationPlotter != NULL)
		{
			CAnimationPlotterParameterEntry *l_pEntry = new CAnimationPlotterParameterEntry();

			in_pAnimationPlotter->GetParameterTable().Add(l_pEntry);
			l_pEntry->m_XSIParameter = in_Parameter;
			l_pEntry->m_pFTKFCurveOwner = in_pFTKTemplate;
			l_pEntry->m_FTKFCurveName = l_ParamName;
			l_pEntry->m_FTKFCurveType = in_eFCurveType;

			return;
		}

		if(l_FCurve.IsA(siFCurveID))
		{
			FTKUFCurveFromXSI(in_pFTKTemplate, in_eFCurveType, l_FCurve, NULL, l_ParamName);
		}
	}
}

void FTKUProxyParameterAnimationFromXSI
	(
		CSLTemplate* in_pFTKTemplate,
		CSLFCurve::EFCurveType in_eFCurveType,
		const Parameter &in_ProxyParameter,
		const Parameter &in_Parameter,
		CAnimationPlotter *in_pAnimationPlotter,
		const CString &in_ParamName
	)
{
	CRef l_CRef = in_ProxyParameter.GetSource();
	CString l_ParamName;

	// check if this parameter is animated
	if (l_CRef.IsValid())
	{
		FCurve l_FCurve = l_CRef;
		
		// if the parameter name isn't provided, extract it
		if (in_ParamName.IsEmpty())
		{
			l_CRef = in_Parameter.GetParent();

			siClassID l_ClassID = l_CRef.GetClassID();

			CString l_Tmp = in_Parameter.GetFullName();
			const char * l_pTmp = l_Tmp.GetAsciiString();

			l_Tmp = in_Parameter.GetName();
			l_pTmp = l_Tmp.GetAsciiString();

			if (l_CRef.IsA(siCompoundParameterID) || l_CRef.IsA(siParameterID))
			{
				Parameter l_Parent = l_CRef;

				l_ParamName = l_Parent.GetScriptName();
				l_ParamName += L".";
				l_ParamName += in_Parameter.GetScriptName();
			}
			else
			{
				l_ParamName = in_Parameter.GetScriptName();
			}
		}
		else
		{
			l_ParamName = in_ParamName;
		}

		// if the animation plotter is not NULL then we want to
		// register this fcurve to be plotted later
		if(in_pAnimationPlotter != NULL)
		{
			CAnimationPlotterParameterEntry *l_pEntry = new CAnimationPlotterParameterEntry();

			l_pEntry->m_XSIParameter = in_Parameter;
			l_pEntry->m_pFTKFCurveOwner = in_pFTKTemplate;
			l_pEntry->m_FTKFCurveName = l_ParamName;
			l_pEntry->m_FTKFCurveType = in_eFCurveType;

			in_pAnimationPlotter->Add(l_pEntry);
			

			return;
		}
	}
}

void FTKUFCurveFromXSI
	(
		CSLTemplate* in_pFTKTemplate,
		CSLFCurve::EFCurveType in_eFCurveType,
		const FCurve &in_FCurve,
		CAnimationPlotter *in_pAnimationPlotter,
		const CString &in_ParamName
	)
{

 	LONG l_ulNumKeys = in_FCurve.GetNumKeys();
	float framerate = 1.0f;

	if(l_ulNumKeys <= 0)
		return;

	CTime::Unit l_Format;
	if(in_pFTKTemplate->Scene()->SceneInfo()->GetTimingType() == CSLSceneInfo::SI_FRAMES)
	{
		l_Format = CTime::Frames;
	}
	else
	{
		CTime l_Time;
		l_Format = CTime::Seconds;
		framerate = (float)l_Time.GetFrameRate();
	}

	// create and fill the FTK FCurve depending on the type of the XSI FCurve
	switch (in_FCurve.GetInterpolation())
	{
		case siConstantInterpolation:
		{
			CSLFCurve* l_pFTKFCurve = NULL;

			if (in_eFCurveType == CSLFCurve::SI_PARAMETER)
			{
				l_pFTKFCurve = in_pFTKTemplate->CreateParameterFCurve( (char*)in_ParamName.GetAsciiString(), CSLFCurve::SI_CONSTANT );
			}
			else
			{
				l_pFTKFCurve = in_pFTKTemplate->CreateFCurve( in_eFCurveType, CSLFCurve::SI_CONSTANT );
			}

			if (l_pFTKFCurve)
			{
				CSLFCurve::CSLConstantKeyArray* l_pConstantKeyArray = l_pFTKFCurve->GetConstantKeyList();
				(*l_pConstantKeyArray).Reserve(l_ulNumKeys);

				for (LONG loop = 0; loop < l_ulNumKeys; loop++)
				{
					FCurveKey in_FCurveKey = in_FCurve.GetKeyAtIndex(loop);

					(*l_pConstantKeyArray)[loop].m_fTime = (float)in_FCurveKey.GetTime().GetTime(l_Format);
					(*l_pConstantKeyArray)[loop].m_fValue = (float)in_FCurveKey.GetValue();
				}
			}
		} break;

		case siCubicInterpolation:
		{
			CSLFCurve* l_pFTKFCurve = NULL;

			if (in_eFCurveType == CSLFCurve::SI_PARAMETER)
			{
				l_pFTKFCurve = in_pFTKTemplate->CreateParameterFCurve( (char*)in_ParamName.GetAsciiString(), CSLFCurve::SI_CUBIC );
			}
			else
			{
				l_pFTKFCurve = in_pFTKTemplate->CreateFCurve( in_eFCurveType, CSLFCurve::SI_CUBIC );
			}

			if (l_pFTKFCurve)
			{
				CSLFCurve::CSLCubicKeyArray* l_pCubicKeyArray = l_pFTKFCurve->GetCubicKeyList();
				(*l_pCubicKeyArray).Reserve(l_ulNumKeys);

				for (LONG loop = 0; loop < l_ulNumKeys; loop++)
				{
					FCurveKey in_FCurveKey = in_FCurve.GetKeyAtIndex(loop);

					(*l_pCubicKeyArray)[loop].m_fTime = (float)in_FCurveKey.GetTime().GetTime(l_Format);
					(*l_pCubicKeyArray)[loop].m_fValue = (float)in_FCurveKey.GetValue();
					(*l_pCubicKeyArray)[loop].m_fLeftTanX = (float)in_FCurveKey.GetLeftTanX() / framerate;
					(*l_pCubicKeyArray)[loop].m_fLeftTanY = (float)in_FCurveKey.GetLeftTanY();
					(*l_pCubicKeyArray)[loop].m_fRightTanX = (float)in_FCurveKey.GetRightTanX() / framerate;
					(*l_pCubicKeyArray)[loop].m_fRightTanY = (float)in_FCurveKey.GetRightTanY();
				}
			}
		} break;

		case siLinearInterpolation:
		default:
		{
			CSLFCurve* l_pFTKFCurve = NULL;

			if (in_eFCurveType == CSLFCurve::SI_PARAMETER)
			{
				l_pFTKFCurve = in_pFTKTemplate->CreateParameterFCurve( (char*)in_ParamName.GetAsciiString(), CSLFCurve::SI_LINEAR );
			}
			else
			{
				l_pFTKFCurve = in_pFTKTemplate->CreateFCurve( in_eFCurveType, CSLFCurve::SI_LINEAR );
			}

			if (l_pFTKFCurve)
			{
				CSLFCurve::CSLLinearKeyArray* l_pLinearKeyArray = l_pFTKFCurve->GetLinearKeyList();
				(*l_pLinearKeyArray).Reserve(l_ulNumKeys);

				for (LONG loop = 0; loop < l_ulNumKeys; loop++)
				{
					FCurveKey in_FCurveKey = in_FCurve.GetKeyAtIndex(loop);

					(*l_pLinearKeyArray)[loop].m_fTime = (float)in_FCurveKey.GetTime().GetTime(l_Format);
					(*l_pLinearKeyArray)[loop].m_fValue = (float)in_FCurveKey.GetValue();
				}
			}
		} break;
	}
}

void FTKUParameterAnimationToXSI
	(
		CSLFCurve* in_pFTKFCurve,
		const CParameterRefArray &in_XSIParameters
	)
{
	CString l_ParamName;

	// extract the parameter script name from the FCurve type
	switch (in_pFTKFCurve->GetFCurveType())
	{
		case CSLFCurve::SI_SCALING_X:		//Scaling
			l_ParamName = L"sclx";
			break;										 
		case CSLFCurve::SI_SCALING_Y:		//Scaling
			l_ParamName = L"scly";
			break;										 
		case CSLFCurve::SI_SCALING_Z:		//Scaling
			l_ParamName = L"sclz";
			break;										 
		case CSLFCurve::SI_ROTATION_X:		//Rotation
			l_ParamName = L"rotx";
			break;										 
		case CSLFCurve::SI_ROTATION_Y:		//Rotation
			l_ParamName = L"roty";
			break;										 
		case CSLFCurve::SI_ROTATION_Z:		//Rotation
			l_ParamName = L"rotz";
			break;										 
		case CSLFCurve::SI_TRANSLATION_X:	//Translation
			l_ParamName = L"posx";
			break;										 
		case CSLFCurve::SI_TRANSLATION_Y:	//Translation
			l_ParamName = L"posy";
			break;										 
		case CSLFCurve::SI_TRANSLATION_Z:	//Translation
			l_ParamName = L"posz";
			break;				
		/*
		case CSLFCurve::SI_SHPANIM:			//ShapeAnimation
			// TODO
			break;
		*/
		case CSLFCurve::SI_NODEVIS:			//Visibility
			l_ParamName = L"viewvis";
			break;

		case CSLFCurve::SI_COLOR_R:			// Color
			if (!in_pFTKFCurve->ParentModel()->Primitive())
			{
				l_ParamName = L"ambience.red";
			}
			else if ( (in_pFTKFCurve->ParentModel()->Primitive()->Type() == CSLTemplate::SI_DIRECTIONAL_LIGHT)
					|| (in_pFTKFCurve->ParentModel()->Primitive()->Type() == CSLTemplate::SI_SPOT_LIGHT)
					|| (in_pFTKFCurve->ParentModel()->Primitive()->Type() == CSLTemplate::SI_POINT_LIGHT) )
			{
				l_ParamName = L"color.red";
			}
			break;
		case CSLFCurve::SI_COLOR_G:			// Color
			if (!in_pFTKFCurve->ParentModel()->Primitive())
			{
				l_ParamName = L"ambience.green";
			}
			else if ( (in_pFTKFCurve->ParentModel()->Primitive()->Type() == CSLTemplate::SI_DIRECTIONAL_LIGHT)
					|| (in_pFTKFCurve->ParentModel()->Primitive()->Type() == CSLTemplate::SI_SPOT_LIGHT)
					|| (in_pFTKFCurve->ParentModel()->Primitive()->Type() == CSLTemplate::SI_POINT_LIGHT) )
			{
				l_ParamName = L"color.green";
			}
			break;
		case CSLFCurve::SI_COLOR_B:			// Color
			if (!in_pFTKFCurve->ParentModel()->Primitive())
			{
				l_ParamName = L"ambience.blue";
			}
			else if ( (in_pFTKFCurve->ParentModel()->Primitive()->Type() == CSLTemplate::SI_DIRECTIONAL_LIGHT)
					|| (in_pFTKFCurve->ParentModel()->Primitive()->Type() == CSLTemplate::SI_SPOT_LIGHT)
					|| (in_pFTKFCurve->ParentModel()->Primitive()->Type() == CSLTemplate::SI_POINT_LIGHT) )
			{
				l_ParamName = L"color.blue";
			}
			break;
		case CSLFCurve::SI_POSITION_X:		// Camera/Light position
			l_ParamName = L"posx";
			break;
		case CSLFCurve::SI_POSITION_Y:		// Camera position
			l_ParamName = L"posy";
			break;
		case CSLFCurve::SI_POSITION_Z:		// Camera position
			l_ParamName = L"posz";
			break;
		case CSLFCurve::SI_ROLL:			// Camera roll
			l_ParamName = L"roll";
			break;
		case CSLFCurve::SI_FOV:				// Camera fov
			l_ParamName = L"fov";
			break;
		case CSLFCurve::SI_NEAR:			// Camera near plane
			l_ParamName = L"near";
			break;
		case CSLFCurve::SI_FAR:				// Camera far plane
			l_ParamName = L"far";
			break;
		case CSLFCurve::SI_INTEREST_X:		//Camera interest position
			l_ParamName = L"posx";
			break;										 
		case CSLFCurve::SI_INTEREST_Y:		//Camera interest position
			l_ParamName = L"posy";
			break;										 
		case CSLFCurve::SI_INTEREST_Z:		//Camera interest position
			l_ParamName = L"posz";
			break;										 
		case CSLFCurve::SI_CONE:			//Light cone angle
			l_ParamName = L"LightCone";
			break;										 
		case CSLFCurve::SI_SPREAD:			//Light spread
			l_ParamName = L"spread";
			break;										 
		case CSLFCurve::SI_ORIENTATION_X:	//Light orientation
			l_ParamName = L"rotx";
			break;										 
		case CSLFCurve::SI_ORIENTATION_Y:	//Light orientation
			l_ParamName = L"roty";
			break;										 
		case CSLFCurve::SI_ORIENTATION_Z:	//Light orientation
			l_ParamName = L"rotz";
			break;										 
		case CSLFCurve::SI_PARAMETER:		//Named parameter
		{
			const char* l_pTemp = in_pFTKFCurve->GetFCurveTypeAsString();

			l_ParamName.PutAsciiString(in_pFTKFCurve->GetFCurveTypeAsString());
		}	break;										 
		case CSLFCurve::SI_FALLOFF_ACTIVE:
			l_ParamName = L"atten";
			break;				
		case CSLFCurve::SI_FALLOFF_START:
			l_ParamName = L"start";
			break;				
		case CSLFCurve::SI_FALLOFF_END:
			l_ParamName = L"stop";
			break;				
		case CSLFCurve::SI_SHADOWS_ENABLED:
			l_ParamName = L"shadow";
			break;				
		case CSLFCurve::SI_UMBRA:
			l_ParamName = L"factor";
			break;				
		case CSLFCurve::SI_LIGHT_AS_ENERGY:
			l_ParamName = L"use_color";
			break;				
		case CSLFCurve::SI_ENERGY_FACTOR:
			l_ParamName = L"energy_factor";
			break;				
		case CSLFCurve::SI_INTENSITY:
			l_ParamName = L"intensity";
			break;
		case CSLFCurve::SI_CROP_MIN_X:
		{
			l_ParamName = L"Xmin";
		}	break;										 
		case CSLFCurve::SI_CROP_MAX_X:
		{
			l_ParamName = L"Xmax";
		}	break;										 
		case CSLFCurve::SI_CROP_MIN_Y:
		{
			l_ParamName = L"Ymin";
		}	break;										 
		case CSLFCurve::SI_CROP_MAX_Y:
		{
			l_ParamName = L"Ymax";
		}	break;										 
		case CSLFCurve::SI_HUE:
		{
			l_ParamName = L"Hue";
		}	break;										 
		case CSLFCurve::SI_GAIN:
		{
			l_ParamName = L"Gain";
		}	break;										 
		case CSLFCurve::SI_SATURATION:
		{
			l_ParamName = L"Saturation";
		}	break;										 
		case CSLFCurve::SI_BRIGHTNESS:
		{
			l_ParamName = L"Brightness";
		}	break;										 
		case CSLFCurve::SI_BLUR_RADIUS:
		{
			l_ParamName = L"Radius";
		}	break;										 
		case CSLFCurve::SI_BLUR_AMOUNT:
		{
			l_ParamName = L"Amount";
		}	break;										 
		case CSLFCurve::SI_BLUR_ALPHA:
		{
			l_ParamName = L"BlurAlpha";
		}	break;										 
		case CSLFCurve::SI_SCALING_TYPE:
		{
			l_ParamName = L"Type";
		}	break;										 
		case CSLFCurve::SI_SCALE_X:
		{
			l_ParamName = L"ResX";
		}	break;										 
		case CSLFCurve::SI_SCALE_Y:
		{
			l_ParamName = L"ResY";
		}	break;										 
		case CSLFCurve::SI_FLIP_X:
		{
			l_ParamName = L"FlipX";
		}	break;										 
		case CSLFCurve::SI_FLIP_Y:
		{
			l_ParamName = L"FlipY";
		}	break;										 
		case CSLFCurve::SI_RGBA2GRAYSCALE:
		{
			l_ParamName = L"GrayScale";
		}	break;										 
		case CSLFCurve::SI_BITS_PER_CHANNEL:
		{
			l_ParamName = L"SixteenBitsPerChannel";
		}	break;
	};

	FTKUParameterAnimationToXSI( in_pFTKFCurve, in_XSIParameters, l_ParamName );
}
	
void FTKUParameterAnimationToXSI
	(
		CSLFCurve* in_pFTKFCurve,
		const CParameterRefArray &in_XSIParameters,
		CString in_ParamName
	)
{
	if (in_ParamName.IsEmpty())
		return;

	// find the XSI parameter
	Parameter l_XSIParameter;

	// split compound parameter if needed
	wchar_t *l_pParamPrefix = _wcsdup(in_ParamName.GetWideString());
				
	// find the sub parameter if needed
	wchar_t *l_pParamSuffix = si_wcstok(l_pParamPrefix, L".");
	l_pParamSuffix = si_wcstok(NULL, L".");

	l_XSIParameter = in_XSIParameters.GetItem(l_pParamPrefix);

	if(l_XSIParameter.IsValid() && l_pParamSuffix != NULL)
	{
		CRefArray l_CompoundParameterCollection = l_XSIParameter.GetParameters();
		l_XSIParameter = l_CompoundParameterCollection.GetItem(l_pParamSuffix);
	}

	free(l_pParamPrefix);

	// now add the FCurve to the param 
	if (l_XSIParameter.IsValid())
	{
		FCurve l_XSIFCurve;
		l_XSIParameter.AddFCurve(siDefaultFCurve, l_XSIFCurve);

		FTKUFCurveToXSI(in_pFTKFCurve, l_XSIFCurve);
	}
}

void FTKUFCurveToXSI
	(
		CSLBaseFCurve* in_pFTKFCurve,
		FCurve &in_XSIFCurve
	)
{
	// set the keys
	LONG	loop, l_nKeyCount;
	float	framerate = 1.0f;

	CTime::Unit l_Format;
	if(in_pFTKFCurve->Scene()->SceneInfo()->GetTimingType() == CSLSceneInfo::SI_FRAMES)
	{
		l_Format = CTime::Frames;
	}
	else
	{
		CTime l_Time;
		l_Format = CTime::Seconds;
		framerate = (float)l_Time.GetFrameRate();
	}


	l_nKeyCount = in_pFTKFCurve->GetKeyCount();

	CTimeArray l_TimeArray(l_nKeyCount);
	CValueArray l_ValueArray(l_nKeyCount);
	CDoubleArray l_TangentsArray(l_nKeyCount*4);

	switch (in_pFTKFCurve->GetInterpolationType())
	{
		case CSLFCurve::SI_CONSTANT:
		{
			CSLConstantKey* l_pKey = in_pFTKFCurve->GetConstantKeyListPtr();

			for (loop = 0; loop < l_nKeyCount; loop++)
			{
				l_TimeArray[loop].PutTime(l_pKey[loop].m_fTime, l_Format);
				l_ValueArray[loop] = l_pKey[loop].m_fValue;
			}

			in_XSIFCurve.SetKeys(l_TimeArray,l_ValueArray);
			in_XSIFCurve.PutInterpolation(siConstantInterpolation);
		} break;
		case CSLFCurve::SI_LINEAR:
		{
			CSLLinearKey* l_pKey = in_pFTKFCurve->GetLinearKeyListPtr();

			for (loop = 0; loop < l_nKeyCount; loop++)
			{
				l_TimeArray[loop].PutTime(l_pKey[loop].m_fTime, l_Format);
				l_ValueArray[loop] = l_pKey[loop].m_fValue;
			}

			in_XSIFCurve.SetKeys(l_TimeArray,l_ValueArray);
			in_XSIFCurve.PutInterpolation(siLinearInterpolation);
		} break;
		case CSLBaseFCurve::SI_CUBIC:
		{
			CSLCubicKey* l_pKey = in_pFTKFCurve->GetCubicKeyListPtr();

			for (loop = 0; loop < l_nKeyCount; loop++)
			{
				l_TimeArray[loop].PutTime(l_pKey[loop].m_fTime, l_Format);
				l_ValueArray[loop] = l_pKey[loop].m_fValue;
				l_TangentsArray[loop*4] = l_pKey[loop].m_fLeftTanX* framerate;
				l_TangentsArray[loop*4+1] = l_pKey[loop].m_fLeftTanY;
				l_TangentsArray[loop*4+2] = l_pKey[loop].m_fRightTanX* framerate;
				l_TangentsArray[loop*4+3] = l_pKey[loop].m_fRightTanY;
			}

			in_XSIFCurve.SetKeys(l_TimeArray,l_ValueArray);
			in_XSIFCurve.SetKeyTangents( l_TangentsArray );
			in_XSIFCurve.PutInterpolation(siCubicInterpolation);
		} break;
		case CSLFCurve::SI_HERMITE:
		{
			CSLHermiteKey* l_pKey = in_pFTKFCurve->GetHermiteKeyListPtr();

			for (loop = 0; loop < l_nKeyCount; loop++)
			{
				//
				// Third to previous key
				//
				float l_fThirdToPrevious = 0;
				if ( loop > 0 )
				{
					float diff = l_pKey[loop].m_fTime - l_pKey[loop-1].m_fTime;
					l_fThirdToPrevious =diff / 3.0f;
				}

				//
				// Third to next key
				//
				float l_fThirdToNext = 0;
				if ( loop < l_nKeyCount-1 )
				{
					float diff = l_pKey[loop+1].m_fTime - l_pKey[loop].m_fTime;
					l_fThirdToNext =diff / 3.0f;
				}

				l_TimeArray[loop].PutTime(l_pKey[loop].m_fTime, l_Format);
				l_ValueArray[loop] = l_pKey[loop].m_fValue;
				l_TangentsArray[loop*4] = -l_fThirdToPrevious;
				l_TangentsArray[loop*4+1] = l_pKey[loop].m_fInTangent;
				l_TangentsArray[loop*4+2] = l_fThirdToNext;
				l_TangentsArray[loop*4+3] = l_pKey[loop].m_fOutTangent;

			}

			in_XSIFCurve.SetKeys(l_TimeArray,l_ValueArray);
			in_XSIFCurve.PutInterpolation(siCubicInterpolation);
			in_XSIFCurve.SetKeyTangents( l_TangentsArray );
			
		} break;
		case CSLFCurve::SI_BEZIER:
		{
			CSLBezierKey* l_pKey = in_pFTKFCurve->GetBezierKeyListPtr();

			for (loop = 0; loop < l_nKeyCount; loop++)
			{
				l_TimeArray[loop].PutTime(l_pKey[loop].m_fTime, l_Format);
				l_ValueArray[loop] = l_pKey[loop].m_fValue;
				l_TangentsArray[loop*4] =  l_pKey[loop].m_fInTangentX* framerate;
				l_TangentsArray[loop*4+1] = l_pKey[loop].m_fInTangentY;
				l_TangentsArray[loop*4+2] = l_pKey[loop].m_fOutTangentX* framerate;
				l_TangentsArray[loop*4+3] = l_pKey[loop].m_fOutTangentY;
			}

			in_XSIFCurve.SetKeys(l_TimeArray,l_ValueArray);
			in_XSIFCurve.PutInterpolation(siCubicInterpolation);
			in_XSIFCurve.SetKeyTangents( l_TangentsArray );
		} break;
	} // end switch
}

bool IsZUp(CSLCoordinateSystem *in_pCoordinateSystem)
{
	return (in_pCoordinateSystem
		&& (in_pCoordinateSystem->GetHandRotation() == CSLCoordinateSystem::SI_RIGHT_HANDED)
		&& (in_pCoordinateSystem->GetUAxisOrientation() == CSLCoordinateSystem::SI_RIGHT_U)
		&& (in_pCoordinateSystem->GetVAxisOrientation() == CSLCoordinateSystem::SI_UP_V)
		&& (in_pCoordinateSystem->GetXAxisOrientation() == CSLCoordinateSystem::SI_RIGHT)
		&& (in_pCoordinateSystem->GetYAxisOrientation() == CSLCoordinateSystem::SI_OUT)
		&& (in_pCoordinateSystem->GetZAxisOrientation() == CSLCoordinateSystem::SI_UP));
}

void	FTKUPathAbsoluteToRelative(char* in_pAbsolutePath, char* in_pFileName, CSIBCString &out_RelativePath)
{
	CSIBCString l_AbsolutePathRoot("");
	CSIBCString l_FileNameRoot("");

	// check for UNC roots
	if(strncmp(in_pAbsolutePath, "\\\\", 2) == 0)
	{
		l_AbsolutePathRoot.SetText(in_pAbsolutePath);
		char *l_pTerminator = strchr(l_AbsolutePathRoot.GetText() + 2, '\\');
		if(l_pTerminator)
			*l_pTerminator = '\0';
	}

	if(strncmp(in_pFileName, "\\\\", 2) == 0)
	{
		l_FileNameRoot.SetText(in_pFileName);
		char *l_pTerminator = strchr(l_FileNameRoot.GetText() + 2, '\\');
		if(l_pTerminator)
			*l_pTerminator = '\0';
	}

	// different UNC roots or drives
	if(	((l_AbsolutePathRoot.GetLength() == 0) ^ (l_FileNameRoot.GetLength() == 0)) ^
		(_stricmp(l_AbsolutePathRoot.GetText(), l_FileNameRoot.GetText())) ^

#ifdef UNIX
		(strncasecmp(in_pAbsolutePath, in_pFileName, 2)))
#else
		(_strnicmp(in_pAbsolutePath, in_pFileName, 2)))
#endif

	{
		out_RelativePath.SetText(in_pAbsolutePath);
		return;
	}

	CSIBCString l_AbsolutePath(in_pAbsolutePath);
	CSIBCString l_ScenePath(in_pFileName);

	// convert all \ to /

	int loop, count = l_AbsolutePath.GetLength();
	for(loop = 0; loop < count; loop++)
	{
		if(l_AbsolutePath.GetText()[loop] == '\\')
			l_AbsolutePath.GetText()[loop] = '/';
	}

	count = l_ScenePath.GetLength();
	for(loop = 0; loop < count; loop++)
	{
		if(l_ScenePath.GetText()[loop] == '\\')
			l_ScenePath.GetText()[loop] = '/';
	}

	CSIBCArray<char *> l_TokensForAbsolutePath;
	CSIBCArray<char *> l_TokensForScenePath;

	char *l_pToken = strtok(l_AbsolutePath.GetText(), "/");
	while(l_pToken)
	{
		l_TokensForAbsolutePath.Add(l_pToken);
		l_pToken = strtok(NULL, "/");
	}

	l_pToken = strtok(l_ScenePath.GetText(), "/");
	while(l_pToken)
	{
		l_TokensForScenePath.Add(l_pToken);
		l_pToken = strtok(NULL, "/");
	}

	int skip = 0;

	// now skip the common leading tokens
	while(_stricmp(l_TokensForAbsolutePath[skip], l_TokensForScenePath[skip]) == 0)
	{
		skip ++;
	}

	out_RelativePath.Clear();

	// add as many .. as we have tokens left for the scene path
	for(loop = skip; loop < l_TokensForScenePath.GetUsed()-1; loop++)
	{
		out_RelativePath.Concat("../");
	}

	// finish with the rest of the absolute path
	for(loop = skip; loop < l_TokensForAbsolutePath.GetUsed(); loop++)
	{
		out_RelativePath.Concat(l_TokensForAbsolutePath[loop]);
		if(loop < (l_TokensForAbsolutePath.GetUsed()-1))
			out_RelativePath.Concat("/");
	}

	// convert all / to \

#ifndef UNIX
	count = out_RelativePath.GetLength();
	for(loop = 0; loop < count; loop++)
	{
		if(out_RelativePath.GetText()[loop] == '/')
			out_RelativePath.GetText()[loop] = '\\';
	}
#endif

}

void	FTKUPathRelativeToAbsolute(char* in_pRelativePath, char* in_pFileName, CSIBCString &out_AbsolutePath)
{
	// absolute path already, return

#ifdef UNIX
	if(in_pRelativePath[0] == '/')
#else
	if((strncmp(in_pRelativePath, "\\\\", 2) == 0) || (in_pRelativePath[1] == ':'))
#endif
	{
		out_AbsolutePath.SetText(in_pRelativePath);
		return;
	}

	CSIBCString l_RelativePath(in_pRelativePath);
	CSIBCString l_ScenePath(in_pFileName);

	// convert all \ to /

	int loop, count = l_RelativePath.GetLength();
	for(loop = 0; loop < count; loop++)
	{
		if(l_RelativePath.GetText()[loop] == '\\')
			l_RelativePath.GetText()[loop] = '/';
	}

	count = l_ScenePath.GetLength();
	for(loop = 0; loop < count; loop++)
	{
		if(l_ScenePath.GetText()[loop] == '\\')
			l_ScenePath.GetText()[loop] = '/';
	}

	CSIBCArray<char *> l_TokensForRelativePath;
	CSIBCArray<char *> l_TokensForScenePath;

	char *l_pToken = strtok(l_RelativePath.GetText(), "/");
	while(l_pToken)
	{
		l_TokensForRelativePath.Add(l_pToken);
		l_pToken = strtok(NULL, "/");
	}

	l_pToken = strtok(l_ScenePath.GetText(), "/");
	while(l_pToken)
	{
		l_TokensForScenePath.Add(l_pToken);
		l_pToken = strtok(NULL, "/");
	}

	// scan for all leading ..
	int skip = 0;

	while(strcmp(l_TokensForRelativePath[skip], "..") == 0)
	{
		skip ++;
	}

	// check if we are UNC
	if(strncmp(in_pFileName, "\\\\", 2) == 0)
	{
		out_AbsolutePath.Concat("\\\\");
	}

	// copy all the leading directories
	for(loop = 0; loop < l_TokensForScenePath.GetUsed() - skip - 1; loop++)
	{
		out_AbsolutePath.Concat(l_TokensForScenePath[loop]);
		out_AbsolutePath.Concat("/");
	}

	skip = 0;
	while((strcmp(l_TokensForRelativePath[skip], "..") == 0) || (strcmp(l_TokensForRelativePath[skip], ".") == 0))
	{
		skip ++;
	}

	// finish with the remaining relative path
	for(loop = skip; loop < l_TokensForRelativePath.GetUsed(); loop++)
	{
		out_AbsolutePath.Concat(l_TokensForRelativePath[loop]);
		if(loop < (l_TokensForRelativePath.GetUsed()-1))
			out_AbsolutePath.Concat("/");
	}

	// convert all / to \

#ifndef UNIX
	count = out_AbsolutePath.GetLength();
	for(loop = 0; loop < count; loop++)
	{
		if(out_AbsolutePath.GetText()[loop] == '/')
			out_AbsolutePath.GetText()[loop] = '\\';
	}
#endif

}