//***************************************************************************************
//
// File supervisor: Softimage Rendering & Pipeline team
//
// (c) Copyright 2001-2005 Avid Technology, Inc. . All rights reserved.
//
//***************************************************************************************

/**************************************************************************************
THIS CODE IS PUBLISHED AS A SAMPLE ONLY AND IS PROVIDED "AS IS".
IN NO EVENT SHALL SOFTIMAGE, AVID TECHNOLOGY, INC. AND/OR THEIR RESPECTIVE
SUPPLIERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
CONNECTION WITH THE USE OR PERFORMANCE OF THIS CODE .

COPYRIGHT NOTICE. Copyright  1999-2005 Avid Technology Inc. . All rights reserved. 

SOFTIMAGE is a registered trademark of Avid Technology Inc. or its subsidiaries 
or divisions. Windows NT is a registered trademark of Microsoft Corp. All other
trademarks contained herein are the property of their respective owners. 
****************************************************************************************/
/*! \file cnv_material.cpp
/*!
	implementation file for material classes
*/

#include "stdafx.h"
#include "cnv_material.h"
#include "cnv_custompset.h"
#include "cnv_userdatablob.h"
#include <xsi_library.h>
#include <xsi_material.h>
#include <xsi_scene.h>
#include <xsi_shader.h>
#include <xsi_time.h>
#include <xsi_image.h>
#include <xsi_imageclip2.h>
#include <xsi_x3dobject.h>
#include <xsi_cluster.h>
#include <xsi_uitoolkit.h>

#include <MaterialLibrary.h>
#include <XSIMaterial.h>
#include <XSIMaterialInfo.h>
#include <XSIShader.h>
#include <XSIImageLibrary.h>
#include <ImageClip.h>
#include <ShaderInstanceData.h>
#include <SceneInfo.h>
#include "../FTKUtil.h"
#include "cmdstubs.h"
#include "plugin_stub.h"


/**************************************************************************************
CMaterialLibraryFromXSI
**************************************************************************************/
CMaterialLibraryFromXSI::CMaterialLibraryFromXSI(short in_ExportType)
{
	m_ExportType = in_ExportType;
};

CMaterialLibraryFromXSI::~CMaterialLibraryFromXSI() 
{
	int loop = 0; 
	for(loop = 0; loop < m_InstanceDataRecords.GetUsed(); loop++)
	{
		if(m_InstanceDataRecords[loop] != NULL)
		{
			delete m_InstanceDataRecords[loop];
			m_InstanceDataRecords[loop] = NULL;
		}
	}
};

CStatus CMaterialLibraryFromXSI::Execute(CdotXSIConverter *in_pContext)
{
	CStatus status = CStatus::OK;
	CXSIMaterialFromXSI l_MaterialConverter(CRY_FILE_TYPE_NONE);

	l_MaterialConverter.m_pMatLibCaller = this;

	CSLMaterialLibrary *l_pFTKMaterialLibrary = in_pContext->ftkscene()->GetMaterialLibrary();

	// we merge all the XSI material libraries into a single FTK material library
	CRefArray l_XSILibraries = in_pContext->scene().GetMaterialLibraries();


	// use the name of the first library
	Library l_XSILibrary = l_XSILibraries.GetItem(0);

	CString l_LibName = l_XSILibrary.GetName();

	l_pFTKMaterialLibrary->SetName((char *) l_LibName.GetAsciiString());

	Application app;
	CRefArray l_RemoveMats;
	l_RemoveMats.Clear();

	for (int loop = 0; loop < l_XSILibraries.GetCount(); loop++)
	{
		l_XSILibrary = l_XSILibraries.GetItem(loop);

		if (m_ExportType == EXPORT_MATERIALS)
		{
			bool l_bExport = false;
			CStringArray l_SelectedMaterials = GetSelectedMaterials();
			for (int i = 0; i < l_SelectedMaterials.GetCount(); i++)
			{
				CString l_SelectedMaterial = l_SelectedMaterials[i];
				if (l_SelectedMaterial == l_XSILibrary.GetName())
					l_bExport = true;
			}
			if (!l_bExport)
				continue;
		}

		// here we do a mini traversal of the material library and we call the CXSIMaterialFromXSI callback
		int loop2;

		bool l_bExportLibrary = false;
		CRefArray l_MaterialList = l_XSILibrary.GetItems();
		for(loop2 = 0; loop2 < l_MaterialList.GetCount(); loop2++)
		{
			CRef l_Material = l_MaterialList.GetItem(loop2);
			CSLTemplate *l_pReturnedMaterial = NULL;

			Material l_XSIMaterial(l_Material);

			// CrySpecific: don't add non cryshaded material
			bool l_bCryShader = false;
			CRefArray &l_Shaders = l_XSIMaterial.GetShaders();
			for (int j = 0; j < l_Shaders.GetCount(); j++)
			{
				Shader l_Shader(l_Shaders[j]);
				CString l_ProgID = l_Shader.GetProgID();
				char* l_strProgID = (char*)l_ProgID.GetAsciiString();
				if (strstr(l_strProgID, "CryShader.1"))
				{
					l_bCryShader = true;
				}
			}

			// CrySpecific: don't add unused material library
			CRefArray l_XSIMaterials = l_XSILibrary.GetItems();
			bool l_bRemove = true;
			for (int i = 0; i < l_XSIMaterials.GetCount(); i++)
			{
				Material l_XSIMat(l_XSIMaterials.GetItem(i));

				CRefArray l_UsedBy = l_XSIMat.GetUsedBy();
				for (int j = 0; j < l_UsedBy.GetCount(); j++)
				{
					ProjectItem l_Obj(l_UsedBy.GetItem(j));
					ProjectItem l_CryExportNode = GetCryExportNode(l_Obj);
					if (l_CryExportNode.GetName() != L"")
						l_bRemove = false;
				}
			}

			if ( l_bRemove || !l_bCryShader )
			{
				l_RemoveMats.Add( l_Material );
			}
			else
			{
				_XSI_CALL(l_MaterialConverter.Execute(in_pContext, l_XSILibrary, l_pFTKMaterialLibrary, NULL, &l_Material, &l_pReturnedMaterial), L"Failed to convert material");
				l_bExportLibrary = true;
			}
		}
	}

	for (int loop = 0; loop < l_XSILibraries.GetCount(); loop++)
	{
		l_XSILibrary = l_XSILibraries.GetItem(loop);

		// here we do a mini traversal of the material library and we call the CXSIMaterialFromXSI callback
		int loop2;

		CRefArray l_MaterialList = l_XSILibrary.GetItems();
		for(loop2 = 0; loop2 < l_MaterialList.GetCount(); loop2++)
		{
			CRef l_Material = l_MaterialList.GetItem(loop2);
			CSLTemplate *l_pReturnedMaterial = NULL;

			for (int i = 0; i < l_RemoveMats.GetCount(); i++ )
			{
				if ( l_Material == l_RemoveMats.GetItem(i) )
					l_MaterialList.Remove(l_Material);
			}
		}
	}

	l_RemoveMats.Clear();

	return status;
}

wchar_t *CMaterialLibraryFromXSI::GetClassID(){return L"CMaterialLibraryFromXSI";}

/**************************************************************************************
CMaterialLibraryFromXSI_RemoveUnused
**************************************************************************************/
CMaterialLibraryFromXSI_RemoveUnused::CMaterialLibraryFromXSI_RemoveUnused() {};

CMaterialLibraryFromXSI_RemoveUnused::~CMaterialLibraryFromXSI_RemoveUnused() 
{
};

void PropagateImageUsedFlag(CSLImageLibrary *l_pImageLibrary, CSLXSIShader *l_pShader)
{
	CSLImage *l_pConnectedImage = NULL;
	int loop;
	int l_nCount = l_pShader->GetConnectionPointCount();
	CSLShaderConnectionPoint **l_pConnectionPointArray = l_pShader->GetConnectionPointList();

	for (loop = 0; loop < l_nCount; loop++)
	{
		CSLXSIShader *l_pConnectedShader = l_pConnectionPointArray[loop]->GetShader();

		if (l_pConnectedShader)
		{
			PropagateImageUsedFlag(l_pImageLibrary, l_pConnectedShader);
		}
		else
		{
			l_pConnectedImage = l_pImageLibrary->FindImage(l_pConnectionPointArray[loop]->GetImage());

			if (l_pConnectedImage)
			{
				CSIBCUserData *l_pUserData = l_pConnectedImage->FindUserData("USED");

				if (!l_pUserData)
				{
					l_pConnectedImage->AttachUserData("USED", 0);
				}
			}
		}
	}
}

CStatus CMaterialLibraryFromXSI_RemoveUnused::Execute(CdotXSIConverter *in_pContext)
{
	CStatus status = CStatus::OK;

	CSLMaterialLibrary *l_pFTKMaterialLibrary = in_pContext->ftkscene()->GetMaterialLibrary();

	LONG loop;

	for (loop = 0; loop < l_pFTKMaterialLibrary->GetMaterialCount(); loop++)
	{
		CSLBaseMaterial **l_pMaterialList = l_pFTKMaterialLibrary->GetMaterialList();

		CSLXSIMaterial *l_pFTKMaterial = (CSLXSIMaterial *)l_pMaterialList[loop];

		CSIBCUserData *l_pUserData = l_pFTKMaterial->FindUserData("USED");

		// CrySpecific: unused materials was removed before this
// 		if (!l_pUserData)
// 		{
//   			l_pFTKMaterialLibrary->RemoveMaterial(loop);
//   			loop--;
// 		}
// 		else
		//CrySpecific: propagate used flag
		{
			int loop2;
			for(loop2 = 0; loop2 < l_pFTKMaterial->GetConnectionPointCount(); loop2++)
			{
				CSLConnectionPoint *l_pConnectionPoint = l_pFTKMaterial->GetConnectionPointList()[loop2];

				CSLXSIShader *l_pShader = l_pConnectionPoint->GetShader();

				if(l_pShader && in_pContext->ftkscene()->GetImageLibrary())
				{
					PropagateImageUsedFlag(in_pContext->ftkscene()->GetImageLibrary(), l_pShader);
				}
			}
		}
	}

	return status;
}

wchar_t *CMaterialLibraryFromXSI_RemoveUnused::GetClassID(){return L"CMaterialLibraryFromXSI_RemoveUnused";}

/**************************************************************************************
CXSIMaterialFromXSI
**************************************************************************************/
CXSIMaterialFromXSI::CXSIMaterialFromXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CXSIMaterialFromXSI::~CXSIMaterialFromXSI() {};

CStatus CXSIMaterialFromXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;

	Material l_XSIMaterial = (Material) *io_pXSIModel;

	Library l_XSILibrary = (Library) in_XSIParent;
	CSLMaterialLibrary *l_pFTKLibrary = (CSLMaterialLibrary *) in_pFTKParent;
	CTime l_time;

	if(l_XSIMaterial.IsValid())
	{
		// Let's add a new material to the material library
		CSLXSIMaterial *l_pFTKMaterial = l_pFTKLibrary->AddXSIMaterial();
		*io_pFTKModel = l_pFTKMaterial;

		// let's convert the name of the material
		// CrySpecific: always append the material library name
		CString l_FTKMaterialName;
		l_FTKMaterialName = l_XSILibrary.GetName() + L"_" + l_XSIMaterial.GetName();

		l_pFTKMaterial->SetName((char *)l_FTKMaterialName.GetAsciiString());
		l_pFTKMaterial->AttachUserData("CREF", in_pContext->AddCRef(l_XSIMaterial));

		// then the extra material info
		CValue l_WrapU, l_WrapV;

		l_WrapU = COMMANDS::GetValue(l_XSIMaterial.GetFullName() + CString(L".wrapu"), l_time.GetTime());
		l_WrapV = COMMANDS::GetValue(l_XSIMaterial.GetFullName() + CString(L".wrapv"), l_time.GetTime());

		CSLXSIMaterialInfo *l_pMaterialInfo = l_pFTKMaterial->CreateMaterialInfo();
		l_pMaterialInfo->SetUWrap((((LONG) l_WrapU) == 10497) ? CSLXSIMaterialInfo::SI_CLAMP : CSLXSIMaterialInfo::SI_REPEAT);
		l_pMaterialInfo->SetVWrap((((LONG) l_WrapV) == 10497) ? CSLXSIMaterialInfo::SI_CLAMP : CSLXSIMaterialInfo::SI_REPEAT);

		// grab it's custom psets
		CCustomPSetFromXSI l_CustomPSetConverter(CRY_FILE_TYPE_NONE);
		CRef l_MaterialCRef = l_XSIMaterial;
		_XSI_CALL(l_CustomPSetConverter.Execute(in_pContext, in_XSIParent, in_pFTKParent, NULL, &l_MaterialCRef, io_pFTKModel), L"Failed to convert custom pset");

		// grab it's user data
		CUserDataBlobFromXSI l_UserDataBlobConverter(CRY_FILE_TYPE_NONE);
		_XSI_CALL(l_UserDataBlobConverter.Execute(in_pContext, in_XSIParent, in_pFTKParent, NULL, &l_MaterialCRef, io_pFTKModel), L"Failed to convert user data blob");

		// let's add all the unique shaders
		int loop;
		CRefArray l_ShaderCollection = l_XSIMaterial.GetShaders();
		CRefArray l_ParameterCollection = l_XSIMaterial.GetParameters();

		CXSIShaderFromXSI l_ShaderConverter(CRY_FILE_TYPE_NONE);
		l_ShaderConverter.m_pMaterialCaller = this;

		for(loop = 0; loop < l_ShaderCollection.GetCount(); loop++)
		{
			CRef l_Shader = l_ShaderCollection.GetItem(loop);

			CSLTemplate *l_pNewShader;
			_XSI_CALL(l_ShaderConverter.Execute(in_pContext, l_XSIMaterial, l_pFTKMaterial, NULL, &l_Shader, &l_pNewShader), L"Failed to convert shader");
		}

		// let's add the connections now
		for(loop = 0; loop < l_ParameterCollection.GetCount(); loop++)
		{
			Parameter parameter = l_ParameterCollection.GetItem(loop);
			Shader l_Shader = parameter.GetSource();

			if(l_Shader.IsValid())
			{
				int loop2;
				CSLXSIShader *l_pShader = NULL;

				for(loop2 = 0; loop2 < l_pFTKMaterial->GetShaderCount(); loop2++)
				{
					if(strcmp(l_pFTKMaterial->GetShaderList()[loop2]->GetName(), l_Shader.GetName().GetAsciiString()) == 0)
					{
						l_pShader = l_pFTKMaterial->GetShaderList()[loop2];
						break;
					}
				}

				if(l_pShader != NULL)
				{
					CSLConnectionPoint *l_pConnectionPoint = FTKUConnectShader(l_pShader, l_pFTKMaterial, (char*) parameter.GetScriptName().GetAsciiString());
				}
				else
				{
					logerror(L"Shader not found in connection creation");
				}
			}
		}
	}

	return status;
}

wchar_t *CXSIMaterialFromXSI::GetClassID(){return L"CXSIMaterialFromXSI";}

/**************************************************************************************
CXSIShaderFromXSI
**************************************************************************************/
CXSIShaderFromXSI::CXSIShaderFromXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CXSIShaderFromXSI::~CXSIShaderFromXSI() {};

CStatus CXSIShaderFromXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;

	Material		l_XSIMaterial = (Material) in_XSIParent;
	Shader			l_XSIShader = (Shader) *io_pXSIModel;
	CSLXSIMaterial	*l_pFTKMaterial = (CSLXSIMaterial *) in_pFTKParent;
	CTime			l_Time;
	int				loop;

	if(l_XSIShader.IsValid())
	{
		
		CRefArray l_Owners = l_XSIMaterial.GetOwners();

		// First let's see if our shader is already in the material shader list
		CSLXSIShader *l_pFTKShader = NULL;

		for(loop = 0; loop < l_pFTKMaterial->GetShaderCount(); loop++)
		{
			CString l_ShaderName;
			l_ShaderName.PutAsciiString(l_pFTKMaterial->GetShaderList()[loop]->GetName());
			if(l_XSIShader.GetName() == l_ShaderName)
			{
				return status;
			}
		}

		// Let's add a new shader to the material
		l_pFTKShader = l_pFTKMaterial->AddShader();
		*io_pFTKModel = l_pFTKShader;

		// let's convert the name of the shader
		char l_Name[2048];

		wcstombs(l_Name, l_XSIShader.GetName().GetWideString(), 2048);
		l_pFTKShader->SetName(l_Name);

		// CrySpecific: special case, export the CryShader.1 as a phong so that it exported as a COLLADA profile_COMMON Phong.
		CString l_ProgID = l_XSIShader.GetProgID();
		char* l_strProgID = (char*)l_ProgID.GetAsciiString();

		if (strlen(l_strProgID) >= strlen("CryShader.1"))
		{
			l_strProgID += strlen(l_strProgID) - strlen("CryShader.1");
		}

		if (strcmp(l_strProgID, "CryShader.1") == 0)
		{
			long l_nPhysics,l_nMaterialID;

			l_strProgID = "Softimage.material-phong.1"; // PHONG_PROG_ID;

			// CrySpecific: encode the material ID and the physics property in the material name.
			l_nPhysics = l_XSIShader.GetParameterValue(L"Physics");
			l_nMaterialID = l_XSIShader.GetParameterValue(L"MaterialID");
			l_nMaterialID += 1;

			// let's convert the name of the material
			CString l_FTKMaterialName;
			l_FTKMaterialName = l_XSIMaterial.GetLibrary().GetName();
			
			l_FTKMaterialName += L"-";
			
			if (l_nMaterialID < 10)
			{
				l_FTKMaterialName += L"0";
			}

			l_FTKMaterialName += CValue(l_nMaterialID);

			l_FTKMaterialName += L"-";
			l_FTKMaterialName += l_XSIMaterial.GetName();
			if (l_nPhysics)
			{
				l_FTKMaterialName += L"-phys";
				l_FTKMaterialName += CValue(l_nPhysics);
			}

			l_pFTKMaterial->SetName((char*)l_FTKMaterialName.GetAsciiString());
		}
		else
		{
			l_strProgID = (char*)l_ProgID.GetAsciiString();
		}

		l_pFTKShader->SetProgID(l_strProgID);
		l_pFTKShader->SetOutputType((CSLXSIShader::EOutputType) l_XSIShader.GetOutputType2());
		l_pFTKShader->AttachUserData("CREF", in_pContext->AddCRef(l_XSIShader));

		// grab it's custom psets
		CCustomPSetFromXSI l_CustomPSetConverter(CRY_FILE_TYPE_NONE);
		CRef l_ShaderCRef = l_XSIShader;
		_XSI_CALL(l_CustomPSetConverter.Execute(in_pContext, in_XSIParent, in_pFTKParent, NULL, &l_ShaderCRef, io_pFTKModel), L"Failed to convert custom pset");

		// grab it's user data blob
		CUserDataBlobFromXSI l_UserDataBlobConverter(CRY_FILE_TYPE_NONE);
		_XSI_CALL(l_UserDataBlobConverter.Execute(in_pContext, in_XSIParent, in_pFTKParent, NULL, &l_ShaderCRef, io_pFTKModel), L"Failed to convert user data blob");

		// let's add all the unique shaders
		CRefArray l_ShaderCollection = l_XSIShader.GetShaders();
		CRefArray l_ParameterCollection = l_XSIShader.GetParameters();

		CXSIShaderFromXSI l_ShaderConverter(CRY_FILE_TYPE_NONE);
		l_ShaderConverter.m_pMaterialCaller = m_pMaterialCaller;

		for(loop = 0; loop < l_ShaderCollection.GetCount(); loop++)
		{
			CRef l_Shader = l_ShaderCollection.GetItem(loop);

			CSLTemplate *l_pFTKShader;
			_XSI_CALL(l_ShaderConverter.Execute(in_pContext, l_XSIMaterial, l_pFTKMaterial, NULL, &l_Shader, &l_pFTKShader), L"Failed to convert shader");
		}

		// let's add the connections and the parameters now
		for(loop = 0; loop < l_ParameterCollection.GetCount(); loop++)
		{
			Parameter parameter = l_ParameterCollection.GetItem(loop);

			Shader l_Shader = parameter.GetSource();
			CRef l_SomethingElse = parameter.GetSource();
			wcstombs(l_Name, parameter.GetScriptName().GetWideString(), 2048);

			// the parameter is connected to a shader
			if(l_Shader.IsValid())
			{
				int loop2;
				CSLXSIShader *l_pShader = NULL;

				for(loop2 = 0; loop2 < l_pFTKMaterial->GetShaderCount(); loop2++)
				{
					if(strcmp(l_pFTKMaterial->GetShaderList()[loop2]->GetName(), l_Shader.GetName().GetAsciiString()) == 0)
					{
						l_pShader = l_pFTKMaterial->GetShaderList()[loop2];
						break;
					}
				}

				if(l_pShader != NULL)
				{
					CSLShaderConnectionPoint *l_pConnectionPoint = FTKUConnectShader(l_pShader, l_pFTKShader, l_Name);
				}
				else
				{
					logmessage(L"Shader not found in connection creation");
				}
			}
			// the parameter is connected to something else (except FCurve)
			else if(l_SomethingElse.IsValid() && !l_SomethingElse.IsA(siFCurveID))
			{
				ImageClip2 l_XSIImage = (ImageClip2) l_SomethingElse;
				if(l_XSIImage.IsValid())
				{
					if(in_pContext->ftkscene()->GetImageLibrary()!=NULL)
					{
						// find the imageclip with the same name and connect
						CSLImage *l_pFTKImageClip = in_pContext->ftkscene()->GetImageLibrary()->FindImage((char*)l_XSIImage.GetName().GetAsciiString());
						if(l_pFTKImageClip)
						{
							CSLShaderConnectionPoint *l_pConnectionPoint = FTKUConnectImage(l_pFTKImageClip, l_pFTKShader, l_Name);
						}
					}
				}
			}
			// the parameter is not connected
			else 
			{
				int					l_ParamCount = 1;
				CRefArray			l_SubParameterCollection;
				Parameter			l_CurrentParameter = parameter;
				int					loop2,loop3;
				bool				l_IsCompound = false;

				// check if we are not dealing with a compound parameter
				// NOTE: we only support one level of nesting here
				if(parameter.IsA(siCompoundParameterID))
				{
					l_IsCompound = true;
					l_SubParameterCollection = parameter.GetParameters();
					l_ParamCount = l_SubParameterCollection.GetCount();
				}

				for(loop2 = 0; loop2 < l_ParamCount; loop2++)
				{
					CString l_ParameterName = parameter.GetScriptName();

					if(l_IsCompound)
					{
						l_CurrentParameter = l_SubParameterCollection.GetItem(loop2);
						l_ParameterName += CString(L".");
						l_ParameterName += l_CurrentParameter.GetScriptName();
						wcstombs(l_Name, l_ParameterName.GetWideString(), 2048);
					}

					CValue l_ParameterValue;
					bool l_HasInstanceValue = false;
					bool bFound = false;

					if (l_CurrentParameter.HasInstanceValue())
					{
						// let's find a primary owner of the material otherwise we will not be able to get the proper type and value

						for(loop3 = 0; loop3 < l_Owners.GetCount(); loop3++)
						{
							ProjectItem l_XSIOwner = (ProjectItem) l_Owners[loop3];
							Cluster l_XSICluster;

							// we want the X3DObject owning this instance value: X3DObject > Primitive > Cluster
							if (l_XSIOwner.IsA(siClusterID))
							{
								l_XSICluster = l_XSIOwner;

								l_XSIOwner = l_XSIOwner.GetParent();

								if (l_XSIOwner.IsA(siPrimitiveID))
								{
									l_XSIOwner = l_XSIOwner.GetParent();
								}
							}

							if(l_XSIOwner.IsValid() && l_XSIOwner.IsA(siX3DObjectID) )
							{
								if(l_CurrentParameter.IsSupportedInstanceValue(l_XSIOwner, false))
								{
									l_ParameterValue = l_CurrentParameter.GetInstanceValue(l_XSIOwner, false);										
									l_HasInstanceValue = true;

									
									for(int i = 0; i< m_pMaterialCaller->m_pMatLibCaller->m_InstanceDataRecords.GetUsed();++i)
									{
										if(m_pMaterialCaller->m_pMatLibCaller->m_InstanceDataRecords[i]->m_Owner == l_XSIOwner &&
											m_pMaterialCaller->m_pMatLibCaller->m_InstanceDataRecords[i]->m_ParamValue == l_ParameterValue &&
											m_pMaterialCaller->m_pMatLibCaller->m_InstanceDataRecords[i]->m_Parameter.IsEqualTo(l_CurrentParameter))
										{
											bFound = true;
											break;
										}
									}
									if(!bFound)
									{
										//New instance data value.
										ShaderInstanceDataRecord *l_pInstanceData = new ShaderInstanceDataRecord();
										l_pInstanceData->m_Owner = l_XSIOwner;
										l_pInstanceData->m_ParamValue = l_ParameterValue;
										l_pInstanceData->m_pShader = l_pFTKShader;
										l_pInstanceData->m_Parameter = l_CurrentParameter;
										m_pMaterialCaller->m_pMatLibCaller->m_InstanceDataRecords.Add(l_pInstanceData);
									}
									break;
								}
							}
						}
					}

					if(!l_HasInstanceValue)
					{
						if (l_CurrentParameter.GetCapabilities() & siAnimatable)
						{
							l_ParameterValue = l_CurrentParameter.GetValue(l_Time.GetTime());
						}
						else
						{
							l_ParameterValue = l_CurrentParameter.GetValue();
						}
					}

					if (l_CurrentParameter.IsValid()&& !bFound)
					{
						// Set the parameter value according to the type
						switch(l_ParameterValue.m_t)
						{
							case CValue::siInt1 :
							case CValue::siInt2 :
							case CValue::siInt4 :
							case CValue::siUInt1 :
							case CValue::siUInt2 :
							case CValue::siUInt4 :
							{
								LONG l_Value = 0;
								l_Value = l_ParameterValue;
								FTKUSetShaderParameterValue(l_pFTKShader, l_Name, l_Value);
							}
							break;

							case CValue::siBool :
							{
								bool l_Value = l_ParameterValue;
								FTKUSetShaderParameterValue(l_pFTKShader, l_Name, l_Value);
							}
							break;

							case CValue::siFloat :
							case CValue::siDouble :
							{
								float l_Value = l_ParameterValue;
								FTKUSetShaderParameterValue(l_pFTKShader, l_Name, l_Value);
							}
							break;

							case CValue::siString :
							{
								CString l_XSIString = l_ParameterValue;
								char *l_pAnsiString = (char*) l_XSIString.GetAsciiString();

								if(_stricmp(l_Name, "filename") == 0)
								{
									CSIBCString RelativeName;

									CString l_ExportPath = in_pContext->exportproperty().GetParameterValue(L"Filename");
									CValue l_PathRelative = in_pContext->exportproperty().GetParameterValue(L"PathRelative");

									if(l_PathRelative == true)
									{
										FTKUPathAbsoluteToRelative(l_pAnsiString, (char*) l_ExportPath.GetAsciiString(), RelativeName);
										FTKUSetShaderParameterValue(l_pFTKShader, l_Name, RelativeName.GetText());
									}
									else
									{
										FTKUSetShaderParameterValue(l_pFTKShader, l_Name, l_pAnsiString);
									}
								}
								else
								{
									FTKUSetShaderParameterValue(l_pFTKShader, l_Name, l_pAnsiString);
								}
							}
							break;
							default: logmessage(L"Unable to export this parameter : %s", l_CurrentParameter.GetFullName().GetWideString());
							break;
						}
					}
				}
			}
		}		
	}

	return status;
}

wchar_t *CXSIShaderFromXSI::GetClassID(){return L"CXSIShaderFromXSI";}

/**************************************************************************************
CXSIInstanceDataFromXSI
**************************************************************************************/
CXSIInstanceDataFromXSI::CXSIInstanceDataFromXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CXSIInstanceDataFromXSI::~CXSIInstanceDataFromXSI() {};

CStatus CXSIInstanceDataFromXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;

	if (*io_pFTKModel)
	{
		CSLModel *l_pFTKModel = (CSLModel *)(*io_pFTKModel);

		// for now, shader instance data is only supported on meshes, because they are the 
		// only primitive that supports multiple texture coordinates in dotXSI
		if (l_pFTKModel->Primitive() && l_pFTKModel->Primitive()->Type() == CSLTemplate::XSI_MESH)
		{	
			for (LONG loop = 0; loop < m_pMatLibConverter->m_InstanceDataRecords.GetUsed();)
			{
				ShaderInstanceDataRecord *l_pRecord = m_pMatLibConverter->m_InstanceDataRecords[loop];
				if(l_pRecord->m_Owner == *io_pXSIModel)
				{
					// find if our instance data already exist for this model
					CSLShaderInstanceData *l_pInstanceData = l_pRecord->m_pShader->AddInstanceData();
					CSLVariantParameter *l_pParameter = l_pInstanceData->AddParameter();
					
					l_pParameter->SetName((char*)l_pRecord->m_Parameter.GetScriptName().GetAsciiString());

					l_pInstanceData->SetReference(l_pFTKModel);

					// Set the parameter value according to the type
					switch(l_pRecord->m_ParamValue.m_t)
					{
						case CValue::siInt1 :
						case CValue::siInt2 :
						case CValue::siInt4 :
						case CValue::siUInt1 :
						case CValue::siUInt2 :
						case CValue::siUInt4 :
						{
							LONG l_Value = 0;
							l_Value = l_pRecord->m_ParamValue;
							FTKUSetParameterValue(l_pParameter, l_Value);
						}
						break;

						case CValue::siBool :
						{
							bool l_Value = l_pRecord->m_ParamValue;
							FTKUSetParameterValue(l_pParameter, l_Value);
						}
						break;

						case CValue::siFloat :
						case CValue::siDouble :
						{
							float l_Value = l_pRecord->m_ParamValue;
							FTKUSetParameterValue(l_pParameter, l_Value);
						}
						break;

						case CValue::siString :
						{
							CString l_XSIString = l_pRecord->m_ParamValue;

							char *l_pAnsiString = (char *) calloc(sizeof(char), wcslen(l_XSIString.GetWideString()) + 1);
							wcstombs(l_pAnsiString, l_XSIString.GetWideString(), wcslen(l_XSIString.GetWideString()));
							FTKUSetParameterValue(l_pParameter, l_pAnsiString);
							free(l_pAnsiString);
						}
						break;
						default: logmessage(L"Unable to export this parameter : %s", l_pRecord->m_Parameter.GetFullName().GetWideString());
						break;
					}					


					// now remove this item from the list so we can reduce it
					delete l_pRecord;
					m_pMatLibConverter->m_InstanceDataRecords.DeleteAt(loop,1);
				}
				else
				{
					loop++;
				}
			}
		}
	}

	return status;
}

wchar_t *CXSIInstanceDataFromXSI::GetClassID(){return L"CXSIInstanceDataFromXSI";}

/**************************************************************************************
CInstanceDataToXSI
**************************************************************************************/
CInstanceDataToXSI::CInstanceDataToXSI() {};

CInstanceDataToXSI::~CInstanceDataToXSI() 
{
	int loop = 0; 
	for(loop = 0; loop < m_InstanceDataRecords.GetUsed(); loop++)
	{
		if(m_InstanceDataRecords[loop] != NULL)
		{
			delete m_InstanceDataRecords[loop];
			m_InstanceDataRecords[loop] = NULL;
		}
	}
};

CStatus CInstanceDataToXSI::Execute(CdotXSIConverter *in_pContext)
{
	CStatus status = CStatus::OK;

	int loop, loop2;

	for(loop = 0; loop < m_InstanceDataRecords.GetUsed(); loop++)
	{
		CSIBCUserData *l_pUserData = m_InstanceDataRecords[loop]->m_pShader->FindUserData("CREF");

		if(l_pUserData)
		{
			CRef *l_pShaderCRef = (CRef*)(l_pUserData->GetData());
			Shader l_XSIShader = (Shader) *l_pShaderCRef;

			CSLModel* l_pFTKOwner = m_InstanceDataRecords[loop]->m_pInstanceData->GetReference();
			CRef *l_pOwnerCRef = NULL;
			CRef l_XSIOwner;
			CRef l_Owner;

			if(l_pFTKOwner == NULL)
			{

				Shader l_CurrentShader = l_XSIShader;
				Material l_Material;

				while(!l_Material.IsValid() && l_CurrentShader.IsValid())
				{
					l_Material = l_CurrentShader.GetOwners()[0];
					if(!l_Material.IsValid())
						l_CurrentShader = l_CurrentShader.GetOwners()[0];
				}


				CRefArray l_Owners = l_Material.GetOwners();
				int loop2;
				int count = l_Owners.GetCount();
				for(loop2 = 0; loop2 < count; loop2 ++)
				{
					siClassID l_ClassID = l_Owners[loop2].GetClassID();

					if(l_ClassID != siMaterialLibraryID)
					{
						l_XSIOwner = l_Owners[loop2];

						if(l_ClassID == siClusterID)
						{
							Cluster l_Cluster = l_XSIOwner;
							l_XSIOwner = l_Cluster.GetParent3DObject();
						}

						break;
					}
				}
			}
			else
			{
				l_pUserData = l_pFTKOwner->FindUserData("CREF");

				if(l_pUserData)
				{
					CRef *l_pOwnerCRef = (CRef*)(l_pUserData->GetData());
					l_XSIOwner = *l_pOwnerCRef;
				}
			}

			if(l_XSIOwner.IsValid() && l_XSIShader.IsValid())
			{
				for(loop2 = 0; loop2 < m_InstanceDataRecords[loop]->m_pInstanceData->GetParameterCount(); loop2++)
				{
					CSLVariantParameter *l_pFTKParameter = m_InstanceDataRecords[loop]->m_pInstanceData->GetParameterList()[loop2];
					CString l_ParamName;
					l_ParamName.PutAsciiString(l_pFTKParameter->GetName());
					Parameter l_XSIParameter = l_XSIShader.GetParameter(l_ParamName);
					if(l_XSIParameter.IsValid())
					{
						// now we set the value
						SI_TinyVariant *l_pFTKParamValue = l_pFTKParameter->GetValue();

						switch(l_pFTKParamValue->variantType)
						{
							case SI_VT_INT:
								l_XSIParameter.PutInstanceValue(l_XSIOwner, XSI::CValue((LONG) l_pFTKParamValue->lVal), true);
							break;
							case SI_VT_FLOAT:
								l_XSIParameter.PutInstanceValue(l_XSIOwner, XSI::CValue(l_pFTKParamValue->fVal), true);
							break;
							case SI_VT_BOOL:
								l_XSIParameter.PutInstanceValue(l_XSIOwner, XSI::CValue((bool) (l_pFTKParamValue->boolVal ? true : false)), true);
							break;
							case SI_VT_PCHAR:
							{
								CString l_Value;
								l_Value.PutAsciiString(l_pFTKParamValue->p_cVal);

								if (!l_Value.IsEmpty())
								{
									l_XSIParameter.PutInstanceValue(l_XSIOwner, l_Value, true);
								}
							}
							break;
							default: logmessage(L"Unable to import this parameter : %s", l_pFTKParameter->GetName());
							break;
						}
					}
				}
			}
		}
	}

	return status;
}

wchar_t *CInstanceDataToXSI::GetClassID(){return L"CInstanceDataToXSI";}

/**************************************************************************************
CMaterialLibraryToXSI
**************************************************************************************/
CMaterialLibraryToXSI::CMaterialLibraryToXSI() {};

CMaterialLibraryToXSI::~CMaterialLibraryToXSI() {};

CStatus CMaterialLibraryToXSI::Execute(CdotXSIConverter *in_pContext)
{
	CStatus status = CStatus::OK;

	CSLMaterialLibrary *l_pFTKMaterialLibrary = in_pContext->ftkscene()->GetMaterialLibrary();
	Library l_XSIMatLib;

	// import the name
	CString l_MatLibName;
	l_MatLibName.PutAsciiString(l_pFTKMaterialLibrary->GetName());
	COMMANDS::CreateLibrary(l_MatLibName, CValue((LONG)1), l_XSIMatLib);

	if(l_XSIMatLib.IsValid())
	{
		l_pFTKMaterialLibrary->AttachUserData("CREF", in_pContext->AddCRef(l_XSIMatLib));
	}

	assert(l_XSIMatLib.IsValid());

	// here we do a mini traversal of the material library and we call the CXSIMaterialToXSI callback
	int loop;
	CXSIMaterialToXSI l_MaterialConverter(CRY_FILE_TYPE_NONE);

	l_MaterialConverter.m_pMatLibCaller = this;

	for(loop = 0; loop < l_pFTKMaterialLibrary->GetMaterialCount(); loop++)
	{
		CRef l_NewMaterial;
		CSLTemplate *l_pFTKMaterial = l_pFTKMaterialLibrary->GetMaterialList()[loop];

		if((l_pFTKMaterial->Type() == CSLTemplate::XSI_MATERIAL) || (l_pFTKMaterial->Type() == CSLTemplate::COLLADA_MATERIAL))
		{
			_XSI_CALL(l_MaterialConverter.Execute(in_pContext, l_XSIMatLib, l_pFTKMaterialLibrary, NULL, &l_NewMaterial, &l_pFTKMaterial), L"Failed to convert material");
		}
		else
		{
			logmessage(L"Found unknown type of materials");
		}
	}

	return status;
}

wchar_t *CMaterialLibraryToXSI::GetClassID(){return L"CMaterialLibraryToXSI";}

/**************************************************************************************
CXSIMaterialToXSI
**************************************************************************************/
CXSIMaterialToXSI::CXSIMaterialToXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CXSIMaterialToXSI::~CXSIMaterialToXSI() {};

CStatus CXSIMaterialToXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;

	Material l_XSIMaterial;
	CSLXSIMaterial *l_pFTKMaterial = (CSLXSIMaterial*) *io_pFTKModel;
	Library l_XSIMatLib = (Library) in_XSIParent;
	CTime l_time;

	// This is the right way to create a new emtpy material, put an empty string in the preset, then the name of the material
	// then the name of the material library, then an empty CValue to tell it not to assign it to any object, then
	// the following two false are use to not assign it to a list and not initialize it

	CString l_MaterialName;
	l_MaterialName.PutAsciiString(l_pFTKMaterial->GetName());
	l_XSIMaterial = (Material) COMMANDS::SICreateMaterial(CString(L""), l_MaterialName, l_XSIMatLib.GetFullName(), CValue(), false, false);

	assert(l_XSIMaterial.IsValid());

	l_pFTKMaterial->AttachUserData("CREF", in_pContext->AddCRef(l_XSIMaterial));

	// check if we have to set the extra material info

	CSLXSIMaterialInfo *l_pMaterialInfo = l_pFTKMaterial->CreateMaterialInfo();
	if(l_pMaterialInfo)
	{
		l_XSIMaterial.PutParameterValue(L"wrapu", (LONG)((l_pMaterialInfo->GetUWrap() == CSLXSIMaterialInfo::SI_CLAMP) ? 10497 : 10496));
		l_XSIMaterial.PutParameterValue(L"wrapv", (LONG)((l_pMaterialInfo->GetUWrap() == CSLXSIMaterialInfo::SI_CLAMP) ? 10497 : 10496));
	}

	// get the custom psets
	CCustomPSetToXSI l_CustomPSetConverter(CRY_FILE_TYPE_NONE);
	CRef l_MaterialCRef = l_XSIMaterial;
	_XSI_CALL(l_CustomPSetConverter.Execute(in_pContext, in_XSIParent, in_pFTKParent, NULL, &l_MaterialCRef, io_pFTKModel), L"Failed to convert custom pset");

	// get the user data blobs
	CUserDataBlobToXSI l_UserDataBlobConverter(CRY_FILE_TYPE_NONE);
	_XSI_CALL(l_UserDataBlobConverter.Execute(in_pContext, in_XSIParent, in_pFTKParent, NULL, &l_MaterialCRef, io_pFTKModel), L"Failed to convert user data blob");


	// we need to create the shaders in order of connection
	CXSIShaderToXSI l_ShaderConverter(CRY_FILE_TYPE_NONE);
	l_ShaderConverter.m_pMaterialCaller = this;

	int loop;
	for(loop = 0; loop < l_pFTKMaterial->GetConnectionPointCount(); loop++)
	{
		CRef			l_NewShader;
		CSLTemplate		*l_pConnectionPoint = (CSLTemplate*)l_pFTKMaterial->GetConnectionPointList()[loop];
		if(l_pFTKMaterial->GetConnectionPointList()[loop]->GetShader() != NULL)
		{
			_XSI_CALL(l_ShaderConverter.Execute(in_pContext, l_XSIMaterial, *io_pFTKModel, NULL, &l_NewShader, &l_pConnectionPoint), L"Failed to create shader");
		}
	}

	return status;
}

wchar_t *CXSIMaterialToXSI::GetClassID(){return L"CXSIMaterialToXSI";}

/**************************************************************************************
CXSIShaderToXSI
**************************************************************************************/
CXSIShaderToXSI::CXSIShaderToXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CXSIShaderToXSI::~CXSIShaderToXSI() {};

CStatus CXSIShaderToXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;

	Material					l_XSIMaterial = (Material) in_XSIParent;
	Shader						l_XSIPreviousShader = (Shader) in_XSIParent;
	CSLXSIMaterial				*l_pFTKMaterial = (CSLXSIMaterial*) in_pFTKParent;
	CSLConnectionPoint			*l_pConnectionPoint = (CSLConnectionPoint*)*io_pFTKModel;
	CSLXSIShader				*l_pFTKShader = l_pConnectionPoint->GetShader();
	CSLImage					*l_pFTKImage = NULL;
	CTime						l_Time;
	int							loop;


	// get the parameter to connect to
	CRefArray l_ParameterCollection;
	Parameter l_parameter;
	CString l_ConnectionName;
	l_ConnectionName.PutAsciiString(l_pConnectionPoint->GetName());

	if(l_XSIMaterial.IsValid())
	{
		l_parameter = l_XSIMaterial.GetParameter(l_ConnectionName);
	}
	else
	{
		l_parameter = l_XSIPreviousShader.GetParameter(l_ConnectionName);
	}

	// if the parameter is not found just bail out
	if(l_parameter.IsValid() == false)
	{
		logmessage(L"Invalid parameter %S", l_pConnectionPoint->GetName());
		status = CStatus::False;
	}
	else
	{
		// check if we are connected to a shader or an image
		if((l_pFTKShader == NULL) && (in_pContext->ftkscene()->GetImageLibrary()))
		{
			CSLShaderConnectionPoint *l_pShaderConnectionPoint = (CSLShaderConnectionPoint*)l_pConnectionPoint;
			l_pFTKImage = in_pContext->ftkscene()->GetImageLibrary()->FindImage(l_pShaderConnectionPoint->GetImage());
			if(l_pFTKImage != NULL)
			{
				CSIBCUserData *l_pUserData = l_pFTKImage->FindUserData("CREF");
				ImageClip2 l_XSIImageClip;

				if(l_pUserData)
				{
					CRef *l_pCRef = (CRef*)(l_pUserData->GetData());

					l_XSIImageClip = (ImageClip2) *l_pCRef;
					if(l_XSIImageClip.IsValid())
					{
						// we connect the existing shader to the port
						CRef l_PreviousObject;

						// This takes the existing shader and attaches it to the material or the previous shader
						l_parameter.Connect(l_XSIImageClip, l_PreviousObject);
					}
				}
			}

			return status;
		}

		// Check if this shader already has a CREF user data
		CSIBCUserData *l_pUserData = l_pFTKShader->FindUserData("CREF");
		Shader l_XSIShader;

		// enumerate the instance data
		for(loop = 0; loop < l_pFTKShader->GetInstanceDataCount(); loop++)
		{
			CSLShaderInstanceData *l_pInstanceData = l_pFTKShader->GetInstanceDataList()[loop];

			ShaderInstanceDataRecord *l_pRecord = new ShaderInstanceDataRecord();
			l_pRecord->m_pShader = l_pFTKShader;
			l_pRecord->m_pInstanceData = l_pInstanceData;
			m_pMaterialCaller->m_pMatLibCaller->m_pInstanceDataCollector->m_InstanceDataRecords.Add(l_pRecord);
		}


		if(l_pUserData)
		{
			CRef *l_pCRef = (CRef*)(l_pUserData->GetData());
			l_XSIShader = (Shader) *l_pCRef;
		}

		// We create a new shader here and attach it to the parent
		if(!l_XSIShader.IsValid())
		{
			CRef l_PreviousObject;

			// This creates a new shader and attaches it to the material
			CString l_ProgID;
			l_ProgID.PutAsciiString(l_pFTKShader->GetProgID());
			l_parameter.ConnectFromProgID(l_ProgID, l_PreviousObject, l_XSIShader);
			if(l_XSIShader.IsValid())
			{
				l_pFTKShader->AttachUserData("CREF", in_pContext->AddCRef(l_XSIShader));
			}
			else
			{
				return CStatus::False;
			}

			*io_pXSIModel = l_XSIShader;

			// now set the shader name and all that stuff
			CString l_ShaderName;
			l_ShaderName.PutAsciiString(l_pFTKShader->GetName());
			l_XSIShader.PutName(l_ShaderName);

			// grab it's custom psets
			CCustomPSetToXSI l_CustomPSetConverter(CRY_FILE_TYPE_NONE);
			CRef l_ShaderCRef = l_XSIShader;
			CSLTemplate *l_pShaderTemplate = (CSLTemplate*) l_pFTKShader;
			_XSI_CALL(l_CustomPSetConverter.Execute(in_pContext, in_XSIParent, in_pFTKParent, NULL, &l_ShaderCRef, &l_pShaderTemplate), L"Failed to convert custom pset");

			// grab it's user data blobs
			CUserDataBlobToXSI l_UserDataBlobConverter(CRY_FILE_TYPE_NONE);
			_XSI_CALL(l_UserDataBlobConverter.Execute(in_pContext, in_XSIParent, in_pFTKParent, NULL, &l_ShaderCRef, &l_pShaderTemplate), L"Failed to convert user data blob");

			char l_ParamName[4096];
			for(loop = 0; loop < l_pFTKShader->GetParameterCount(); loop++)
			{
				CSLVariantParameter *l_pFTKParameter = l_pFTKShader->GetParameterList()[loop];
				
				// get the parameter name
				strcpy(l_ParamName, l_pFTKParameter->GetName());
				
				// split compound if needed
				char *l_pParamPrefix = strtok(l_ParamName, ".");
				char *l_pParamSuffix = strtok(NULL, ".");

				CString l_ParamPrefix;
				l_ParamPrefix.PutAsciiString(l_pParamPrefix);

				// find the sub parameter if needed
				if(l_pParamSuffix != NULL)
				{
					// find the parameter with the same name
					XSI::Parameter l_XSIParameter = l_XSIShader.GetParameter(l_ParamPrefix);
					CString l_ParamSuffix;
					l_ParamSuffix.PutAsciiString(l_pParamSuffix);

					// now we set the value
					SI_TinyVariant *l_pFTKParamValue = l_pFTKParameter->GetValue();

					switch(l_pFTKParamValue->variantType)
					{
						case SI_VT_INT:
							l_XSIParameter.PutParameterValue(l_ParamSuffix, XSI::CValue((LONG) l_pFTKParamValue->lVal));
						break;
						case SI_VT_FLOAT:
							l_XSIParameter.PutParameterValue(l_ParamSuffix, XSI::CValue(l_pFTKParamValue->fVal));
						break;
						case SI_VT_BOOL:
							l_XSIParameter.PutParameterValue(l_ParamSuffix, XSI::CValue((bool) (l_pFTKParamValue->boolVal ? true : false)));
						break;
						case SI_VT_PCHAR:
						{
							CString l_Value;
							l_Value.PutAsciiString(l_pFTKParamValue->p_cVal);
							l_XSIParameter.PutParameterValue(l_ParamSuffix, l_Value);
						}
						break;
						default: logmessage(L"Unable to import this parameter : %s", l_pFTKParameter->GetName());
						break;
					}
				}
				else
				{

					// now we set the value
					SI_TinyVariant *l_pFTKParamValue = l_pFTKParameter->GetValue();

					switch(l_pFTKParamValue->variantType)
					{
						case SI_VT_INT:
							l_XSIShader.PutParameterValue(l_ParamPrefix, XSI::CValue((LONG) l_pFTKParamValue->lVal));
						break;
						case SI_VT_FLOAT:
							l_XSIShader.PutParameterValue(l_ParamPrefix, XSI::CValue(l_pFTKParamValue->fVal));
						break;
						case SI_VT_BOOL:
							l_XSIShader.PutParameterValue(l_ParamPrefix, XSI::CValue((bool) (l_pFTKParamValue->boolVal ? true : false)));
						break;
						case SI_VT_PCHAR:
						{
							CString l_Value;
							if(l_pFTKParamValue->p_cVal && strlen(l_pFTKParamValue->p_cVal))
							{
								if(_stricmp(l_ParamName, "filename") == 0)
								{
									CSIBCString l_AbsolutePath;
									FTKUPathRelativeToAbsolute(l_pFTKParamValue->p_cVal, in_pContext->ftkscene()->Parser()->FileName().GetText(), l_AbsolutePath);
									l_Value.PutAsciiString(l_AbsolutePath.GetText());
								}
								else
								{
									l_Value.PutAsciiString(l_pFTKParamValue->p_cVal);
								}

								CStatus l_Status = l_XSIShader.PutParameterValue(l_ParamPrefix, l_Value);

								// register a new instance data 
								if(l_Status != CStatus::OK)
								{
									CSLShaderInstanceData *l_pInstanceData = l_pFTKShader->AddInstanceData();
									CSLVariantParameter *l_pParam = l_pInstanceData->AddParameter();
									l_pParam->SetValue(l_pFTKParamValue);
									l_pParam->SetName((char*) l_ParamPrefix.GetAsciiString());
									l_pInstanceData->SetReference(NULL);

									ShaderInstanceDataRecord *l_pRecord = new ShaderInstanceDataRecord();
									l_pRecord->m_pShader = l_pFTKShader;
									l_pRecord->m_pInstanceData = l_pInstanceData;
									m_pMaterialCaller->m_pMatLibCaller->m_pInstanceDataCollector->m_InstanceDataRecords.Add(l_pRecord);								
								}
							}
						}
						break;
						default: logmessage(L"Unable to import this parameter : %s", l_pFTKParameter->GetName());
						break;
					}
				}
			}

			// we need to create the shaders in order of connection
			CXSIShaderToXSI l_ShaderConverter(CRY_FILE_TYPE_NONE);
			l_ShaderConverter.m_pMaterialCaller = m_pMaterialCaller;

			int loop;
			for(loop = 0; loop < l_pFTKShader->GetConnectionPointCount(); loop++)
			{
				CRef			l_NewShader;
				CSLTemplate		*l_pConnectionPoint = (CSLTemplate*)l_pFTKShader->GetConnectionPointList()[loop];

				_XSI_CALL(l_ShaderConverter.Execute(in_pContext, l_XSIShader, *io_pFTKModel, NULL, &l_NewShader, &l_pConnectionPoint), L"Failed to create shader");
			}
		}
		else
		// we connect the existing shader to the port
		{
			CRef l_PreviousObject;

			// This takes the existing shader and attaches it to the material or the previous shader
			l_parameter.Connect(l_XSIShader, l_PreviousObject);
		}
	}

	return status;
}

wchar_t *CXSIShaderToXSI::GetClassID(){return L"CXSIShaderToXSI";}
