//***************************************************************************************
//
// File supervisor: Softimage Rendering & Pipeline team
//
// (c) Copyright 2001-2005 Avid Technology, Inc. . All rights reserved.
//
//***************************************************************************************

/**************************************************************************************
THIS CODE IS PUBLISHED AS A SAMPLE ONLY AND IS PROVIDED "AS IS".
IN NO EVENT SHALL SOFTIMAGE, AVID TECHNOLOGY, INC. AND/OR THEIR RESPECTIVE
SUPPLIERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
CONNECTION WITH THE USE OR PERFORMANCE OF THIS CODE .

COPYRIGHT NOTICE. Copyright  1999-2005 Avid Technology Inc. . All rights reserved. 

SOFTIMAGE is a registered trademark of Avid Technology Inc. or its subsidiaries 
or divisions. Windows NT is a registered trademark of Microsoft Corp. All other
trademarks contained herein are the property of their respective owners. 
****************************************************************************************/

/*! \file cnv_modelinfo.cpp
/*!
	implementation file for model level info related classes
*/

#include "stdafx.h"
#include "cnv_modelinfo.h"
#include "../FTKUtil.h"
#include <Model.h>
#include <Visibility.h>
#include <XSITransform.h>
#include <XSILimit.h>
#include <Angle.h>
#include <xsi_x3dobject.h>
#include "cmdstubs.h"
#include <xsi_fcurve.h>
#include <xsi_time.h>
#include <xsi_kinematics.h>
#include <xsi_kinematicstate.h>
#include <xsi_statickinematicstate.h>
#include <xsi_math.h>
#include <xsi_model.h>
#include <xsi_material.h>
#include <xsi_rotation.h>
#include <xsi_customoperator.h>
#include <xsi_sceneitem.h>
#include <xsi_userdatablob.h>
#include <XSIMaterial.h>
#include <GlobalMaterial.h>
#include <MaterialLibrary.h>
#include "plugin_stub.h"
#include "../helper.h"
#include <string>
#include <map>
#include <vector>
#include <algorithm>
#include <cctype>

struct JointParameterEntry
{
	JointParameterEntry(const std::wstring& propName, float localOffset): propName(propName), localOffset(localOffset) {}
	std::wstring propName;
	float localOffset;
};

/**************************************************************************************
CModelInfoFromXSI
**************************************************************************************/
CModelInfoFromXSI::CModelInfoFromXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CModelInfoFromXSI::~CModelInfoFromXSI() {};

CStatus CModelInfoFromXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;
	
	if(*io_pFTKModel != NULL)
	{
		CSLModel *l_pFTKModel = (CSLModel *) *io_pFTKModel;
		X3DObject l_XSIModel = (X3DObject) *io_pXSIModel;
		CTime	  l_Time;

		// check if we must bail out sooner
		if((l_pFTKModel->Template() == NULL) && (l_pFTKModel->Primitive() == NULL))
			return status;

		// CrySpecific
//		Model l_CryExportNode = l_XSIModel.GetModel();
		Model l_CryExportNode = GetCryExportNode(l_XSIModel);

 		if (l_CryExportNode.IsEqualTo(l_XSIModel))
		{
			// Get the name 
			l_pFTKModel->SetName((char*)l_XSIModel.GetName().GetAsciiString());
		}
		else
		{
			// CrySpecific : concatenation of the CryExportNode name in the children's name 
//			l_pFTKModel->SetName((char*)(l_XSIModel.GetName() + L"-" + l_CryExportNode.GetName()).GetAsciiString());

			char l_strNewNodename[MAX_PATH];
			strcpy(l_strNewNodename, l_XSIModel.GetName().GetAsciiString());
			if (strstr(l_strNewNodename,"CryJointNode"))
			{
				char l_strPostfix[MAX_PATH];
				strcpy(l_strPostfix, &l_strNewNodename[12]);
				strcpy(l_strNewNodename, "_joint");
				strcat(l_strNewNodename, l_strPostfix);
			}

//			if (l_XSIModel.GetType() != L"null")
			{
				CString l_CryExportNodename = l_CryExportNode.GetName();
				char *l_strNodename = (char*)(l_CryExportNodename.GetAsciiString());
				char *l_strCEN = strstr(l_strNodename, "CryExportNode");
				if (l_strCEN)
				{
					char l_strPost[MAX_PATH];
					memset(l_strPost, 0, MAX_PATH);
					memcpy(l_strPost, l_strNodename, l_strCEN-l_strNodename-1);
					strcat(l_strPost, &l_strCEN[13]);

					if (l_strPost)
					{
						strcat(l_strNewNodename, "%");
						strcat(l_strNewNodename, l_strPost);
						strcat(l_strNewNodename, "%");
					}
				}
			}

//			_strlwr(l_strNewNodename);
			l_pFTKModel->SetName((char*)l_strNewNodename);
		}

		if(l_pFTKModel->Primitive() && strlen(l_pFTKModel->Primitive()->GetName()) == 0)
		{
			if(	(l_pFTKModel->Primitive()->Type() == CSLTemplate::SI_NULL_OBJECT) || 
				(l_pFTKModel->Primitive()->Type() == CSLTemplate::SI_POINT_LIGHT) || 
				(l_pFTKModel->Primitive()->Type() == CSLTemplate::SI_DIRECTIONAL_LIGHT) || 
				(l_pFTKModel->Primitive()->Type() == CSLTemplate::SI_INFINITE_LIGHT) || 
				(l_pFTKModel->Primitive()->Type() == CSLTemplate::SI_SPOT_LIGHT)
				)
			{
				l_pFTKModel->Primitive()->SetName((char*)l_XSIModel.GetName().GetAsciiString());		
			}
		}


		// Get the visility, it's stored in the visibility.viewvis property
		// Note: this property does not existe for the Scene_Root
		if ((!in_pContext->scene().GetRoot().IsEqualTo(l_XSIModel)) && (l_pFTKModel->Visibility() != NULL))
		{			
			Property l_Visibility = l_XSIModel.GetProperties().GetItem(L"Visibility");
			l_pFTKModel->Visibility()->SetVisibility(l_Visibility.GetParameterValue(L"viewvis"));
		}

		// scene root has no transform (at least none that can be useful)

		if(in_pContext->scene().GetRoot().IsEqualTo(l_XSIModel))
		{
			CSIBCVector3D l_FTKVector(1.0f, 1.0f, 1.0f);
			l_pFTKModel->XSITransform()->SetRotationOrder(CSLTemplate::SI_XYZ);
			l_pFTKModel->XSITransform()->SetHierarchicalScaling(true);
			l_pFTKModel->XSITransform()->SetScale(l_FTKVector);
			l_pFTKModel->XSITransform()->SetNeutralPoseScale(l_FTKVector);
			l_pFTKModel->XSITransform()->SetPivotScale(l_FTKVector);
			l_pFTKModel->XSITransform()->SetPivotCompScale(l_FTKVector);

			l_pFTKModel->XSITransform()->ComputeLocalMatrix();
			l_pFTKModel->XSITransform()->ComputeGlobalMatrix();

		}
		else if(l_pFTKModel->XSITransform() != NULL)
		{
			// Fill in the transform now
			CSIBCVector3D l_FTKVector;
			Parameter l_XSIParameter;

			// this code gets the local kinematics transform
			KinematicState				l_XSIKineState = l_XSIModel.GetKinematics().GetLocal();
			MATH::CTransformation		l_XSITransform = l_XSIKineState.GetTransform();

			// rotation order
			MATH::CRotation::RotationOrder l_XSIRotationOrder = l_XSITransform.GetRotationOrder();

			switch (l_XSIRotationOrder)
			{
				case MATH::CRotation::siXZY:
					l_pFTKModel->XSITransform()->SetRotationOrder(CSLTemplate::SI_XZY);
					break;
				case MATH::CRotation::siYXZ:
					l_pFTKModel->XSITransform()->SetRotationOrder(CSLTemplate::SI_YXZ);
					break;
				case MATH::CRotation::siYZX:
					l_pFTKModel->XSITransform()->SetRotationOrder(CSLTemplate::SI_YZX);
					break;
				case MATH::CRotation::siZXY:
					l_pFTKModel->XSITransform()->SetRotationOrder(CSLTemplate::SI_ZXY);
					break;
				case MATH::CRotation::siZYX:
					l_pFTKModel->XSITransform()->SetRotationOrder(CSLTemplate::SI_ZYX);
					break;
				case MATH::CRotation::siXYZ:
				default:
					l_pFTKModel->XSITransform()->SetRotationOrder(CSLTemplate::SI_XYZ);
					break;
			}

			CParameterRefArray ksParams = l_XSIKineState.GetParameters();
			
			// scaling type
			l_pFTKModel->XSITransform()->SetHierarchicalScaling(ksParams.GetValue(L"siscaling"));

			// position			
			l_FTKVector.m_fX = ksParams.GetValue(L"posx");
			l_FTKVector.m_fY = ksParams.GetValue(L"posy");
			l_FTKVector.m_fZ = ksParams.GetValue(L"posz");
			l_pFTKModel->XSITransform()->SetTranslation(l_FTKVector);

			// rotation
			l_FTKVector.m_fX = in_pContext->AngleToFTK(ksParams.GetValue(L"rotx"), false);
			l_FTKVector.m_fY = in_pContext->AngleToFTK(ksParams.GetValue(L"roty"), false);
			l_FTKVector.m_fZ = in_pContext->AngleToFTK(ksParams.GetValue(L"rotz"), false);
			l_pFTKModel->XSITransform()->SetEulerRotation(l_FTKVector);

			// scaling
			l_FTKVector.m_fX = ksParams.GetValue(L"sclx");
			l_FTKVector.m_fY = ksParams.GetValue(L"scly");
			l_FTKVector.m_fZ = ksParams.GetValue(L"sclz");
			l_pFTKModel->XSITransform()->SetScale(l_FTKVector);

			// shear
			l_FTKVector.m_fX = in_pContext->AngleToFTK(ksParams.GetValue(L"sclorix"), false);
			l_FTKVector.m_fY = in_pContext->AngleToFTK(ksParams.GetValue(L"scloriy"), false);
			l_FTKVector.m_fZ = in_pContext->AngleToFTK(ksParams.GetValue(L"scloriz"), false);
			l_pFTKModel->XSITransform()->SetShear(l_FTKVector);

			// pivot scaling
			l_FTKVector.m_fX = ksParams.GetValue(L"psclx");
			l_FTKVector.m_fY = ksParams.GetValue(L"pscly");
			l_FTKVector.m_fZ = ksParams.GetValue(L"psclz");
			l_pFTKModel->XSITransform()->SetPivotScale(l_FTKVector);

			// pivot rotation
			l_FTKVector.m_fX = in_pContext->AngleToFTK(ksParams.GetValue(L"protx"), false);
			l_FTKVector.m_fY = in_pContext->AngleToFTK(ksParams.GetValue(L"proty"), false);
			l_FTKVector.m_fZ = in_pContext->AngleToFTK(ksParams.GetValue(L"protz"), false);
			l_pFTKModel->XSITransform()->SetPivotRotation(l_FTKVector);

			// pivot position
			l_FTKVector.m_fX = ksParams.GetValue(L"pposx");
			l_FTKVector.m_fY = ksParams.GetValue(L"pposy");
			l_FTKVector.m_fZ = ksParams.GetValue(L"pposz");
			l_pFTKModel->XSITransform()->SetPivotPosition(l_FTKVector);

			// pivot compensation scaling
			l_FTKVector.m_fX = ksParams.GetValue(L"pcsclx");
			l_FTKVector.m_fY = ksParams.GetValue(L"pcscly");
			l_FTKVector.m_fZ = ksParams.GetValue(L"pcsclz");
			l_pFTKModel->XSITransform()->SetPivotCompScale(l_FTKVector);

			// pivot compensation rotation
			l_FTKVector.m_fX = in_pContext->AngleToFTK(ksParams.GetValue(L"pcrotx"), false);
			l_FTKVector.m_fY = in_pContext->AngleToFTK(ksParams.GetValue(L"pcroty"), false);
			l_FTKVector.m_fZ = in_pContext->AngleToFTK(ksParams.GetValue(L"pcrotz"), false);
			l_pFTKModel->XSITransform()->SetPivotCompRotation(l_FTKVector);

			// pivot compensation position
			l_FTKVector.m_fX = ksParams.GetValue(L"pcposx");
			l_FTKVector.m_fY = ksParams.GetValue(L"pcposy");
			l_FTKVector.m_fZ = ksParams.GetValue(L"pcposz");
			l_pFTKModel->XSITransform()->SetPivotCompPosition(l_FTKVector);

			// neutral pose scaling
			l_FTKVector.m_fX = ksParams.GetValue(L"nsclx");
			l_FTKVector.m_fY = ksParams.GetValue(L"nscly");
			l_FTKVector.m_fZ = ksParams.GetValue(L"nsclz");
			l_pFTKModel->XSITransform()->SetNeutralPoseScale(l_FTKVector);

			// neutral pose rotation
			l_FTKVector.m_fX = in_pContext->AngleToFTK(ksParams.GetValue(L"nrotx"), false);
			l_FTKVector.m_fY = in_pContext->AngleToFTK(ksParams.GetValue(L"nroty"), false);
			l_FTKVector.m_fZ = in_pContext->AngleToFTK(ksParams.GetValue(L"nrotz"), false);
			l_pFTKModel->XSITransform()->SetNeutralPoseRotation(l_FTKVector);

			// neutral pose position
			l_FTKVector.m_fX = ksParams.GetValue(L"nposx");
			l_FTKVector.m_fY = ksParams.GetValue(L"nposy");
			l_FTKVector.m_fZ = ksParams.GetValue(L"nposz");
			l_pFTKModel->XSITransform()->SetNeutralPosePosition(l_FTKVector);

			// neutral pose shear
			l_FTKVector.m_fX = in_pContext->AngleToFTK(ksParams.GetValue(L"nsclorix"), false);
			l_FTKVector.m_fY = in_pContext->AngleToFTK(ksParams.GetValue(L"nscloriy"), false);
			l_FTKVector.m_fZ = in_pContext->AngleToFTK(ksParams.GetValue(L"nscloriz"), false);
			l_pFTKModel->XSITransform()->SetNeutralPoseShear(l_FTKVector);

			// local and global matrices
			l_pFTKModel->XSITransform()->ComputeLocalMatrix();
			l_pFTKModel->XSITransform()->ComputeGlobalMatrix();
	
			CSLXSILimit *l_pFTKLimit;

			// posx limit
			l_pFTKLimit = l_pFTKModel->XSITransform()->AddXSILimit(&(l_pFTKModel->XSITransform()->TranslationProxy()->X()));
			l_pFTKLimit->SetParameterName("posx");
			l_pFTKLimit->SetMaximum(ksParams.GetValue(L"posxmaxlimit"));
			l_pFTKLimit->SetMaximumActive(ksParams.GetValue(L"posxmaxactive"));
			l_pFTKLimit->SetMinimum(ksParams.GetValue(L"posxminlimit"));
			l_pFTKLimit->SetMinimumActive(ksParams.GetValue(L"posxminactive"));

			// posy limit
			l_pFTKLimit = l_pFTKModel->XSITransform()->AddXSILimit(&(l_pFTKModel->XSITransform()->TranslationProxy()->Y()));
			l_pFTKLimit->SetParameterName("posy");
			l_pFTKLimit->SetMaximum(ksParams.GetValue(L"posymaxlimit"));
			l_pFTKLimit->SetMaximumActive(ksParams.GetValue(L"posymaxactive"));
			l_pFTKLimit->SetMinimum(ksParams.GetValue(L"posyminlimit"));
			l_pFTKLimit->SetMinimumActive(ksParams.GetValue(L"posyminactive"));

			// posz limit
			l_pFTKLimit = l_pFTKModel->XSITransform()->AddXSILimit(&l_pFTKModel->XSITransform()->TranslationProxy()->Z());
			l_pFTKLimit->SetParameterName("posz");
			l_pFTKLimit->SetMaximum(ksParams.GetValue(L"poszmaxlimit"));
			l_pFTKLimit->SetMaximumActive(ksParams.GetValue(L"poszmaxactive"));
			l_pFTKLimit->SetMinimum(ksParams.GetValue(L"poszminlimit"));
			l_pFTKLimit->SetMinimumActive(ksParams.GetValue(L"poszminactive"));

			// rotx limit
			l_pFTKLimit = l_pFTKModel->XSITransform()->AddXSILimit(&(l_pFTKModel->XSITransform()->RotationProxy()->X()));
			l_pFTKLimit->SetParameterName("rotx");
			l_pFTKLimit->SetMaximum(in_pContext->AngleToFTK(ksParams.GetValue(L"rotxmaxlimit"), false));
			l_pFTKLimit->SetMaximumActive(ksParams.GetValue(L"rotxmaxactive"));
			l_pFTKLimit->SetMinimum(in_pContext->AngleToFTK(ksParams.GetValue(L"rotxminlimit"), false));
			l_pFTKLimit->SetMinimumActive(ksParams.GetValue(L"rotxminactive"));

			// roty limit
			l_pFTKLimit = l_pFTKModel->XSITransform()->AddXSILimit(&(l_pFTKModel->XSITransform()->RotationProxy()->Y()));
			l_pFTKLimit->SetParameterName("roty");
			l_pFTKLimit->SetMaximum(in_pContext->AngleToFTK(ksParams.GetValue(L"rotymaxlimit"), false));
			l_pFTKLimit->SetMaximumActive(ksParams.GetValue(L"rotymaxactive"));
			l_pFTKLimit->SetMinimum(in_pContext->AngleToFTK(ksParams.GetValue(L"rotyminlimit"), false));
			l_pFTKLimit->SetMinimumActive(ksParams.GetValue(L"rotyminactive"));

			// rotz limit
			l_pFTKLimit = l_pFTKModel->XSITransform()->AddXSILimit(&(l_pFTKModel->XSITransform()->RotationProxy()->Z()));
			l_pFTKLimit->SetParameterName("rotz");
			l_pFTKLimit->SetMaximum(in_pContext->AngleToFTK(ksParams.GetValue(L"rotzmaxlimit"), false));
			l_pFTKLimit->SetMaximumActive(ksParams.GetValue(L"rotzmaxactive"));
			l_pFTKLimit->SetMinimum(in_pContext->AngleToFTK(ksParams.GetValue(L"rotzminlimit"), false));
			l_pFTKLimit->SetMinimumActive(ksParams.GetValue(L"rotzminactive"));


			LONG Format = in_pContext->exportproperty().GetParameterValue (L"format");

			if((Format != DOTXSI_FORMAT_5_0) && (Format != DOTXSI_FORMAT_5_0_BINARY))
			{
				// Check for polymatricks 
				CRefArray l_NestedComponents = l_XSIKineState.GetNestedObjects();
				CustomOperator op;

				int loop, count = l_NestedComponents.GetCount();
				for(loop = count-1; loop >= 0; loop--)
				{
					op = l_NestedComponents[loop];
					if(op.IsValid() && (op.GetType() == L"polymatricks"))
					{
						CSLXSIPolymatricks *l_pPolymatricks = l_pFTKModel->XSITransform()->CreatePolymatricks();

						// grab the parameters
						CParameterRefArray l_Parameters = op.GetParameters();
						int loop2, count2 = l_Parameters.GetCount();
						for(loop2 = 0; loop2 < count2; loop2++)
						{
							Parameter l_Param = l_Parameters[loop2];
							CString l_ParamName = l_Param.GetScriptName();
							if(l_Param.GetValue() == L"translate")
							{
								CSLXSITranslate* l_pTranslate = l_pPolymatricks->AddXSITranslate();
								l_pTranslate->SetName((char*)l_ParamName.GetAsciiString());
								CSIBCVector3D l_Vector;
								l_Vector.Set
								(
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_X"),
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_Y"),
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_Z")
								);
								l_pTranslate->SetTranslation(l_Vector);
							}
							else if(l_Param.GetValue() == L"rotate")
							{
								CSLXSIRotate* l_pRotate = l_pPolymatricks->AddXSIRotate();
								l_pRotate->SetName((char*)l_ParamName.GetAsciiString());
								CSIBCVector3D l_Vector;
								l_Vector.Set
								(
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_X"),
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_Y"),
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_Z")
								);
								l_pRotate->SetAxis(l_Vector);
								l_pRotate->SetAngle((float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_Angle"));
							}
							else if(l_Param.GetValue() == L"scale")
							{
								CSLXSIScale* l_pScale = l_pPolymatricks->AddXSIScale();
								l_pScale->SetName((char*)l_ParamName.GetAsciiString());
								CSIBCVector3D l_Vector;
								l_Vector.Set
								(
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_X"),
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_Y"),
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_Z")
								);
								l_pScale->SetScale(l_Vector);
							}
							else if(l_Param.GetValue() == L"shear")
							{
								CSLXSIShear* l_pShear = l_pPolymatricks->AddXSIShear();
								l_pShear->SetName((char*)l_ParamName.GetAsciiString());
								CSIBCVector3D l_Vector;
								l_Vector.Set
								(
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_X1"),
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_Y1"),
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_Z1")
								);
								l_pShear->SetFirstAxis(l_Vector);
								l_Vector.Set
								(
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_X2"),
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_Y2"),
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_Z2")
								);
								l_pShear->SetSecondAxis(l_Vector);
								l_pShear->SetAngle((float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_Angle"));
							}
							else if(l_Param.GetValue() == L"lookat")
							{
								CSLXSILookat* l_pLookat = l_pPolymatricks->AddXSILookat();
								l_pLookat->SetName((char*)l_ParamName.GetAsciiString());
								CSIBCVector3D l_Vector;
								l_Vector.Set
								(
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_PX"),
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_PY"),
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_PZ")
								);
								l_pLookat->SetPosition(l_Vector);
								l_Vector.Set
								(
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_IX"),
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_IY"),
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_IZ")
								);
								l_pLookat->SetInterest(l_Vector);
								l_Vector.Set
								(
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_UX"),
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_UY"),
									(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_UZ")
								);
								l_pLookat->SetUpvector(l_Vector);
							}
							else if(l_Param.GetValue() == L"matrix")
							{
								CSLXSIMatrix* l_pMatrix = l_pPolymatricks->AddXSIMatrix();
								l_pMatrix->SetName((char*)l_ParamName.GetAsciiString());
								CSIBCMatrix4x4 l_Matrix;
								l_Matrix.Set(0,0,(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_00"));
								l_Matrix.Set(1,0,(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_01"));
								l_Matrix.Set(2,0,(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_02"));
								l_Matrix.Set(3,0,(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_03"));
								l_Matrix.Set(0,1,(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_10"));
								l_Matrix.Set(1,1,(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_11"));
								l_Matrix.Set(2,1,(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_12"));
								l_Matrix.Set(3,1,(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_13"));
								l_Matrix.Set(0,2,(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_20"));
								l_Matrix.Set(1,2,(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_21"));
								l_Matrix.Set(2,2,(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_22"));
								l_Matrix.Set(3,2,(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_23"));
								l_Matrix.Set(0,3,(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_30"));
								l_Matrix.Set(1,3,(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_31"));
								l_Matrix.Set(2,3,(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_32"));
								l_Matrix.Set(3,3,(float)l_Parameters.GetValue(l_Param.GetScriptName() + L"_33"));
								l_pMatrix->SetMatrix(l_Matrix);
							}
						}
						break;
					}
				}
			}
		}

		// Fill in the base pose now
		if( (l_pFTKModel->Template() != NULL) && (l_XSIModel.HasStaticKinematicState()) )
		{
			// this code gets the static kinematics transform AKA the base pose
			StaticKinematicState l_XSIKineState = l_XSIModel.GetStaticKinematicState();
			assert(l_XSIKineState.IsValid());
		
			CSIBCVector3D l_FTKVector;
			Parameter l_XSIParameter;
			l_pFTKModel->CreateXSIBasePose();

			// this code gets the local kinematics transform
			MATH::CTransformation		l_XSITransform = l_XSIKineState.GetTransform();

			// check if we must flip the base pose

			if(in_pContext->ftkscene()->Root()->FindUserData("Z_UP"))
			{
				MATH::CTransformation l_ZUPConverter;
				l_ZUPConverter.SetRotationFromXYZAnglesValues(1.570796f, 0.0f, 0.0f);
				l_XSITransform.MulInPlace(l_ZUPConverter);
			}

			double						l_dX, l_dY, l_dZ;

			// position
			l_XSITransform.GetTranslationValues(l_dX, l_dY, l_dZ);
			l_FTKVector.Set((float) l_dX, (float) l_dY, (float) l_dZ);
			l_pFTKModel->GetXSIBasePose()->SetTranslation(l_FTKVector);

			// rotation
			l_XSITransform.GetRotation().GetXYZAngles(l_dX, l_dY, l_dZ);
			l_FTKVector.Set((float) in_pContext->AngleToFTK((float)l_dX, true), (float) in_pContext->AngleToFTK((float)l_dY, true), (float) in_pContext->AngleToFTK((float)l_dZ, true));
			l_pFTKModel->GetXSIBasePose()->SetEulerRotation(l_FTKVector);

			// scaling

			l_XSITransform.GetScalingValues(l_dX, l_dY, l_dZ);
			l_FTKVector.Set((float) l_dX, (float) l_dY, (float) l_dZ);
			l_pFTKModel->GetXSIBasePose()->SetScale(l_FTKVector);

			l_pFTKModel->GetXSIBasePose()->ComputeMatrix();
		
		}

		// CrySpecific: If it's a CryExportNode, append the "Property" parameters to it's name.
		if (IsCryExportNode(l_XSIModel))
		{
			CustomProperty l_Properties(l_XSIModel.GetProperties().GetItem(L"ExportProperties"));

			if (l_Properties.IsValid())
			{
				LONG l_Filetype = GetCryExportNodeExportFiletype(l_XSIModel);

				CString l_FiletypeString = CryFileTypeToString(l_Filetype);

				CString l_Filename = l_Properties.GetParameterValue(L"Filename");
				bool l_bExportable = l_Properties.GetParameterValue(L"Exportable");
				bool l_bMergeObject = l_Properties.GetParameterValue(L"MergeObjects");

				CString l_Exportable;
				CString l_MergeObject;

				if (l_bExportable)
				{
					l_Exportable = L"DoExport";
				}
				else
				{
					l_Exportable = L"NoExport";
				}

				if (l_bMergeObject)
				{
					l_MergeObject = L"MergeNodes";
				}
				else
				{
					l_MergeObject = L"NoMerge";
				}

				CString l_Name = l_XSIModel.GetName();

				char l_strNewNodename[MAX_PATH];
				char *l_strNodename = (char*)(l_Name.GetAsciiString());
				char *l_strCEN = strstr(l_strNodename, "CryExportNode");
				if (l_strCEN)
				{
					strcpy(l_strNewNodename, l_strCEN);
					strcat(l_strNewNodename, "_");
					char l_strPost[MAX_PATH];
					memset(l_strPost, 0, MAX_PATH);
					memcpy(l_strPost, l_strNodename, l_strCEN-l_strNodename-1);
					strcat(l_strNewNodename, l_strPost);
				}
				l_Name.PutAsciiString(l_strNewNodename);

				l_Name += L"-";
				l_Name += l_FiletypeString;
				l_Name += L"-";
				l_Name += l_Filename;
				l_Name += L"-";
				l_Name += l_Exportable;
				l_Name += L"-";
				l_Name += l_MergeObject;

				l_pFTKModel->SetName((char*)l_Name.GetAsciiString());
			}
		}
		// CrySpecific: If it's under a CryExportNode, append the "NodeProperty" and "ObjectProperty" parameters to it's name.
		else
		{
			CustomProperty l_JointProperties(l_XSIModel.GetProperties().GetItem(L"JointProperties"));
			CustomProperty l_ObjectProperties(l_XSIModel.GetProperties().GetItem(L"ObjectProperties"));

			CRefArray l_Nodes;
			CStringArray l_NodeNames;
			GetNodesUnderNode( l_XSIModel, l_Nodes );
			GetNodeNamesUnderNode( l_CryExportNode, l_NodeNames );

			std::vector<std::wstring> l_Properties;
			if (l_ObjectProperties.IsValid())
			{
				CustomProperty l_CENProperties(l_CryExportNode.GetProperties().GetItem(L"ExportProperties"));
				CString l_Filename;
				if (l_CENProperties.IsValid())
				{
					l_Filename = l_CENProperties.GetParameterValue(L"Filename");
				}

/*
				bool l_Entity = l_ObjectProperties.GetParameterValue(L"Entity");
				bool l_Box = l_ObjectProperties.GetParameterValue(L"Box");
				int l_Mass = l_ObjectProperties.GetParameterValue(L"Mass");
				int l_Density = l_ObjectProperties.GetParameterValue(L"Density");
*/
				CString l_Props = l_ObjectProperties.GetParameterValue(L"Props");

				ConvertObjectProperties(l_Props, l_Properties, l_NodeNames, l_XSIModel.GetName());

/*
				if (l_Box)
				{
					l_Properties += L"--PRbox";
				}
				if (l_Entity)
				{
					l_Properties += L"--PRentity";
				}
				if (l_Mass != -1)
				{
					l_Properties += L"--PRmass_";
					l_Properties += CValue(l_Mass);
				}
				if (l_Density != -1)
				{
					l_Properties += L"--PRdensity_";
					l_Properties += CValue(l_Density);
				}

				//automatic pieces selection
				CValueArray l_Items;
				if (l_XSIModel.GetName() != L"")
				{
					for (int i = 0; i < l_Nodes.GetCount(); i++)
					{
						SIObject l_Node = l_Nodes.GetItem(i);
						CString l_Nodename = l_Node.GetName();
						const char *l_strNodename = l_Nodename.GetAsciiString();
						if (l_strNodename[0] == '-' && !strstr(l_strNodename, "-LOD"))
						{
							l_Properties += L"--PRpieces_";
							l_Properties += l_Nodename;
							l_Properties += L"_";
							l_Properties += l_Filename;
						}
					}
				}
*/

/*
				CString l_PiecesList = l_ObjectProperties.GetParameterValue(L"Pieces");
				if (l_PiecesList.Length() == 0 || l_PiecesList == L"-1")
				{
				}
				else
				{
					CStringArray l_SplitList = l_PiecesList.Split(L";");

					for (LONG i = 0; i < l_SplitList.GetCount(); i++)
					{
						bool l_bFound = false;
						for (LONG j = 0; j < l_Nodes.GetCount(); j++)
						{
							SIObject l_Node = l_Nodes.GetItem(j);
							CString l_Nodename = l_Node.GetName();

							if (l_SplitList[i] == l_Nodename)
								l_bFound = true;
						}
						if (!l_bFound)
							continue;

						l_Properties += L"--PRpieces_";
						l_Properties += l_SplitList[i];
						l_Properties += L"_";
						l_Properties += l_Filename;
					}
				}
*/
			}

			if (l_JointProperties.IsValid())
			{
/*
				int l_Limit = l_JointProperties.GetParameterValue(L"Limit");
				int l_Bend = l_JointProperties.GetParameterValue(L"Bend");
				int l_Twist = l_JointProperties.GetParameterValue(L"Twist");
				int l_Pull = l_JointProperties.GetParameterValue(L"Pull");
				int l_Push = l_JointProperties.GetParameterValue(L"Push");
				int l_Shift = l_JointProperties.GetParameterValue(L"Shift");

				if (l_Limit != -1)
				{
					l_Properties += L"--PRlimit_";
					l_Properties += CValue(l_Limit);
				}
				if (l_Bend != -1)
				{
					l_Properties += L"--PRbend_";
					l_Properties += CValue(l_Bend);
				}
				if (l_Twist != -1)
				{
					l_Properties += L"--PRtwist_";
					l_Properties += CValue(l_Twist);
				}
				if (l_Pull != -1)
				{
					l_Properties += L"--PRpull_";
					l_Properties += CValue(l_Pull);
				}
				if (l_Push != -1)
				{
					l_Properties += L"--PRpush_";
					l_Properties += CValue(l_Push);
				}
				if (l_Shift != -1)
				{
					l_Properties += L"--PRshift_";
					l_Properties += CValue(l_Shift);
				}
*/

				CString l_Props = l_JointProperties.GetParameterValue(L"Props");

				if (l_Props.Length() != 0 && l_Props != L"-1")
				{
					CStringArray l_SplitList = l_Props.Split(L"\n");
					for (LONG i = 0; i < l_SplitList.GetCount(); i++)
					{
						CString l_Row = l_SplitList[i];
						const char *l_strRow = l_Row.GetAsciiString();
						std::wstring text;
						for (DWORD j = 0; j < strlen(l_strRow); j++)
						{
							if (l_strRow[j] != 0x0d && l_strRow[j] != 0x0a)
								text.push_back(l_strRow[j]);
						}
						l_Properties.push_back(text);
					}
				}
			}

			// Read joint limits.
			{
				CRefArray l_Parameters = l_XSIModel.GetKinematics().GetLocal().GetParameters();

				// Since XSI limits are relative to the current node position, whereas the engine
				// expects them relative to the parent, we have to subtract the local euler rotations
				// from the limits.
				KinematicState l_XSIKineState = l_XSIModel.GetKinematics().GetLocal();
				MATH::CTransformation l_XSITransform = l_XSIKineState.GetTransform();
				float offsets[] = {float(l_XSITransform.GetRotX()), float(l_XSITransform.GetRotY()), float(l_XSITransform.GetRotZ())};

				typedef std::map<std::string, JointParameterEntry> NameJointParameterMap;
				NameJointParameterMap nameJointParameterMap;
				nameJointParameterMap.insert(std::make_pair("rotxminlimit", JointParameterEntry(L"xmin", offsets[0])));
				nameJointParameterMap.insert(std::make_pair("rotyminlimit", JointParameterEntry(L"ymin", offsets[1])));
				nameJointParameterMap.insert(std::make_pair("rotzminlimit", JointParameterEntry(L"zmin", offsets[2])));
				nameJointParameterMap.insert(std::make_pair("rotxmaxlimit", JointParameterEntry(L"xmax", offsets[0])));
				nameJointParameterMap.insert(std::make_pair("rotymaxlimit", JointParameterEntry(L"ymax", offsets[1])));
				nameJointParameterMap.insert(std::make_pair("rotzmaxlimit", JointParameterEntry(L"zmax", offsets[2])));

				for (int paramIndex = 0, paramCount = l_Parameters.GetCount(); paramIndex < paramCount; ++paramIndex)
				{
					Parameter l_Parameter = l_Parameters[paramIndex];
					std::string paramName = l_Parameter.GetScriptName().GetAsciiString();
					std::transform(paramName.begin(), paramName.end(), paramName.begin(), std::tolower);
					NameJointParameterMap::iterator jointEntryPos = nameJointParameterMap.find(paramName);
					if (jointEntryPos != nameJointParameterMap.end())
					{
						float value = (float)l_Parameter.GetValue(0.0f);
						const JointParameterEntry& jointEntry = (*jointEntryPos).second;
						value += jointEntry.localOffset;
						wchar_t buffer[1000];
						swprintf(buffer, L"%d", int(value));
						l_Properties.push_back(jointEntry.propName + L"=" + buffer);
					}
				}

			}


			CString l_Name;
			l_Name.PutAsciiString(l_pFTKModel->GetName());

			if (!l_Properties.empty())
			{
				l_Name += L"--PRprops";
				for (int i = 0, count = int(l_Properties.size()); i < count; ++i)
				{
					std::wstring safeProperty = l_Properties[i];
					std::replace(safeProperty.begin(), safeProperty.end(), ' ', '*');
					l_Name += (L"_" + safeProperty).c_str();
				}
				l_Name += L"__";
			}

			l_pFTKModel->SetName((char*)l_Name.GetAsciiString());
		}

		// CrySpecific: If it's a Model, don't set it's global material, we don't need it.
		if (l_pFTKModel->Primitive() && (l_pFTKModel->Primitive()->Type() == CSLTemplate::XSI_MESH))
		{
			// Then the material
			Material l_XSIMaterial = l_XSIModel.GetMaterial();
			if(l_XSIMaterial.IsValid() && (in_pContext->ftkscene()->GetMaterialLibrary()))
			{
				// find the corresponding material in the FTK material library
				for(int loop = 0; loop < in_pContext->ftkscene()->GetMaterialLibrary()->GetMaterialCount(); loop++)
				{
					CSLBaseMaterial *l_pFTKMaterial = in_pContext->ftkscene()->GetMaterialLibrary()->GetMaterialList()[loop];
					CSIBCUserData *l_pUserData = l_pFTKMaterial->FindUserData("CREF");

					if(l_pUserData)
					{
						CRef *l_pCRef = (CRef*)(l_pUserData->GetData());
						Material l_XSIMaterialToCompareWith = (Material) *l_pCRef;

						if(l_XSIMaterialToCompareWith == l_XSIMaterial)
						{
							// we found the material, set it.
							if(l_pFTKModel->GlobalMaterial() == NULL)
								l_pFTKModel->AddGlobalMaterial();

							if(l_pFTKModel->GlobalMaterial() != NULL)
							{
								l_pFTKModel->GlobalMaterial()->SetMaterial(l_pFTKMaterial);

								// CrySpecific: Flag material as "USED"
								CSIBCUserData *l_pUserData = l_pFTKMaterial->FindUserData("USED");
								
								if (!l_pUserData)
								{
									l_pFTKMaterial->AttachUserData("USED", 0);
								}

								// we do not care about node or branch material for now, we'll always set material as node.
								l_pFTKModel->GlobalMaterial()->SetPropagationType(CSLGlobalMaterial::SI_NODE);
							}

							break;
						}
					}
				}
			}
		}
	}

	return status;
}

wchar_t *CModelInfoFromXSI::GetClassID(){return L"CModelInfoFromXSI";}

/**************************************************************************************
CModelInfoToXSI
**************************************************************************************/
CModelInfoToXSI::CModelInfoToXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CModelInfoToXSI::~CModelInfoToXSI() {};

CStatus CModelInfoToXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;
	X3DObject l_XSIModel = (X3DObject) *io_pXSIModel;

	if((*io_pFTKModel != NULL) && (!l_XSIModel.IsEqualTo(in_pContext->scene().GetRoot())))
	{
		CSLModel *l_pFTKModel = (CSLModel *) *io_pFTKModel;
		CTime	  l_Time;

		// Get the name 
		CString l_name;
		l_name.PutAsciiString(l_pFTKModel->GetName());
		l_XSIModel.PutName(l_name);			

		// Get the visility, it's stored in the visibility.viewvis property
		if((l_pFTKModel->Visibility() != NULL) && (!l_pFTKModel->Visibility()->GetVisibility()))
		{			
			Property l_Visibility = l_XSIModel.GetProperties().GetItem(L"Visibility");
			l_Visibility.PutParameterValue(L"viewvis", l_pFTKModel->Visibility()->GetVisibility() ? true : false);
		}

		// transforms for everyone except lights (for now)
		if(	!l_pFTKModel->Primitive() || (l_pFTKModel->Primitive() && 
			(l_pFTKModel->Primitive()->Type() != CSLTemplate::SI_POINT_LIGHT) && 
			(l_pFTKModel->Primitive()->Type() != CSLTemplate::SI_DIRECTIONAL_LIGHT) && 
			(l_pFTKModel->Primitive()->Type() != CSLTemplate::SI_INFINITE_LIGHT) && 
			(l_pFTKModel->Primitive()->Type() != CSLTemplate::SI_SPOT_LIGHT)))
		{

			// Fill in the transform now
			if(l_pFTKModel->XSITransform() != NULL)
			{
				// this code sets the local kinematics transform
				KinematicState l_XSIKineState = l_XSIModel.GetKinematics().GetLocal();

				if(l_pFTKModel->XSITransform()->GetPolymatricks() == NULL)
				{

					CSIBCVector3D l_FTKVector;
					MATH::CVector3 l_XSIVector;

					MATH::CTransformation l_Transformation;

					// scaling type
					CParameterRefArray ksParams = l_XSIKineState.GetParameters();
					ksParams.PutValue(L"siscaling",l_pFTKModel->XSITransform()->GetHierarchicalScaling() ? true : false);

					// do not try the following it will not work because the GetTransform returns a transient object
					// l_XSIModel.GetKinematics().GetLocal().GetTransform().SetMatrix4(l_XSIMatrix);
					// instead the right way to set the transform is as follows
					int l_iRotationOrder = (int)l_pFTKModel->XSITransform()->GetRotationOrder();
					// if out of bound
					if (l_iRotationOrder < (int)CSLTemplate::SI_XYZ || l_iRotationOrder > (int)CSLTemplate::SI_ZYX)
						l_iRotationOrder = 0;
					// Set the rotation order
					MATH::CRotation::RotationOrder l_CRotationOrder = (MATH::CRotation::RotationOrder) l_iRotationOrder;
					
					ksParams.PutValue(L"rotorder",l_CRotationOrder);

					if ( l_pFTKModel->XSITransform()->CanDecompose() )
					{

						// scaling
						l_FTKVector = l_pFTKModel->XSITransform()->GetScale();
						l_Transformation.SetScalingFromValues(l_FTKVector.m_fX, l_FTKVector.m_fY, l_FTKVector.m_fZ);

						// rotation
						// l_Transformation.PutRotationOrder(MATH::CRotation::siXYZ);
						l_FTKVector = l_pFTKModel->XSITransform()->GetEulerRotation();
						l_XSIVector.Set(in_pContext->AngleFromFTK(l_FTKVector.m_fX, true), in_pContext->AngleFromFTK(l_FTKVector.m_fY, true), in_pContext->AngleFromFTK(l_FTKVector.m_fZ, true));

						l_Transformation.SetRotationFromXYZAnglesValues(l_XSIVector.GetX(), l_XSIVector.GetY(), l_XSIVector.GetZ(), 
							l_CRotationOrder);

						// shear
						l_FTKVector = l_pFTKModel->XSITransform()->GetShear();
						l_Transformation.SetScalingOrientationFromXYZAngles
						(
							in_pContext->AngleFromFTK(l_FTKVector.m_fX, true), 
							in_pContext->AngleFromFTK(l_FTKVector.m_fY, true), 
							in_pContext->AngleFromFTK(l_FTKVector.m_fZ, true)
						);

						// position
						l_FTKVector = l_pFTKModel->XSITransform()->GetTranslation();
						l_Transformation.SetTranslationFromValues(l_FTKVector.m_fX, l_FTKVector.m_fY, l_FTKVector.m_fZ);


					} else {
						CSIBCMatrix4x4 xmat = l_pFTKModel->XSITransform()->GetMatrix ();

						MATH::CMatrix4 mat;
						
						float * l_pRaw = xmat.Raw ();
						mat.Set(l_pRaw[0], l_pRaw[1],l_pRaw[2],l_pRaw[3],
								l_pRaw[4], l_pRaw[5],l_pRaw[6],l_pRaw[7],
								l_pRaw[8], l_pRaw[9],l_pRaw[10],l_pRaw[11],
								l_pRaw[12], l_pRaw[13],l_pRaw[14],l_pRaw[15] );
						

						for (int r=0;r<4;r++)
							for (int c=0;c<4;c++)
								mat.SetValue ( r,c,xmat.Get (r,c));
								
						l_Transformation.SetMatrix4 ( mat );
						
					}
					
					l_XSIKineState.PutTransform(l_Transformation);

					// disable pivot compensation, we assume that we might want to force these values
					// the user will have to enable them back in order to make them auto update
					ksParams.PutValue(L"pivotcompactive",false);

					// pivot scaling
					l_FTKVector = l_pFTKModel->XSITransform()->GetPivotScale();

					// let's optimize a bit here and see if we have default values 
					if((l_FTKVector.m_fX != 1.0f) || (l_FTKVector.m_fY != 1.0f) || (l_FTKVector.m_fZ != 1.0f))
					{
						ksParams.PutValue(L"psclx",l_FTKVector.m_fX);
						ksParams.PutValue(L"pscly",l_FTKVector.m_fY);
						ksParams.PutValue(L"psclz",l_FTKVector.m_fZ);
					}

					// pivot rotation
					l_FTKVector = l_pFTKModel->XSITransform()->GetPivotRotation();

					if((l_FTKVector.m_fX != 0.0f) || (l_FTKVector.m_fY != 0.0f) || (l_FTKVector.m_fZ != 0.0f))
					{
						ksParams.PutValue(L"protx",in_pContext->AngleFromFTK(l_FTKVector.m_fX, false));
						ksParams.PutValue(L"proty",in_pContext->AngleFromFTK(l_FTKVector.m_fY, false));
						ksParams.PutValue(L"protz",in_pContext->AngleFromFTK(l_FTKVector.m_fZ, false));
					}

					// pivot position
					l_FTKVector = l_pFTKModel->XSITransform()->GetPivotPosition();

					if((l_FTKVector.m_fX != 0.0f) || (l_FTKVector.m_fY != 0.0f) || (l_FTKVector.m_fZ != 0.0f))
					{
						ksParams.PutValue(L"pposx",l_FTKVector.m_fX);
						ksParams.PutValue(L"pposy",l_FTKVector.m_fY);
						ksParams.PutValue(L"pposz",l_FTKVector.m_fZ);
					}

					// pivot compensation scaling
					l_FTKVector = l_pFTKModel->XSITransform()->GetPivotCompScale();

					if((l_FTKVector.m_fX != 1.0f) || (l_FTKVector.m_fY != 1.0f) || (l_FTKVector.m_fZ != 1.0f))
					{
						ksParams.PutValue(L"pcsclx",l_FTKVector.m_fX);
						ksParams.PutValue(L"pcscly",l_FTKVector.m_fY);
						ksParams.PutValue(L"pcsclz",l_FTKVector.m_fZ);
					}

					// pivot compensation rotation
					l_FTKVector = l_pFTKModel->XSITransform()->GetPivotCompRotation();

					if((l_FTKVector.m_fX != 0.0f) || (l_FTKVector.m_fY != 0.0f) || (l_FTKVector.m_fZ != 0.0f))
					{
						ksParams.PutValue(L"pcrotx",in_pContext->AngleFromFTK(l_FTKVector.m_fX, false));
						ksParams.PutValue(L"pcroty",in_pContext->AngleFromFTK(l_FTKVector.m_fY, false));
						ksParams.PutValue(L"pcrotz",in_pContext->AngleFromFTK(l_FTKVector.m_fZ, false));
					}

					// pivot compensation position
					l_FTKVector = l_pFTKModel->XSITransform()->GetPivotCompPosition();

					if((l_FTKVector.m_fX != 0.0f) || (l_FTKVector.m_fY != 0.0f) || (l_FTKVector.m_fZ != 0.0f))
					{
						ksParams.PutValue(L"pcposx",l_FTKVector.m_fX);
						ksParams.PutValue(L"pcposy",l_FTKVector.m_fY);
						ksParams.PutValue(L"pcposz",l_FTKVector.m_fZ);
					}

					// neutral pose scaling
					l_FTKVector = l_pFTKModel->XSITransform()->GetNeutralPoseScale();

					if((l_FTKVector.m_fX != 1.0f) || (l_FTKVector.m_fY != 1.0f) || (l_FTKVector.m_fZ != 1.0f))
					{
						ksParams.PutValue(L"nsclx",l_FTKVector.m_fX);
						ksParams.PutValue(L"nscly",l_FTKVector.m_fY);
						ksParams.PutValue(L"nsclz",l_FTKVector.m_fZ);
					}

					// neutral pose rotation
					l_FTKVector = l_pFTKModel->XSITransform()->GetNeutralPoseRotation();

					if((l_FTKVector.m_fX != 0.0f) || (l_FTKVector.m_fY != 0.0f) || (l_FTKVector.m_fZ != 0.0f))
					{
						ksParams.PutValue(L"nrotx",in_pContext->AngleFromFTK(l_FTKVector.m_fX, false));
						ksParams.PutValue(L"nroty",in_pContext->AngleFromFTK(l_FTKVector.m_fY, false));
						ksParams.PutValue(L"nrotz",in_pContext->AngleFromFTK(l_FTKVector.m_fZ, false));
					}

					// neutral pose position
					l_FTKVector = l_pFTKModel->XSITransform()->GetNeutralPosePosition();

					if((l_FTKVector.m_fX != 0.0f) || (l_FTKVector.m_fY != 0.0f) || (l_FTKVector.m_fZ != 0.0f))
					{
						ksParams.PutValue(L"nposx",l_FTKVector.m_fX);
						ksParams.PutValue(L"nposy",l_FTKVector.m_fY);
						ksParams.PutValue(L"nposz",l_FTKVector.m_fZ);
					}

					// neutral pose shear
					l_FTKVector = l_pFTKModel->XSITransform()->GetNeutralPoseShear();

					if((l_FTKVector.m_fX != 0.0f) || (l_FTKVector.m_fY != 0.0f) || (l_FTKVector.m_fZ != 0.0f))
					{
						ksParams.PutValue(L"nsclorix",in_pContext->AngleFromFTK(l_FTKVector.m_fX, false));
						ksParams.PutValue(L"nscloriy",in_pContext->AngleFromFTK(l_FTKVector.m_fY, false));
						ksParams.PutValue(L"nscloriz",in_pContext->AngleFromFTK(l_FTKVector.m_fZ, false));
					}

					// limits
					CSLXSILimit** l_pFTKLimits = l_pFTKModel->XSITransform()->GetXSILimitList();
					
					for (int loop = 0; loop < l_pFTKModel->XSITransform()->GetXSILimitCount(); loop++)
					{
						const SI_Char* l_pParamName = l_pFTKLimits[loop]->GetParameterName();

						CString l_ParamName;
						l_ParamName.PutAsciiString(l_pParamName);

						if(!l_pFTKLimits[loop]->GetMinimumActive() && (l_pFTKLimits[loop]->GetMinimum() != 0.0f))
						{
							ksParams.PutValue(l_ParamName + L"minactive",l_pFTKLimits[loop]->GetMinimumActive() ? true : false);
							ksParams.PutValue(l_ParamName + L"minlimit",l_pFTKLimits[loop]->GetMinimum());
						}

						if(!l_pFTKLimits[loop]->GetMaximumActive() && (l_pFTKLimits[loop]->GetMaximum() != 0.0f))
						{
							ksParams.PutValue(l_ParamName + L"maxactive",l_pFTKLimits[loop]->GetMaximumActive() ? true : false);
							ksParams.PutValue(l_ParamName + L"maxlimit",l_pFTKLimits[loop]->GetMaximum());
						}
					}
				}

				//sync polymatricks if it exists
				if((l_pFTKModel->XSITransform()->GetPolymatricks() != NULL) && (l_pFTKModel->GetEnvelopeCount() == 0))
				{

					// scaling type
					CParameterRefArray ksParams = l_XSIKineState.GetParameters();
					ksParams.PutValue(L"siscaling", false);

					CSLXSIPolymatricks *l_pPolyMatricks = l_pFTKModel->XSITransform()->GetPolymatricks();

					// build the list of arguments
					CValueArray l_Stack;

					int loop, count = l_pPolyMatricks->GetTransformNodeCount();
					for(loop = 0; loop < count; loop++)
					{
						CSLTemplate *l_pCurrentNode = l_pPolyMatricks->GetTransformNodes()[loop];
						CString l_NodeName;
						l_NodeName.PutAsciiString(l_pCurrentNode->GetName());

						if( l_pCurrentNode->Type() == CSLTemplate::XSI_TRANSLATE)
						{
							l_Stack.Add(L"translate");
						}
						else if( l_pCurrentNode->Type() == CSLTemplate::XSI_ROTATE)
						{
							l_Stack.Add(L"rotate");
						}
						else if( l_pCurrentNode->Type() == CSLTemplate::XSI_SCALE)
						{
							l_Stack.Add(L"scale");
						}
						else if( l_pCurrentNode->Type() == CSLTemplate::XSI_SHEAR)
						{
							l_Stack.Add(L"shear");
						}
						else if( l_pCurrentNode->Type() == CSLTemplate::XSI_LOOKAT)
						{
							l_Stack.Add(L"lookat");
						}
						else if( l_pCurrentNode->Type() == CSLTemplate::XSI_MATRIX)
						{
							l_Stack.Add(L"matrix");
						}

						l_Stack.Add(l_NodeName.GetWideString());
					}

					COMMANDS::Applypolymatricks(l_XSIModel.GetFullName() + L".kine.local", l_Stack); 

					// first, find the operator

					CRefArray l_NestedComponents = l_XSIKineState.GetNestedObjects();
					CustomOperator op;

					count = l_NestedComponents.GetCount();
					for(loop = 0; loop < count; loop++)
					{
						op = l_NestedComponents[loop];
						if(op.IsValid() && (op.GetType() == L"polymatricks"))
						{
							break;
						}
					}

					CParameterRefArray l_OpParams = op.GetParameters();

					// put the values now
					count = l_pPolyMatricks->GetTransformNodeCount();
					int l_NonameID = -1;

					for(loop = count-1; loop >= 0; loop--)
					{
						CSLTemplate *l_pCurrentNode = l_pPolyMatricks->GetTransformNodes()[loop];
						CString l_NodeName;
						l_NodeName.PutAsciiString(l_pCurrentNode->GetName());

						if(( l_NodeName == L"" ) || (l_NodeName.IsEmpty()))
						{
							l_NonameID++;
						}
					}

					for(loop = count-1; loop >= 0; loop--)
					{
						CSLTemplate *l_pCurrentNode = l_pPolyMatricks->GetTransformNodes()[loop];
						CString l_NodeName;
						l_NodeName.PutAsciiString(l_pCurrentNode->GetName());

						if(( l_NodeName == L"" ) || (l_NodeName.IsEmpty()))
						{
							l_NodeName = L"noname_";
							l_NodeName += CString((LONG)l_NonameID);
							l_NonameID--;
						}

						if( l_pCurrentNode->Type() == CSLTemplate::XSI_TRANSLATE)
						{
							CSIBCVector3D l_Translation = ((CSLXSITranslate*) l_pCurrentNode)->GetTranslation();
							l_OpParams.PutValue(l_NodeName + L"_X", l_Translation.GetX());
							l_OpParams.PutValue(l_NodeName + L"_Y", l_Translation.GetY());
							l_OpParams.PutValue(l_NodeName + L"_Z", l_Translation.GetZ());						
						}
						else if( l_pCurrentNode->Type() == CSLTemplate::XSI_ROTATE)
						{
							CSIBCVector3D l_Rotation = ((CSLXSIRotate*) l_pCurrentNode)->GetAxis();
							l_OpParams.PutValue(l_NodeName + L"_X", l_Rotation.GetX());
							l_OpParams.PutValue(l_NodeName + L"_Y", l_Rotation.GetY());
							l_OpParams.PutValue(l_NodeName + L"_Z", l_Rotation.GetZ());						
							l_OpParams.PutValue(l_NodeName + L"_Angle", ((CSLXSIRotate*) l_pCurrentNode)->GetAngle());						
						}
						else if( l_pCurrentNode->Type() == CSLTemplate::XSI_SCALE)
						{
							CSIBCVector3D l_Scale = ((CSLXSIScale*) l_pCurrentNode)->GetScale();
							l_OpParams.PutValue(l_NodeName + L"_X", l_Scale.GetX());
							l_OpParams.PutValue(l_NodeName + L"_Y", l_Scale.GetY());
							l_OpParams.PutValue(l_NodeName + L"_Z", l_Scale.GetZ());						
						}
						else if( l_pCurrentNode->Type() == CSLTemplate::XSI_SHEAR)
						{
							CSIBCVector3D l_Axis = ((CSLXSIShear*) l_pCurrentNode)->GetFirstAxis();
							l_OpParams.PutValue(l_NodeName + L"_X1", l_Axis.GetX());
							l_OpParams.PutValue(l_NodeName + L"_Y1", l_Axis.GetY());
							l_OpParams.PutValue(l_NodeName + L"_Z1", l_Axis.GetZ());						
							l_Axis = ((CSLXSIShear*) l_pCurrentNode)->GetSecondAxis();
							l_OpParams.PutValue(l_NodeName + L"_X2", l_Axis.GetX());
							l_OpParams.PutValue(l_NodeName + L"_Y2", l_Axis.GetY());
							l_OpParams.PutValue(l_NodeName + L"_Z2", l_Axis.GetZ());						
							l_OpParams.PutValue(l_NodeName + L"_Angle", ((CSLXSIShear*) l_pCurrentNode)->GetAngle());						
						}
						else if( l_pCurrentNode->Type() == CSLTemplate::XSI_LOOKAT)
						{
							CSIBCVector3D l_Vector = ((CSLXSILookat*) l_pCurrentNode)->GetPosition();
							l_OpParams.PutValue(l_NodeName + L"_PX", l_Vector.GetX());
							l_OpParams.PutValue(l_NodeName + L"_PY", l_Vector.GetY());
							l_OpParams.PutValue(l_NodeName + L"_PZ", l_Vector.GetZ());						
							l_Vector = ((CSLXSILookat*) l_pCurrentNode)->GetInterest();
							l_OpParams.PutValue(l_NodeName + L"_IX", l_Vector.GetX());
							l_OpParams.PutValue(l_NodeName + L"_IY", l_Vector.GetY());
							l_OpParams.PutValue(l_NodeName + L"_IZ", l_Vector.GetZ());	
							l_Vector = ((CSLXSILookat*) l_pCurrentNode)->GetUpvector();
							l_OpParams.PutValue(l_NodeName + L"_UX", l_Vector.GetX());
							l_OpParams.PutValue(l_NodeName + L"_UY", l_Vector.GetY());
							l_OpParams.PutValue(l_NodeName + L"_UZ", l_Vector.GetZ());	
						}
						else if( l_pCurrentNode->Type() == CSLTemplate::XSI_MATRIX)
						{
							CSIBCMatrix4x4 l_Matrix = ((CSLXSIMatrix*) l_pCurrentNode)->GetMatrix();
							l_OpParams.PutValue(l_NodeName + L"_00", l_Matrix.Get(0,0));
							l_OpParams.PutValue(l_NodeName + L"_01", l_Matrix.Get(1,0));
							l_OpParams.PutValue(l_NodeName + L"_02", l_Matrix.Get(2,0));						
							l_OpParams.PutValue(l_NodeName + L"_03", l_Matrix.Get(3,0));						
							l_OpParams.PutValue(l_NodeName + L"_10", l_Matrix.Get(0,1));
							l_OpParams.PutValue(l_NodeName + L"_11", l_Matrix.Get(1,1));
							l_OpParams.PutValue(l_NodeName + L"_12", l_Matrix.Get(2,1));						
							l_OpParams.PutValue(l_NodeName + L"_13", l_Matrix.Get(3,1));	
							l_OpParams.PutValue(l_NodeName + L"_20", l_Matrix.Get(0,2));
							l_OpParams.PutValue(l_NodeName + L"_21", l_Matrix.Get(1,2));
							l_OpParams.PutValue(l_NodeName + L"_22", l_Matrix.Get(2,2));						
							l_OpParams.PutValue(l_NodeName + L"_23", l_Matrix.Get(3,2));	
							l_OpParams.PutValue(l_NodeName + L"_30", l_Matrix.Get(0,3));
							l_OpParams.PutValue(l_NodeName + L"_31", l_Matrix.Get(1,3));
							l_OpParams.PutValue(l_NodeName + L"_32", l_Matrix.Get(2,3));						
							l_OpParams.PutValue(l_NodeName + L"_33", l_Matrix.Get(3,3));	
						}
					}
				}
			}
		}
	}

	return status;
}

wchar_t *CModelInfoToXSI::GetClassID(){return L"CModelInfoToXSI";}

/**************************************************************************************
CModelPropertiesToXSI
**************************************************************************************/
CModelPropertiesToXSI::CModelPropertiesToXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CModelPropertiesToXSI::~CModelPropertiesToXSI() {};

CStatus CModelPropertiesToXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;

	if(*io_pFTKModel != NULL)
	{
		CSLModel *l_pFTKModel = (CSLModel *) *io_pFTKModel;
		CSIBCUserData *l_pModelUserData = l_pFTKModel->FindUserData("CREF");
		CRef* l_pCRef = (CRef*)(l_pModelUserData->GetData());
		X3DObject l_XSIModel = (X3DObject) *l_pCRef;


		// assign global materials
		if(l_pFTKModel->GlobalMaterial() != NULL)
		{
			CSLBaseMaterial *l_pFTKMaterial = l_pFTKModel->GlobalMaterial()->GetMaterial();

			CSIBCUserData *l_pMaterialUserData = l_pFTKMaterial->FindUserData("CREF");

			if(l_pMaterialUserData && l_pModelUserData)
			{
				l_pCRef = (CRef*)(l_pMaterialUserData->GetData());
				Material l_XSIMaterial = (Material) *l_pCRef;

				// we do not care about node or branch material for now, we'll always set material as node.
				l_XSIModel.SetMaterial(l_XSIMaterial);
			}
		}

		// set the static kinestate
		// Fill in the base pose for IK roots
		if(	l_XSIModel.HasStaticKinematicState() && (l_pFTKModel->GetXSIBasePose() != NULL) && 
				((l_pFTKModel->GetPrimitiveType() == CSLTemplate::SI_IK_ROOT) || 
				(l_pFTKModel->GetPrimitiveType() == CSLTemplate::SI_IK_EFFECTOR)))
		{
			// this code gets the static kinematics transform AKA the base pose
			StaticKinematicState l_XSIKineState = l_XSIModel.GetStaticKinematicState();
			assert(l_XSIKineState.IsValid());
		
			CSIBCVector3D l_FTKVector;
			Parameter l_XSIParameter;
			l_pFTKModel->CreateXSIBasePose();

			// this code gets the local kinematics transform
			MATH::CTransformation		l_Transformation;
			MATH::CVector3				l_XSIVector;

			// scaling
			l_FTKVector = l_pFTKModel->GetXSIBasePose()->GetScale();
			l_Transformation.SetScalingFromValues(l_FTKVector.m_fX, l_FTKVector.m_fY, l_FTKVector.m_fZ);

			// rotation
			// l_Transformation.PutRotationOrder(MATH::CRotation::siXYZ);
			l_FTKVector = l_pFTKModel->GetXSIBasePose()->GetEulerRotation();
			l_XSIVector.Set(in_pContext->AngleFromFTK(l_FTKVector.m_fX, true), in_pContext->AngleFromFTK(l_FTKVector.m_fY, true), in_pContext->AngleFromFTK(l_FTKVector.m_fZ, true));

			// position
			l_FTKVector = l_pFTKModel->GetXSIBasePose()->GetTranslation();
			l_Transformation.SetTranslationFromValues(l_FTKVector.m_fX, l_FTKVector.m_fY, l_FTKVector.m_fZ);

			l_XSIKineState.PutTransform(l_Transformation);
		
		}

	}

	return status;
}

wchar_t *CModelPropertiesToXSI::GetClassID(){return L"CModelPropertiesToXSI";}

/**************************************************************************************
CCRefFromUserData
**************************************************************************************/
CCRefFromUserData::CCRefFromUserData(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CCRefFromUserData::~CCRefFromUserData() {};

CStatus CCRefFromUserData::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;

	if(*io_pFTKModel != NULL)
	{
		CSLModel *l_pFTKModel = (CSLModel *) *io_pFTKModel;
		CSIBCUserData *l_pModelUserData = l_pFTKModel->FindUserData("CREF");

		if(l_pModelUserData)
		{
			*io_pXSIModel = *((CRef*)(l_pModelUserData->GetData()));
		}		
	}

	return status;
}

wchar_t *CCRefFromUserData::GetClassID(){return L"CCRefFromUserData";}


/**************************************************************************************
CRegisterTransformsToPlot
**************************************************************************************/
CRegisterTransformsToPlot::CRegisterTransformsToPlot(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CRegisterTransformsToPlot::~CRegisterTransformsToPlot() {};

CStatus CRegisterTransformsToPlot::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;
	X3DObject l_XSIModel = (X3DObject) *io_pXSIModel;

	if((*io_pFTKModel != NULL) && (!l_XSIModel.IsEqualTo(in_pContext->scene().GetRoot())))
	{
		CSLModel *l_pFTKModel = (CSLModel *) *io_pFTKModel;

		// transforms for everyone except lights (for now)
		if(	!l_pFTKModel->Primitive() || (l_pFTKModel->Primitive() && 
			(l_pFTKModel->Primitive()->Type() != CSLTemplate::SI_POINT_LIGHT) && 
			(l_pFTKModel->Primitive()->Type() != CSLTemplate::SI_DIRECTIONAL_LIGHT) && 
			(l_pFTKModel->Primitive()->Type() != CSLTemplate::SI_INFINITE_LIGHT) && 
			(l_pFTKModel->Primitive()->Type() != CSLTemplate::SI_SPOT_LIGHT)))
		{

			// Fill in the transform now
			if(l_pFTKModel->XSITransform() != NULL)
			{
				// this code sets the local kinematics transform
				KinematicState l_XSIKineState = l_XSIModel.GetKinematics().GetLocal();

				//sync polymatricks if it exists
				if((l_pFTKModel->XSITransform()->GetPolymatricks() != NULL) && (l_pFTKModel->GetEnvelopeCount() == 0))
				{
					int loop;
					for(loop = 0; loop < l_pFTKModel->XSITransform()->GetPolymatricks()->GetTransformNodeCount(); loop++)
					{
						CSLTemplate *l_pTransform  = l_pFTKModel->XSITransform()->GetPolymatricks()->GetTransformNodes()[loop];
						if(l_pTransform->GetFCurveCount())
						{
							GetParamList().Add(l_XSIKineState.GetParameter(L"posx"));
							GetParamList().Add(l_XSIKineState.GetParameter(L"posy"));
							GetParamList().Add(l_XSIKineState.GetParameter(L"posz"));
							GetParamList().Add(l_XSIKineState.GetParameter(L"sclx"));
							GetParamList().Add(l_XSIKineState.GetParameter(L"scly"));
							GetParamList().Add(l_XSIKineState.GetParameter(L"sclz"));
							GetParamList().Add(l_XSIKineState.GetParameter(L"rotx"));
							GetParamList().Add(l_XSIKineState.GetParameter(L"roty"));
							GetParamList().Add(l_XSIKineState.GetParameter(L"rotz"));
							GetParamList().Add(l_XSIKineState.GetParameter(L"sclorix"));
							GetParamList().Add(l_XSIKineState.GetParameter(L"scloriy"));
							GetParamList().Add(l_XSIKineState.GetParameter(L"scloriz"));
							break;
						}
					}
				}
			}
		}
	}

	return status;
}

CParameterRefArray		CRegisterTransformsToPlot::m_Parameters;

wchar_t *CRegisterTransformsToPlot::GetClassID(){return L"CRegisterTransformsToPlot";}

void					CRegisterTransformsToPlot::ClearParamList()
{
	m_Parameters = CParameterRefArray();
}

CParameterRefArray&		CRegisterTransformsToPlot::GetParamList()
{
	return m_Parameters;
}

/**************************************************************************************
CXSIPlotPolymatricksAnimation
**************************************************************************************/
CXSIPlotPolymatricksAnimation::CXSIPlotPolymatricksAnimation() 
{
	
};

CXSIPlotPolymatricksAnimation::~CXSIPlotPolymatricksAnimation() 
{

};

CStatus CXSIPlotPolymatricksAnimation::Execute(CdotXSIConverter *in_pContext)
{
	int loop, count = CRegisterTransformsToPlot::GetParamList().GetCount();

	if(count)
	{
		Application App;

		CRefArray l_FCurves = CRegisterTransformsToPlot::GetParamList().PlotAnimation(DBL_MAX, DBL_MAX, 1, siStandardFCurve, siLinearInterpolation, true, 0.0, true, true);

		for(loop = 0; loop < count; loop++)
		{
			FCurve l_SrcFCurve = (FCurve) l_FCurves[loop];
			FCurve l_DstFCurve;
			Parameter l_Param = (Parameter) CRegisterTransformsToPlot::GetParamList()[loop];
			if(l_SrcFCurve.IsValid())
			{
				bool l_bSync = true;
				// check if the fcurve is constant or not
				if(l_SrcFCurve.GetKeys().GetCount() == 2)
				{
					FCurveKey l_FirstKey, l_LastKey;
					l_FirstKey = l_SrcFCurve.GetKeys()[0];
					l_LastKey = l_SrcFCurve.GetKeys()[1];

					if(l_FirstKey.GetValue() == l_LastKey.GetValue())
						l_bSync = false;
				}

				if(l_bSync)
				{
					l_Param.AddFCurve(siStandardFCurve, l_DstFCurve);
					l_DstFCurve.Set(l_SrcFCurve);
				}
			}
		}

		CRegisterTransformsToPlot::ClearParamList();
	}

	return CStatus::OK;
}

wchar_t *CXSIPlotPolymatricksAnimation::GetClassID(){return L"CXSIPlotPolymatricksAnimation";}


/**************************************************************************************
COptimizePolymatricks
**************************************************************************************/
COptimizePolymatricks::COptimizePolymatricks(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

COptimizePolymatricks::~COptimizePolymatricks() {};

CStatus COptimizePolymatricks::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;
	X3DObject l_XSIModel = (X3DObject) *io_pXSIModel;

	if((*io_pFTKModel != NULL) && (!l_XSIModel.IsEqualTo(in_pContext->scene().GetRoot())))
	{
		CSLModel *l_pFTKModel = (CSLModel *) *io_pFTKModel;

		// transforms for everyone except lights (for now)
		if(	!l_pFTKModel->Primitive() || (l_pFTKModel->Primitive() && 
			(l_pFTKModel->Primitive()->Type() != CSLTemplate::SI_POINT_LIGHT) && 
			(l_pFTKModel->Primitive()->Type() != CSLTemplate::SI_DIRECTIONAL_LIGHT) && 
			(l_pFTKModel->Primitive()->Type() != CSLTemplate::SI_INFINITE_LIGHT) && 
			(l_pFTKModel->Primitive()->Type() != CSLTemplate::SI_SPOT_LIGHT)))
		{

			// Fill in the transform now
			if(l_pFTKModel->XSITransform() != NULL)
			{
				// back up the transform before deleting the polymatricks
				// since the transform will be reseted to identity by the delete operation.
				KinematicState l_XSIKineState = l_XSIModel.GetKinematics().GetLocal();

				MATH::CTransformation l_Transformation;
				l_Transformation = l_XSIKineState.GetTransform();

				//sync polymatricks if it exists
				if((l_pFTKModel->XSITransform()->GetPolymatricks() != NULL) && (l_pFTKModel->GetEnvelopeCount() == 0))
				{
					CString l_OpName = l_XSIKineState.GetFullName();
					l_OpName += L".polymatricks";
					COMMANDS::DeleteObj(l_OpName);

					// set back the transform once the polymatricks is deleted
					l_XSIKineState.PutTransform(l_Transformation);
				}
			}
		}
	}

	return status;
}

wchar_t *COptimizePolymatricks::GetClassID(){return L"COptimizePolymatricks";}
