//***************************************************************************************
//
// File supervisor: Softimage Rendering & Pipeline team
//
// (c) Copyright 2001-2005 Avid Technology, Inc. . All rights reserved.
//
//***************************************************************************************

/****************************************************************************************
THIS CODE IS PUBLISHED AS A SAMPLE ONLY AND IS PROVIDED "AS IS".
IN NO EVENT SHALL SOFTIMAGE, AVID TECHNOLOGY, INC. AND/OR THEIR RESPECTIVE
SUPPLIERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
CONNECTION WITH THE USE OR PERFORMANCE OF THIS CODE .

COPYRIGHT NOTICE. Copyright  1999-2005 Avid Technology Inc. . All rights reserved. 

SOFTIMAGE is a registered trademark of Avid Technology Inc. or its subsidiaries 
or divisions. Windows NT is a registered trademark of Microsoft Corp. All other
trademarks contained herein are the property of their respective owners. 
****************************************************************************************/

/*! \file dotXSIConverter.h   
    Definition file for the CdotXSIConverter class
*/

#ifndef _DOTXSICONVERTER_H
#define _DOTXSICONVERTER_H

#include "stdafx.h"
#include <xsi_status.h>
#include <xsi_application.h>
#include <xsi_project.h>
#include <xsi_scene.h>
#include <xsi_property.h>
#include <SIBCArray.h>
#include <Scene.h>
#include <COLLADAScene.h>
#include "Crosswalk/dotXSIConverterCallback.h"
#include "FTKUtil.h"

#define CNV_DEGTORAD(x) x *= 0.0174532925199f
#define CNV_RADTODEG(x) x *= 57.295779513082f

#define EXPORT_ALL						0
#define EXPORT_SUBSET					1
#define EXPORT_ALLMATERIALS		2
#define EXPORT_MATERIALS			3

#define CRY_FILE_TYPE_NONE		0x0000
#define CRY_FILE_TYPE_CGF		0x0001
#define CRY_FILE_TYPE_CGA		0x0002
#define CRY_FILE_TYPE_CHR		0x0004
#define CRY_FILE_TYPE_CAF		0x0008
#define CRY_FILE_TYPE_ANM		0x0010

#define EXPORT_TYPE_ALL					0
#define EXPORT_TYPE_ANIMATION_CLIPS		1
#define EXPORT_TYPE_SKELETON_AND_MESH	2

#define DOTXSI_FORMAT_5_0			0
#define COLLADA_FORMAT_1_4_1		1
#define DOTXSI_FORMAT_5_0_BINARY	2
#define DOTXSI_FORMAT_6_0			3
#define DOTXSI_FORMAT_6_0_BINARY	4

#define INFO_XSI_TO_MAX				0
#define INFO_XSI_TO_MAYA			1
#define INFO_XSI_TO_XSI				2
#define INFO_CUSTOM					3
#define INFO_MAX_TO_XSI				0
#define INFO_MAYA_TO_XSI			1

#define CROSSWALK_VERSION			L"Crosswalk"
#define CROSSWALK_VERSION_NUMBER	L"2.5"

/****************************************************************************************
To make sure we use the XSI namespace
****************************************************************************************/
using namespace XSI;

class CHierarchyElementInfo;

//! Class CdotXSIConverter
/*!
    This is the class that implements the high level dotXSI converter functionality. 
	This class contains all the global contexts that are needed by the conversion process.
*/

class CdotXSIConverter
{

public:
/****************************************************************************************
Constructors/Destructors
****************************************************************************************/

//! Default constructor
	CdotXSIConverter();

//! Default destructor
	~CdotXSIConverter();

/****************************************************************************************
Member access
****************************************************************************************/

//! Sets the format 0 = dotXSI, 1 = COLLADA
	void	SetFormat( int in_iFormat );

//! Returns the format 0 = dotXSI, 1 = COLLADA
/*!
	\return int the format 0 = dotXSI, 1 = COLLADA 
*/
	int		GetFormat();

//! Returns the XSI application instance
/*!
	\return Application the XSI application instance 
*/
	Application	&application();

//! Returns the XSI project instance
/*!
	\return Project the XSI project instance 
*/
	Project	&project();

//! Returns the XSI scene instance
/*!
	\return Application the XSI scene instance 
*/
	Scene	&scene();

//! Returns the list of arguments
/*!
	\return CValueArray the list of arguments
*/
	CValueArray	 &args();

//! Returns the export property
/*!
	\return Property the export property
*/
	Property	&exportproperty();

//! Returns the import property
/*!
	\return Property the import property
*/
	Property	&importproperty();

//! Returns the FTK scene
/*!
	\return CSLScene* the dotXSI scene
*/
	CSLScene	*ftkscene();


//! Returns the list of export callbacks
/*!
	\return CSIBCArray<CdotXSIConverterCallback*> the list of export callbacks
*/
	CSIBCArray<CdotXSIConverterCallback *>	&exportcallbacks();

//! Returns the list of import callbacks
/*!
	\return CSIBCArray<CdotXSIConverterCallback*> the list of import callbacks
*/
	CSIBCArray<CdotXSIConverterCallback *>	&importcallbacks();

//! Returns the list of write callbacks
/*!
	\return CSIBCArray<CdotXSIConverterCallback*> the list of write callbacks
*/
	CSIBCArray<CdotXSIConverterCallback *>	&writecallbacks();

//! Returns the list of read callbacks
/*!
	\return CSIBCArray<CdotXSIConverterCallback*> the list of read callbacks
*/
	CSIBCArray<CdotXSIConverterCallback *>	&readcallbacks();


//! Clears the list of export callbacks
/*!
	\return void
*/
	void ClearExportCallbacks();

//! Clears the list of import callbacks
/*!
	\return void
*/
	void ClearImportCallbacks();

//! Clears the list of write callbacks
/*!
	\return void
*/
	void ClearWriteCallbacks();

//! Clears the list of Read callbacks
/*!
	\return void
*/
	void ClearReadCallbacks();


//! Sets the export property by name
/*!
	\return CStatus to know whether the function failed 
*/
	CStatus		SetExportProperty(CString in_PropertyName);

//! Sets the import property by name
/*!
	\return CStatus to know whether the function failed 
*/
	CStatus		SetImportProperty(CString in_PropertyName);


//! Adds a CRef in the internal CRef garbage collection
/*!
	\param in_ref the CRef to add to the garbage collection, the CRef will remain here until the CdotXSIConverter is destroyed
	\return CRef* the CRef that was added to the garbage collection 
*/
	CRef		*AddCRef(CRef &in_ref);

//! Returns the first hierarchy element info node in the list
/*!
	\return CHierarchyElementInfoList* pointer on the first hierarchy element info node
*/
	CHierarchyElementInfo*		GetFirstHierarchyElementInfo();

//! Returns the next hierarchy element info node in the list
/*!
	\return CHierarchyElementInfoList* pointer on the next hierarchy element info node
*/
	CHierarchyElementInfo*		GetNextHierarchyElementInfo();

//! Adds a hierarchy element info to the end of the elem list
/*!
	\return CHierarchyElementInfoList* pointer on the newly created hierarchy element info node
*/
	CHierarchyElementInfo*		AddHierarchyElementInfo();

/****************************************************************************************
Conversion methods
****************************************************************************************/

//! Converts the XSI content to dotXSI format
/*!
	\return CStatus to know whether the function failed 
*/

	CStatus	ConvertFromXSI();

//! Converts the dotXSI content back in XSI
/*!
	\return CStatus to know whether the function failed 
*/
	CStatus ConvertToXSI();

/****************************************************************************************
Utility methods
****************************************************************************************/

//! Gets an angle from the FTK based on the scene angle type
/*!
	\param	in_fAngle the angle to get the value from
	\param	in_bRadian if we want the angle in radians or degrees
	\return float the angle value in degrees of radians
*/

	float	AngleFromFTK(float in_fAngle,bool in_bRadian);

//! Sets an angle to the FTK based on the scene angle type
/*!
	\param	in_fAngle the angle to get the value from
	\param	in_bRadian if we want the angle in radians or degrees
	\return float the angle value in degrees of radians
*/

	float	AngleToFTK(float in_fAngle,bool in_bRadian);

//! Returns the Semantic layer model associated with the CRef
/*!
	\param	in_pRootModel The Semantic Layer model to start the search from
	\param	in_cref The CRef of the object to search for
	\return CSLModel* The Semantic Layer model or NULL on error
*/
	CSLModel* FindModelByCRef ( CSLModel* in_pRootModel, CRef in_cref );

	//! Sets a subtitle for the progress bar. This subtitle will appear only for the duration of the current callback
/*!
	\param	in_szSubTitle CString holding the subtitle
	\return nothing
*/
	void SetProgressBarSubtitle( CString in_Subtitle );

	void SetCancelPressed(bool in_bCancelPressed);

	bool IsCancelPressed();


/****************************************************************************************
IO methods
****************************************************************************************/

//! Writes the dotXSI content to a file
/*!
	\return CStatus to know whether the function failed 
*/

	CStatus	Write();

//! Reads the dotXSI content from a file
/*!
	\return CStatus to know whether the function failed 
*/
	CStatus Read();

private:

	Application									m_Application;
	Project										m_Project;
	Scene										m_Scene;
	Property									m_ExportProperty;
	Property									m_ImportProperty;
	CValueArray									m_Args;
	CStatus										m_Status;
	CSLScene									*m_pScene;

	CHierarchyElementInfo						*m_pHierarchyElementInfoRoot;
	CHierarchyElementInfo						*m_pHierarchyElementInfoLast;
	CHierarchyElementInfo						*m_pHierarchyElementInfoCurrent;
	

	CSIBCArray<CdotXSIConverterCallback *>		m_ExportCallbacks;
	CSIBCArray<CdotXSIConverterCallback *>		m_WriteCallbacks;
	CSIBCArray<CdotXSIConverterCallback *>		m_ImportCallbacks;
	CSIBCArray<CdotXSIConverterCallback *>		m_ReadCallbacks;
	CSIBCArray<CRef*>							m_CRefGarbageCollection;

	bool										m_bIsCancelPressed;

	int											m_iFormat;
};

/****************************************************************************************
dotXSI converter utility function
****************************************************************************************/

//! Utility function that prints a variable number of arguments in the XSI log window when operating in _DEBUG mode
/*!
    \param in_pFormat the string format (limited to 4096 characters)
    \return void
*/
extern void debugprintf(wchar_t *in_pFormat, ...);

//! Utility function that sets a global state for the verbose mode.
/*!
\param in_bVerbose the verbose state.
\return void
*/
extern void setVerboseMode(bool in_bVerbose);

//! Utility function that prints a variable number of arguments in the XSI log window
/*!
    \param in_pFormat the string format (limited to 4096 characters)
    \return void
*/
extern void logmessage(wchar_t *in_pFormat, ...);

//! Utility function that prints an error message containing a variable number of arguments in the XSI log window
/*!
\param in_pFormat the string format (limited to 4096 characters)
\return void
*/
extern void logerrormessage(wchar_t *in_pFormat, ...);



//! Utility macro to wrap some very basic error handling
#ifdef _DEBUG
	#define _XSI_CALL(function,message) if(status != CStatus::Fail) status = function; if(status == CStatus::Fail) debugprintf(L"%s Function %s failed in file %s at line %d\n", message, __FUNCTION__, __FILE__, __LINE__)
#else
	#define _XSI_CALL(function,message) if(status != CStatus::Fail) status = function; if(status == CStatus::Fail) logmessage(message);
#endif	

//! Utility macro to output an explicit error message
#ifdef _DEBUG
	#define logerror(message) debugprintf(L"%s Function %s failed in file %s at line %d\n", message, __FUNCTION__, __FILE__, __LINE__)
#else
	#define logerror(message) logerrormessage(message);
#endif	




#endif _DOTXSICONVERTER_H
