﻿using System;
using System.Collections.Generic;
using System.Linq;
//using System.Windows.Forms;
//using Xbox360CallStackUtils;
//using DebugMonitor;
using System.IO;
using System.Text.RegularExpressions;
using System.Text;
//using System.Drawing;
using CrashHandler;
using DebugHelper;

namespace Xbox360CrashHandler
{
    public class XboxCallStackDecoder
    {
        private DiaAccess xboxFunctionDecoder;
        private readonly Log m_log;
        private String errorLogContent;
				private String errorLogContentHeader;
        private const String LOADED_MODULES_STRING = "Loaded Modules:";
        private const String CALL_STACK_TRACE_STRING = "Call Stack Trace:";
        public static readonly String XENON_EXCEPTION_HANDLER = "CryXenonUnhandledExceptionHandler()";
        private const String HEX_VALUE = "[A-Fa-f0-9]";
        private const String ANY_HEX = "(" + HEX_VALUE + "{2,9})";
        private const String EIGHT_HEX = "(" + HEX_VALUE + "{8})";
        private const String FOUR_HEX = "(" + HEX_VALUE + "{4})";
        private const String TWO_HEX = "(" + HEX_VALUE + "{2})";
        private Settings m_settings;

        private readonly String pathToError;
        private readonly String pathToPdb;

        private readonly ExceptionHandler exceptionHelper;
        //private readonly XboxFilesystem m_xboxFilesystem;
        private SymbolServer symbolServer;

        public XboxCallStackDecoder(String errorLogInFileLocation, Log log, String pathToError, String pathToPdb, Settings settings, ExceptionHandler exceptionHelper) //, XboxFilesystem xboxFilesystem)
        {
			try
			{
				using (StreamReader fileStream = new StreamReader(errorLogInFileLocation))
				{
					errorLogContent = fileStream.ReadToEnd();
					errorLogContentHeader = StringHelper.GetLinesBetweenPatterns(errorLogContent, ".*", "Call Stack.*", StringHelper.LineOptions.includeStart);
				}
			}
			catch (Exception e)
			{
				System.Console.Error.WriteLine("XboxCallStackDecoder() Exception:\n" + e.Message);
                Environment.Exit(1);
			}

            this.pathToError = pathToError;
			this.pathToPdb = pathToPdb;
            m_settings = settings;
            
            this.exceptionHelper = exceptionHelper;
            m_log = log;
            xboxFunctionDecoder = new DiaAccess(log);
            //m_xboxFilesystem = xboxFilesystem;
            try{
                symbolServer = new SymbolServer(m_settings.SymbolServerPath);
            } catch (Exception e){
                m_log.Info("An error occurred trying to connect with the symbol server. No requests to it will be made:" + e.Message);
            }
            LoadSignature();
        }


        public String GetCallStackTrace()
        {
            String result = DecodeCallStackTrace();

            if (Util.CountStringOccurrences(result.ToLower(), ") handleexception()") > 0)
            {
                result = StringHelper.GetLinesBetweenPatterns(result, @".*mainCRTStartup\(\).*", "\n\n", StringHelper.LineOptions.includeEnd); 
            }

            return errorLogContentHeader + Environment.NewLine + "Call Stack Trace:" + Environment.NewLine + result;
        }

        private static String GetCallStackTrace(String file)
        {
            const string callStackTraceSectionStart = CALL_STACK_TRACE_STRING;
            return StringHelper.GetLinesBetweenPatterns(file, callStackTraceSectionStart, StringHelper.EMPTY_LINE_REGEX, StringHelper.LineOptions.exclusive);
        }

        private void LoadSignature()
        {
            String loadedModules = GetLoadedModules(errorLogContent);
            m_log.Info("************************ Loaded modules ************************");
            m_log.Info(loadedModules);
            m_log.Info("***************************** End ******************************");
            
            String[] moduleLines = StringHelper.GetNonEmptyLines(loadedModules);

            foreach (String line in moduleLines)
            {
                LoadPdbLine(line);
            }
        }

        private static String GetLoadedModules(String file)
        {
            return StringHelper.GetLinesBetweenPatterns(file, LOADED_MODULES_STRING, StringHelper.EMPTY_LINE_REGEX, StringHelper.LineOptions.exclusive);
        }

        private void LoadPdbLine(String line)
        {
            //AName.dll d:\projectName\projectName.xbox360\build\binxb\configuration\AName.pdb 82740000, 00350000, 4AEEA65B,{D36D77BF-ED25-4FD3-B4 CC-26 72 14 C6 65 6C}, 1
            const String NAME_LOCATION = "(.*) (.*) ";
            const String BASEADDRESS_SIZE_TIMESTAMP = EIGHT_HEX + ", " + EIGHT_HEX + ", " + EIGHT_HEX + ",\\s*" ;
            const String GUID = "\\{" + EIGHT_HEX + "-" + FOUR_HEX + "-" + FOUR_HEX + "-" + TWO_HEX + " " + TWO_HEX + "-" + TWO_HEX + " " + TWO_HEX + " " + TWO_HEX + " " + TWO_HEX + " " + TWO_HEX + " " + TWO_HEX + "\\}, ";
            const String AGE = "([0-9]+)";
            Match match = Regex.Match(line, NAME_LOCATION + BASEADDRESS_SIZE_TIMESTAMP + GUID + AGE);
            if (!match.Success)
            {
                throw new Exception("The "+m_settings.ErrorLogName+" has a wrong file format" + Environment.NewLine + "Cannot match the following line:" + Environment.NewLine + line);
            }
            String exeName = GetMatchValue(match, EXE_INDEX);
            String pdbPath = GetMatchValue(match, PDB_PATH_INDEX);
            ulong baseAddress = GetUlongMatch(match, BASE_ADDRESS_INDEX);
            uint size = GetUInt32Match(match, SIZE_INDEX);
            uint timestamp = GetUInt32Match(match, TIMESTAMP_INDEX);
            uint age = GetAge(match, AGE_INDEX);

            Guid guid = GetGuid(match);
            
            String moduleName = Path.GetFileNameWithoutExtension(pdbPath);
			String moduleNameWithExt = Path.GetFileName(pdbPath);

            XboxSignature signature = new XboxSignature(guid, age, pdbPath);

            bool success = LoadPDB(baseAddress, size, signature, moduleNameWithExt, pdbPath);
            PrintResult(success, moduleName);
        }

        private Guid GetGuid(Match match)
        {
            StringBuilder result = new StringBuilder();
            for(int guidIndex = GUID_START; guidIndex <= GUID_END; ++guidIndex)
            {
                result.Append(GetMatchValue(match, guidIndex));
            }
            return new Guid(result.ToString());
        }

        private bool LoadPDB(ulong baseAddress, uint size, XboxSignature signature, String pdbModuleName, String pdbPath)
        {
			m_log.Info("LoadPDB() trying with pdbPath=" + pdbPath + "; but overridePath = " + pathToPdb);

			signature.PdbLocation = pathToPdb + pdbModuleName; //pdbPath;
            bool success = xboxFunctionDecoder.LoadPdb(baseAddress, size, signature);
            if (!success)
            {
                success = TryLoadFromSymbolServer(pdbPath, signature, baseAddress, size);
            }

			return success;

			/*
            bool success = xboxFunctionDecoder.LoadPdb(baseAddress, size, signature);

            if (!success)
            {
                success = TryLoadFromCache(pdbPath, signature, baseAddress, size);
            }
			*/

			/*
            if (!success)
            {
                success = TryLoadFromSymbolServer(pdbPath, signature, baseAddress, size);
            }
            if(!success)
            {
                success = TryLoadFromXbox(pdbPath, signature, baseAddress, size);
            }
            if(!success){
                success = TryLoadFromCommonLocations(pdbPath, signature, baseAddress, size);
            }
			
            return success;
			*/
        }

        private bool TryLoadFromSymbolServer(String pdbPath, XboxSignature signature, ulong baseAddress,uint size){
            String foundPath = string.Empty;

            bool success = false;
            if(symbolServer!= null)
            {
                success = symbolServer.FindFileInPath(pdbPath, signature.Guid, (int)signature.Age, ref foundPath).Equals(true);
                if (success)
                {
                    signature.PdbLocation = foundPath;
                    success = xboxFunctionDecoder.LoadPdb(baseAddress, size, signature);
                }
                if(success)
                {
                    m_log.Info("Loaded " + signature.PdbFilename + " from the symbol server");
                }
            }
            return success;
        }


		/*
        private bool TryLoadFromCache(String pdbPath, XboxSignature signature, ulong baseAddress, uint size)
        {
            String relativePath = Path.Combine(XboxCrashHandler.PDB_FOLDER, Path.GetFileName(pdbPath));
            signature.PdbLocation = relativePath;
            return xboxFunctionDecoder.LoadPdb(baseAddress, size, signature);
            
        }
		*/

		/*
        private bool TryLoadFromXbox(String pdbPath, XboxSignature signature, ulong baseAddress, uint size){
            String pdbName = Path.GetFileName(pdbPath);
            m_log.Info("Trying to load " + pdbName + " from the crash folder");
            bool success = m_xboxFilesystem.ReceiveFile(Path.Combine(pathToError, pdbName), Path.Combine(XboxCrashHandler.PDB_FOLDER, pdbName));
            if (success)
            {
                return TryLoadFromCache(pdbPath, signature, baseAddress, size);
            }
            return false;
        }
		 */

		/*
        private bool TryLoadFromCommonLocations(String pdbPath, XboxSignature signature, ulong baseAddress, uint size)
        {
            String buildNumber = String.Empty;
            try
            {
                buildNumber = DataExtractor.GetBuildNumber(errorLogContent);
            } catch(Exception e){
                exceptionHelper.HandleFailedSubmit("Cannot get the build number from the " + m_settings.ErrorLogName+ " ", e);
                Environment.Exit(0);
            }
            if(String.IsNullOrEmpty(m_settings.PdbSymbolsLookupFolders)){
                return false;
            }
            String folderWithReplacements = m_settings.PdbSymbolsLookupFolders.Replace(":buildNumber", buildNumber);

            String filename = Path.GetFileName(pdbPath);
            String[] alternatives = folderWithReplacements.Split(new char[] { ';' });

            for (int index = 0; index < alternatives.Length; ++index)
            {
                List<String> foldersFound = Util.GetMatchingFolders(alternatives[index]);
                foreach (String folder in foldersFound)
                {
                    String pdbFileLocation = Path.Combine(folder, filename);
                    signature.PdbLocation = pdbFileLocation;
                    bool success = xboxFunctionDecoder.LoadPdb(baseAddress, size, signature);
                    if (success)
                    {
                        return true;
                    } 
                }
            }
            m_log.Warn("Couldn't find " + filename + " in the lookup folders ");
            return false;
        }
		 */

        private void PrintResult(bool success, String moduleName)
        {
            if (success)
            {
                m_log.Info("Loaded symbols for: " + moduleName);
            }
            else
            {
                m_log.Info("Failed to load symbols for: " + moduleName);
            }
        }

        private static ulong GetUlongMatch(Match match, int mc)
        {
            return Convert.ToUInt64(GetMatchValue(match, mc), 16);
        }

        private static uint GetAge(Match match, int mc)
        {
            return Convert.ToUInt32(GetMatchValue(match, mc), 10);
        }

        private static uint GetUInt32Match(Match match, int mc)
        {
            return Convert.ToUInt32(GetMatchValue(match, mc), 16);
        }

        private static string GetMatchValue(Match match, int mc)
        {
            return match.Groups[mc].Value;
        }

				private void DecodeExceptionAddressInHeader()
				{
            StringBuilder headerBuilder = new StringBuilder();
            String[] headerLines = StringHelper.GetAllLines(GetCallStackTrace(errorLogContentHeader));
 
						foreach (String line in headerLines)
            {
							if (Regex.IsMatch(line, "Exception Addr:"))
							{
								m_log.Info("found exception addr: line in header");
								String strippedLine = Regex.Replace(line, "Exception Addr: ", "");
								m_log.Info("used replace to remove the Exception Addr: part. Left with " + strippedLine + "; which is " + strippedLine.Length + "long");

								// TODO - if required we can save this exception addr string ready for substituting into any 0xFFFFFFFF addresses found in the stack
                uint code = Convert.ToUInt32(strippedLine, 16);
                XboxCrashLineInfo crashLine = xboxFunctionDecoder.DecodeSymbol(code);
								if (crashLine != null)
								{
									string newline = "Exception Addr: " + strippedLine + " - " + crashLine.FunctionName + "() [" + Path.GetFileName(crashLine.Filename) + ":" + crashLine.LineNumber + "]" + Environment.NewLine;
									headerBuilder.Append(newline);
								}
								else
								{
									headerBuilder.Append(line+Environment.NewLine);
								}
							}
							else
							{
								headerBuilder.Append(line+Environment.NewLine);
							}
            }
						
					errorLogContentHeader = headerBuilder.ToString();
				}

        private String DecodeCallStackTrace()
        {
            String[] functionLines = StringHelper.GetNonEmptyLines(GetCallStackTrace(errorLogContent));
            String result = String.Empty;

            foreach (String line in functionLines)
            {
                m_log.Info("Decoding function...");
                result = DecodeFunctionLine(result, line);
            }

						DecodeExceptionAddressInHeader();

            return result;
        }

        private String DecodeFunctionLine(String result, String line)
        {
            const String numberAndParenthesis = @"(\s*[0-9]*\))\s*";
            Match functionCode = Regex.Match(line, numberAndParenthesis + ANY_HEX);
            if (functionCode.Success)
            {
                uint code = Convert.ToUInt32(functionCode.Groups[2].Value, 16);
                XboxCrashLineInfo crashLine = xboxFunctionDecoder.DecodeSymbol(code);

                String callstackNumber = functionCode.Groups[1] + " ";

                if (crashLine != null )
                {
                    result += callstackNumber + crashLine.FunctionName + "() [" + Path.GetFileName(crashLine.Filename) + ":" + crashLine.LineNumber+ "]" + Environment.NewLine;
                    m_log.Info("O " + code + "=>" + crashLine.FunctionName);
                }
                else
                {
                    result += callstackNumber + "function=0x" + code.ToString("X") + Environment.NewLine;
                    m_log.Info("X " + code);
                }
            }
            return result;
        }

        private const int EXE_INDEX = 1;
        private const int PDB_PATH_INDEX = 2;
        private const int BASE_ADDRESS_INDEX = 3;
        private const int SIZE_INDEX = 4;
        private const int TIMESTAMP_INDEX = 5;
        private const int GUID_START = 6;
        private const int GUID_END = 16;
        private const int AGE_INDEX = 17;
    }
}
