import struct

def null_termin(value):
	return repr(value[:value.find('\000')])

char_map = range(ord('0'), ord('9') + 1) + range(ord('A'), ord('F') + 1)
char_map = [chr(x) for x in char_map]
char_group_size = 4
def hex_dump(value):
	string = ''
	for index, char in enumerate(value):
		nibbles = (ord(char) >> 16), (ord(char) & 0xF)
		string += char_map[nibbles[0]] + char_map[nibbles[1]]
		if (index & (char_group_size - 1)) == char_group_size - 1:
			string += ' '
	return string

def strz(value):
	return value[:value.find('\000')]

class FieldDef(object):

	def __init__(self, name, format, print_fmt=str):
		self.name = name
		self.format = format
		self.print_fmt = print_fmt

	def value_str(self, value):
		return self.print_fmt(value)

	def flag_set(self, name, value):
		return False

class FlagsFormatter(object):

	def __init__(self, flags):
		self.flags = flags

	def __call__(self, value):
		bits = [1 << x for x in xrange(len(self.flags))]
		values = [((value & bit) != 0) for bit in bits]
		digits = [{False: '0', True: '1'}[x] for x in values]
		bit_string = ''.join(reversed(digits))
		flag_strings = [flag + '=' + digit + ', ' for flag, digit in zip(self.flags, digits)]
		flag_string = ''.join(flag_strings)[:-2]
		known_mask = (1 << len(self.flags)) - 1
		error_string = ''
		if value & ~known_mask:
			error_string = ' <UNKNOWN FLAGS SET>'
		return '%s (%s)%s' % (bit_string, flag_string, error_string)

class EnumFormatter(object):

	def __init__(self, values):
		self.values = values

	def __call__(self, value):
		try:
			description = self.values[value]
		except:
			description = '<INVALID ENUM VALUE>'
		return '%d (%s)' % (value, description)

class FlagsDef(FieldDef):

	def __init__(self, name, format, flags):
		self.flags = flags
		FieldDef.__init__(self, name, format, FlagsFormatter(self.flags))
		self.flag_map = dict([(flag, index) for index, flag in enumerate(self.flags)])

	def flag_set(self, name, value):
		return (value & (1 << self.flag_map[name])) != 0

class EnumDef(FieldDef):

	def __init__(self, name, format, values):
		self.values = values
		FieldDef.__init__(self, name, format, EnumFormatter(self.values))

	def get_enum_string(self, value):
		try:
			description = self.values[value]
		except:
			description = '<INVALID ENUM VALUE>'
		return description

class Field(object):

	def __init__(self, field_def, value, pos):
		self.field_def = field_def
		self.value = value
		self.pos = pos

	def format(self, pos):
		print '%.8X:    %s = %s' % (pos + self.pos, self.field_def.name, self.field_def.value_str(self.value))

	def flag_set(self, name):
		return self.field_def.flag_set(name, self.value)

	def get_enum_string(self):
		return self.field_def.get_enum_string(self.value)

class DataObject(object):

	def __init__(self, pos, name, fields):
		self.pos = pos
		self.name = name
		self.fields = fields
		self.name_field_map = dict([(field.field_def.name, field) for field in self.fields])

	def find_field(self, name):
		return self.name_field_map[name]

	def format(self, pos):

		print '%.8X:  %s' % (self.pos + pos, self.name)
		for field in self.fields:
			field.format(pos)

def cumulative_sum(seq):
	csum = []
	total = 0
	for val in seq:
		total += val
		csum.append(total)
	return csum

def unpack_fields(name, field_defs, data, pos):

	def get_cumulative_format(index, formats):

		format = ''.join(formats[:index])
		alignment_format = formats[index]
		alignment_format = '0' + alignment_format[-1]
		format += alignment_format
		return format

	formats = [field.format for field in field_defs]
	format = ''.join(formats)
	sizes = [struct.calcsize(x) for x in formats]
	cumulative_formats = [get_cumulative_format(x, formats) for x in xrange(len(formats))]
	offsets = [struct.calcsize(f) for f in cumulative_formats]
	values = []
	for offset, size, f in zip(offsets, sizes, formats):
		value = struct.unpack(f, data[offset:offset+size])
		if len(value) == 1:
			value, = value
		values.append(value)
	attrs = zip(field_defs, values, offsets)
	fields = [Field(n, value, pos + p) for n, value, p in attrs]
	return DataObject(pos, name, fields)

def read_fields(name, field_defs, reader):

	formats = [field.format for field in field_defs]
	format = ''.join(formats)
	pos = reader.position
	data = reader.read(struct.calcsize(format))
	return unpack_fields(name, field_defs, data, pos)

def get_fields_size(field_defs):
	formats = [field.format for field in field_defs]
	format = ''.join(formats)
	format += '0i'
	return struct.calcsize(format)

