import struct
import chunks

class ChunkDocument(object):

	def __init__(self):

		self.chunks = []
		self.id_chunk_map = {}

	def add_chunk(self, chunk):

		assert chunk.id not in self.id_chunk_map
		self.chunks.append(chunk)
		self.id_chunk_map[chunk.id] = chunk

class ChunkEntry(object):

	def __init__(self, chunk, pos):

		self.chunk = chunk
		self.pos = pos

class IDError(Exception):
	pass

class OffsetMapEntry(object):

	def __init__(self, bounds):
		self.bounds = bounds

	def format(self):
		print '[%.8X - %.8X] %s' % (self.bounds[0], self.bounds[1], self.get_description())

class FileHeaderOffsetMapEntry(OffsetMapEntry):

	def __init__(self, bounds):
		OffsetMapEntry.__init__(self, bounds)

	def get_description(self):
		return 'File Header'

class ChunkListOffsetMapEntry(OffsetMapEntry):

	def __init__(self, bounds):
		OffsetMapEntry.__init__(self, bounds)

	def get_description(self):
		return 'Chunk List'

class ChunkOffsetMapEntry(OffsetMapEntry):

	def __init__(self, bounds, chunk):
		OffsetMapEntry.__init__(self, bounds)
		self.chunk = chunk

	def get_description(self):
		return 'Chunk ID = 0x%X: %s' % (self.chunk.id, self.chunk.get_name())

class ChunkFile(object):

	def __init__(self):

		self.doc = ChunkDocument()
		self.chunk_entries = []
		self.id_entry_map = {}
		self.offset_map = []

	def add_chunk(self, chunk, pos):
		
		assert chunk.id not in self.id_entry_map
		entry = ChunkEntry(chunk, pos)
		self.chunk_entries.append(entry)
		self.id_entry_map[chunk.id] = entry
		self.doc.add_chunk(chunk)

	def get_chunk(self, chunk_id):
		try:
			chunk = self.id_entry_map[chunk_id] 
		except KeyError:
			raise IDError
		return chunk.pos, chunk.chunk

	def iter_chunk_entries(self):

		for entry in self.chunk_entries:
			try:
				name = entry.chunk.get_name()
			except chunks.UnknownChunkError:
				name = '[UNKNOWN CHUNK TYPE]'
			yield entry.chunk.id, entry.chunk.typecode, entry.pos, name

def read_header(f, offset_map):
	header_fmt = '6s2xiii'
	offset_map.append(FileHeaderOffsetMapEntry((f.tell(), f.tell() + struct.calcsize(header_fmt))))
	header = f.read(struct.calcsize(header_fmt))
	sig, filetype, version, chunktablepos = struct.unpack(header_fmt, header)
	assert sig == 'CryTek'
	return chunktablepos, version

entry_fmt = ''
def read_chunk_table_entry(f, v):
	listent_fmt = 'Iiii'
	if ( v > 1860 ):
		listent_fmt = 'Iiiii'
		
	entry_fmt = listent_fmt
	entry = f.read(struct.calcsize(listent_fmt))
	chunksize = 0
	if ( v > 1860 ):
		type, version, pos, id, chunksize = struct.unpack(listent_fmt, entry)
	else:
		type, version, pos, id = struct.unpack(listent_fmt, entry)
	return id, type, version, pos, chunksize

def read_chunk_list_header(f, offset_map):
	header_fmt = 'i'
	position = f.tell()
	header = f.read(struct.calcsize(header_fmt))
	num_chunks, = struct.unpack(header_fmt, header)
	offset_map.append(ChunkListOffsetMapEntry((position, position + struct.calcsize(header_fmt) + num_chunks * struct.calcsize(entry_fmt))))
	return num_chunks

def load_chunk_file(filename):

	chunk_file = ChunkFile()

	f = file(filename, 'rb')

	f.seek(0, 2)
	file_size = f.tell()
	f.seek(0, 0)

	chunktablepos,fileversion = read_header(f, chunk_file.offset_map)
	

	f.seek(chunktablepos)
	num_chunks = read_chunk_list_header(f, chunk_file.offset_map)
	
	chunk_descriptors = []
	for index in xrange(num_chunks):
		id, type, version, pos, chunksize = read_chunk_table_entry(f,fileversion)
		chunk_descriptors.append((id, type, version, pos))

	chunk_descriptors.sort(lambda a,b: cmp(a[3], b[3]))

	chunk_starts = [pos for id, type, version, pos in chunk_descriptors]
	chunk_ends = chunk_starts[1:] + [file_size]

	# If the chunk table is at the end of the file, then the last chunk ends with it.
	if chunktablepos > chunk_starts[-1]:
		chunk_ends[-1] = chunktablepos

	chunk_sizes = [end - start for start, end in zip(chunk_starts, chunk_ends)]
	chunk_descriptors = zip(chunk_descriptors, chunk_sizes)

	for (id, type, version, pos), size in chunk_descriptors:
		f.seek(pos)
		data = f.read(size)
		chunk = chunks.create_chunk(id, type, version, data)
		chunk_file.add_chunk(chunk, pos)
		chunk_file.offset_map.append(ChunkOffsetMapEntry((pos, pos + size), chunk))

	return chunk_file
