from chunk import Chunk, VersionError, DataReader
from chunk_utils import *
import struct

geom_types = ([
	('GEOM_TRIMESH',    	1, 'format_trimesh_primitive_data'),
	('GEOM_HEIGHTFIELD',	2, None),
	('GEOM_CYLINDER',   	5, None),
	('GEOM_CAPSULE',    	6, None),
	('GEOM_RAY',        	3, None),
	('GEOM_SPHERE',     	4, None),
	('GEOM_BOX',        	0, 'format_box_primitive_data'),
	('GEOM_VOXELGRID',   	7, None)
	])

bv_tree_types = ([
	('BVT_OBB', 0, None),
	('BVT_AABB', 1, 'format_trimesh_aabb_tree'),
	('BVT_SINGLEBOX', 2, 'format_trimesh_box_tree'),
	('BVT_RAY', 3, None),
	('BVT_HEIGHTFIELD', 4, None),
	('BVT_VOXEL', 5, None)
	])

class MeshPhysicsDataChunk(Chunk):

	def format(self, pos):

		if self.version != 0x800:
			raise VersionError

		reader = DataReader(self.data)

		header = self.read_header(reader)
		header.format(pos)

		data_length = header.find_field('nDataSize').value
		physics_data_header = self.read_physics_data(reader, data_length)
		physics_data_header.format(pos)

		geom_type = physics_data_header.find_field('GeomType').get_enum_string()
		geom_readers = dict([(name, func) for name, index, func in geom_types])
		func = geom_readers[geom_type]
		if func:
			getattr(self, func)(reader, pos)
		else:
			print '<GEOM TYPE NOT UNDERSTOOD>'

	def read_header(self, reader):

		field_defs = ([
			FieldDef('type', 'I', hex),
			FieldDef('version', 'i', hex),
			FieldDef('pos', 'i', hex),
			FieldDef('id', 'i', hex),
			FieldDef('nDataSize', 'i'),
			FieldDef('nFlags (unused)', 'i', hex),
			FieldDef('nTetraHedraDataSize', 'i'),
			FieldDef('nTetreHedraChunkId', 'i'),
			FieldDef('reserved', '2i')
			])

		return read_fields('Mesh Physics Data Header', field_defs, reader)

	def read_physics_data(self, reader, length):

		physics_types = dict([(index, name) for name, index, func in geom_types])

		field_defs = ([
			FieldDef('version', 'i'),
			FieldDef('dummy0', 'i'),
			FieldDef('Ibody', '3f'),
			FieldDef('q', '4f'),
			FieldDef('origin', '3f'),
			FieldDef('V', 'f'),
			FieldDef('nRefCount', 'i'),
			FieldDef('surface_idx', 'i'),
			FieldDef('dummy1', 'i'),
			FieldDef('nMats', 'i'),
			EnumDef('GeomType', 'i', physics_types)
			])

		return read_fields('Physics Data', field_defs, reader)

	def format_box_primitive_data(self, reader, pos):

		self.read_box(reader, 'm_box').format(pos)
		self.read_box(reader, 'm_Tree.m_Box').format(pos)
		self.read_prims_count(reader).format(pos)
		
	def read_prims_count(self, reader):
		field_defs = ([
			FieldDef('m_nPrims', 'i')
			])
		return read_fields('m_Tree prims count', field_defs, reader)

	def read_box(self, reader, title):

		field_defs = ([
			FieldDef('Basis[0]', '3f'),
			FieldDef('Basis[1]', '3f'),
			FieldDef('Basis[2]', '3f'),
			FieldDef('bOriented', 'i'),
			FieldDef('center', '3f'),
			FieldDef('size', '3f'),
			])

		return read_fields(title, field_defs, reader)

	def format_trimesh_primitive_data(self, reader, pos):
		
		header = self.read_trimesh_primitive_header(reader)
		header.format(pos)

		has_vertex_map_header = self.read_trimesh_has_vertex_map_flag(reader)
		has_vertex_map_header.format(pos)

		num_vertices = header.find_field('m_nVertices').value
		if has_vertex_map_header.find_field('bVtxMap').value:
			for index in xrange(num_vertices):
				self.read_vertex_map_element(reader, index).format(pos)

		has_foreign_idx_header = self.read_trimesh_has_foreign_indices_flag(reader)
		has_foreign_idx_header.format(pos)

		num_tris = header.find_field('m_nTris').value
		if has_foreign_idx_header.find_field('bForeignIdx').value:
			for index in xrange(num_tris):
				self.read_foreign_idx(reader, index).format(pos)
		else:
			for index in xrange(num_vertices):
				self.read_vertex(reader, index).format(pos)
			for index in xrange(num_tris):
				self.read_triangle_indices(reader, index).format(pos)

			has_ids_header = self.read_trimesh_has_ids_flag(reader)
			has_ids_header.format(pos)
			if has_ids_header.find_field('bIds').value:
				for index in xrange(num_tris):
					self.read_trimesh_id(reader, index).format(pos)

		self.read_trimesh_dummy(reader).format(pos)

		i_header = self.read_trimesh_i_flag(reader)
		i_header.format(pos)

		if i_header.find_field('i').value:
			return

		for index in xrange(4):
			self.read_trimesh_convexity(reader, index).format(pos)

		tree_type_header = self.read_trimesh_tree_type_header(reader)
		tree_type_header.format(pos)

		tree_type = tree_type_header.find_field('type').get_enum_string()
		tree_funcs = dict([(name, func) for name, index, func in bv_tree_types])
		func = tree_funcs[tree_type]
		if func:
			getattr(self, func)(reader, pos)
		else:
			print '<TREE TYPE NOT UNDERSTOOD>'

	def read_trimesh_primitive_header(self, reader):
		mesh_flags = ([
			'mesh_shared_vtx',
			'mesh_shared_idx',
			'mesh_shared_mats',
			'mesh_shared_foreign_idx',
			'mesh_shared_normals',
			'mesh_OBB',
			'mesh_AABB',
			'mesh_SingleBB',
			'mesh_multicontact0',
			'mesh_multicontact1',
			'mesh_multicontact2',
			'mesh_approx_cylinder0',
			'mesh_approx_box',
			'mesh_approx_sphere ',
			'<UNKNOWN_FLAG>',
			'mesh_keep_vtxmap',
			'mesh_keep_vtxmap_for_saving',
			'mesh_no_vtx_merge',
			'mesh_AABB_rotated',
			'mesh_VoxelGrid',
			'mesh_always_static',
			'mesh_approx_capsule'
			])

		field_defs = ([
			FieldDef('m_nVertices', 'i'),
			FieldDef('m_nTris', 'i'),
			FieldDef('m_nMaxVertexValency', 'i'),
			FlagsDef('m_flags', 'i', mesh_flags)
			])

		return read_fields('Trimesh Primitive', field_defs, reader)


	def read_trimesh_has_vertex_map_flag(self, reader):

		field_defs = ([
			FieldDef('bVtxMap', 'B')
			])

		return read_fields('HasVertexMap', field_defs, reader)

	def read_trimesh_has_foreign_indices_flag(self, reader):

		field_defs = ([
			FieldDef('bForeignIdx', 'B')
			])

		return read_fields('HasForeignIdx', field_defs, reader)

	def read_vertex_map_element(self, reader, index):

		field_defs = ([
			FieldDef('index', 'H')
			])

		return read_fields('Vertex Map Element (%d)' % index, field_defs, reader)

	def read_foreign_idx(self, reader, index):

		field_defs = ([
			FieldDef('index', 'H')
			])

		return read_fields('Foreign Idx (%d)' % index, field_defs, reader)

	def read_vertex(self, reader, index):

		field_defs = ([
			FieldDef('v', '3f')
			])

		return read_fields('Vertex (%d)' % index, field_defs, reader)

	def read_triangle_indices(self, reader, index):

		field_defs = ([
			FieldDef('indices', '3H')
			])

		return read_fields('Triangle Indices (%d)' % index, field_defs, reader)

	def read_trimesh_has_ids_flag(self, reader):

		field_defs = ([
			FieldDef('bIds', 'B')
			])

		return read_fields('HasIds', field_defs, reader)

	def read_trimesh_id(self, reader, index):

		field_defs = ([
			FieldDef('id', 'B')
			])

		return read_fields('Id (%d)' % index, field_defs, reader)

	def read_trimesh_dummy(self, reader):

		field_defs = ([
			FieldDef('dummy', '4i')
			])

		return read_fields('Dummy', field_defs, reader)

	def read_trimesh_i_flag(self, reader):

		field_defs = ([
			FieldDef('i', 'i')
			])

		return read_fields('I', field_defs, reader)

	def read_trimesh_convexity(self, reader, index):

		field_defs = ([
			FieldDef('m_bConvex', 'i'),
			FieldDef('m_ConvexityTolerance', 'f')
			])

		return read_fields('Convexity (%d)' % index, field_defs, reader)

	def read_trimesh_tree_type_header(self, reader):

		bv_enum_defs = dict([(index, name) for name, index, func in bv_tree_types])

		field_defs = ([
			EnumDef('type', 'i', bv_enum_defs)
			])

		return read_fields('Trimesh Tree Type Header', field_defs, reader)

	def format_trimesh_box_tree(self, reader, pos):

		self.read_box(reader, 'm_Box').format(pos)
		self.read_prims_count(reader).format(pos)

	def format_trimesh_aabb_tree(self, reader, pos):

		version_header = self.read_aabb_tree_version(reader)
		version_header.format(pos)
		version = version_header.find_field('ver').value

		if version >= 0:
			num_nodes = version
		else:
			num_nodes_header = self.read_aabb_tree_num_nodes(reader)
			num_nodes_header.format(pos)
			num_nodes = num_nodes_header.find_field('m_nNodes').value

		if version == -1:
			for index in xrange(num_nodes):
				self.read_aabb_node(reader, index).format(pos)
		elif version >= 0:
			print '<AABB TREE VERSION NOT SUPPORTED>'
			return

		self.read_aabb_data(reader).format(pos)

	def read_aabb_tree_version(self, reader):

		field_defs = ([
			FieldDef('ver', 'i')
			])

		return read_fields('AABB Tree Version', field_defs, reader)

	def read_aabb_tree_num_nodes(self, reader):

		field_defs = ([
			FieldDef('m_nNodes', 'i')
			])

		return read_fields('Num Nodes', field_defs, reader)

	def read_aabb_node(self, reader, index):

		field_defs = ([
			FieldDef('ichild', 'i'),
			FieldDef('minx', 'B'),
			FieldDef('maxx', 'B'),
			FieldDef('miny', 'B'),
			FieldDef('maxy', 'B'),
			FieldDef('minz', 'B'),
			FieldDef('maxz', 'B'),
			FieldDef('ntris', 'B'),
			FieldDef('bSingleColl', 'B')
			])

		return read_fields('AABB Node (%d)' % index, field_defs, reader)

	def read_aabb_data(self, reader):

		field_defs = ([
			FieldDef('m_center', '3f'),
			FieldDef('m_size', '3f'),
			FieldDef('m_Basis[0]', '3f'),
			FieldDef('m_Basis[1]', '3f'),
			FieldDef('m_Basis[2]', '3f'),
			FieldDef('m_nMaxTrisInNode', 'i'),
			FieldDef('m_nMinTrisPerNode', 'i'),
			FieldDef('m_nMaxTrisPerNode', 'i'),
			FieldDef('m_maxSkipDim', 'f')
			])

		return read_fields('AABB Data', field_defs, reader)

