////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek, 1999-2009.
// -------------------------------------------------------------------------
//  File name:   Config.cpp
//  Version:     v1.00
//  Created:     10/10/2009 by Alex McCarthy (refactored from StatsTool.cpp).
//  Description: Common code to parse configuration files
////////////////////////////////////////////////////////////////////////////

// Headers
#include "stdafx.h"

#include <fstream>
#include <iostream>

#include "Config.h"
#include "Job.h"

using namespace std;

enum EParseResult
{
	ePR_SUCCESS,
	ePR_FAILURE,
	ePR_EOF,
};

void FixSlashes(string& sDirectory)
{
	const size_t length = sDirectory.size();
	if(length == 0)
	{
		return;
	}

	size_t slashPos;
	while ( ( slashPos = sDirectory.find( '\\' ) ) != string::npos )
	{
		sDirectory.replace(slashPos, 1, "/");
	}

	if(sDirectory[length - 1] != '/')
	{
		sDirectory.append("/");
	}
}

string GetTrimmmed(const string& sText)
{
	size_t firstCharPos = sText.find_first_not_of(" \t");
	if ( firstCharPos == string::npos )
	{
		firstCharPos = 0;
	}

	size_t lastCharPos = sText.find_last_not_of(" \t");

	return sText.substr(firstCharPos, lastCharPos);
}

template<typename T>
static EParseResult Parse( ifstream& file, T& val )
{
	return (file >> val) ? ePR_SUCCESS : ePR_FAILURE;
}

static EParseResult Parse( ifstream& file, std::string& str )
{
	// Overridden as we want to ignore all white space and just stop at eol
	file >> ws;
	std::getline( file, str, '\n' );
	if ( file.eof() )
	{
		return ePR_EOF;
	}
	else if ( file.bad() )
	{
		return ePR_FAILURE;
	}
	else
	{
		return ePR_SUCCESS;
	}
}

static EParseResult Parse( ifstream& file, bool& val )
{
	string sToken;
	if( !(file >> sToken)) return ePR_FAILURE;

	if ( !sToken.compare("true") )
	{
		val = true;
	}
	else if ( !sToken.compare("false") )
	{
		val = false;
	}
	else if ( !sToken.compare("1") )
	{
		val = true;
	}
	else if ( !sToken.compare("0") )
	{
		val = false;
	}
	else
	{
		return ePR_FAILURE;
	}

	return ePR_SUCCESS;
}

static EParseResult Parse( ifstream& file, SConfig::EPlatform& ePlatform )
{
	string sPlatform;
	if( !(file >> sPlatform) ) return ePR_FAILURE;

	if ( !sPlatform.compare( "ps3" ) )
	{
		ePlatform = SConfig::ePL_PS3;
	}
	else if ( !sPlatform.compare( "360" ) )
	{
		ePlatform = SConfig::ePL_XBOX360;
	}
	else if ( !sPlatform.compare( "xbox" ) )
	{
		ePlatform = SConfig::ePL_XBOX360;
	}
	else if ( !sPlatform.compare( "xbox360" ) )
	{
		ePlatform = SConfig::ePL_XBOX360;
	}
	else
	{
		return ePR_FAILURE;
	}

	return ePR_SUCCESS;
}

static EParseResult Parse( ifstream& file, SConfig::EPollCVARAction& eAction )
{
	string sAction;
	if( !(file >> sAction) ) return ePR_FAILURE;

	if ( !sAction.compare( "none" ) )
	{
		eAction = SConfig::ePCA_NONE;
	}
	else if ( !sAction.compare( "finish" ) )
	{
		eAction = SConfig::ePCA_FINISH;
	}
	else
	{
		return ePR_FAILURE;
	}

	return ePR_SUCCESS;
}

static EParseResult Parse( ifstream& file, SConfig::ELoggingType &eLoggingType )
{
	string sLogType;
	if ( !(file >> sLogType) ) return ePR_FAILURE;

	if ( !sLogType.compare( "memreplay") )
	{
		eLoggingType = SConfig::eLT_MEM_REPLAY;
	}
	else if ( !sLogType.compare( "perfstats") )
	{
		eLoggingType = SConfig::eLT_PERF_STATS;
	}
	else
	{
		return ePR_FAILURE;
	}

	return ePR_SUCCESS;
}

static EParseResult Parse( ifstream& file, CMapList& mapList )
{
	string sMapName;
	if(!getline(file, sMapName)) return ePR_FAILURE;

	sMapName = GetTrimmmed(sMapName);

	mapList.AddMap( sMapName );

	return ePR_SUCCESS;
}

static EParseResult Parse( ifstream& file, std::vector<std::string>& stringList )
{
	string str;
	if(!getline(file, str)) return ePR_FAILURE;

	str = GetTrimmmed(str);

	stringList.push_back( str );

	return ePR_SUCCESS;
}

static EParseResult ParseConfigLine( SConfig& config, ifstream& file, int& iLine, JobList* const pJobList )
{
	string sCommand;
	if( !(file >> sCommand) )
	{
		return file.eof() ? ePR_EOF : ePR_FAILURE;
	}

	++iLine;

	if ( !sCommand.compare( "postprocesstool" ) )
	{
		return Parse( file, config.sPostprocessTool );
	}
	else if ( !sCommand.compare( "pauseonconnect" ) )
	{
		return Parse( file, config.bPauseOnConnect );
	}
	else if ( !sCommand.compare( "pauseonerror" ) )
	{
		return Parse( file, config.bPauseOnError );
	}
	else if ( !sCommand.compare( "executable" ) )
	{
		return Parse( file, config.sExecutable );
	}
	else if ( !sCommand.compare( "arguments" ) )
	{
		return Parse( file, config.sArguments );
	}
	else if ( !sCommand.compare( "maxerrors" ) )
	{
		return Parse( file, config.iMaxErrors );
	}
	else if ( !sCommand.compare( "platform" ) )
	{
		return Parse( file, config.ePlatform );
	}
	else if ( !sCommand.compare( "verbose" ) )
	{
		return Parse( file, config.bVerbose );
	}
	else if ( !sCommand.compare( "showtty" ) )
	{
		return Parse( file, config.bOutputTTY );
	}
	else if ( !sCommand.compare( "addmap" ) )
	{
		if ( pJobList ) config.mapList.Clear();
		EParseResult res = Parse( file, config.mapList );
		if ( pJobList )
		{
			pJobList->push_back( SJob( &config ) );
		}
		return res;
	}
	else if ( !sCommand.compare( "memoryStatsDir") )
	{
		return Parse( file, config.sMemoryStatsDir );
	}
	else if ( !sCommand.compare( "perfStatsDir") )
	{
		return Parse( file, config.sPerfStatsDir );
	}
	else if ( !sCommand.compare( "reportOutDir") )
	{
		return Parse( file, config.sReportOutDir );
	}
	else if ( !sCommand.compare( "target" ) )
	{
		return Parse( file, config.sTargetName );
	}
	else if ( !sCommand.compare( "cycles" ) )
	{
		// Cycles are deprecated but we should maintain compatibility for now
		int i;
		fprintf( stderr, "[Warning] 'cycles' configuration line is now deprecated. It can be emulated with multiple 'addmap' lines.\n" );
		return Parse( file, i );
	}
	else if ( !sCommand.compare( "delay" ) )
	{
		return Parse( file, config.iDelay );
	}
	else if ( !sCommand.compare( "log" ) )
	{
		return Parse( file, config.eLoggingType );
	}
	else if ( !sCommand.compare( "pipeName" ) )
	{
		return Parse( file, config.sPipeName );
	}
	else if ( !sCommand.compare( "confluenceSpace" ) )
	{
		return Parse( file, config.sConfluenceSpace );
	}
	else if ( !sCommand.compare( "confluencePage" ) )
	{
		return Parse( file, config.sConfluencePage );
	}
	else if ( !sCommand.compare( "confluenceUser" ) )
	{
		return Parse( file, config.sConfluenceUser );
	}
	else if ( !sCommand.compare( "confluencePassword" ) )
	{
		return Parse( file, config.sConfluencePassword );
	}
	else if ( !sCommand.compare( "gnuplotPath" ) )
	{
		return Parse( file, config.sGnuplotPath );
	}
	else if ( !sCommand.compare( "perfstatsperiod" ) )
	{
		return Parse( file, config.iPerfStatsPeriod );	// this could be made a float
	}
	else if ( !sCommand.compare( "pollCVARName" ) )
	{
		return Parse( file, config.sPollCVARName );
	}
	else if ( !sCommand.compare( "pollCVARValue" ) )
	{
		return Parse( file, config.iPollCVARValue );
	}
	else if ( !sCommand.compare( "pollCVARAction" ) )
	{
		return Parse( file, config.ePollCVARAction );
	}
	else if ( !sCommand.compare( "smtpServer" ) )
	{
		return Parse( file, config.sSMTPServer );
	}
	else if ( !sCommand.compare( "smtpPort" ) )
	{
		return Parse( file, config.uSMTPPort );
	}
	else if ( !sCommand.compare( "telnet" ) )
	{
		return Parse( file, config.bTelnet );
	}
	else if ( !sCommand.compare( "telnetPort" ) )
	{
		return Parse( file, config.uTelnetPort );
	}
	else if ( !sCommand.compare( "smtpSender" ) )
	{
		return Parse( file, config.sSMTPSender );
	}
	else if ( !sCommand.compare( "subjectSuffixFile" ) )
	{
		return Parse( file, config.sSubjectSuffixFile );
	}
	else if ( !sCommand.compare( "addRecipient" ) )
	{
		return Parse( file, config.recipients );
	}
	else if ( sCommand[0] == '-' && sCommand[1] == '-' )
	{
		std::string dummy;
		return Parse( file, dummy );
	}
	else
	{
		return ePR_FAILURE;
	}
}

SConfig::SConfig()
: ePlatform(ePL_NONE)
, sSMTPSender( "stephen@crytek.com" )
, bVerbose(true)
, bOutputTTY(true)
, bPauseOnConnect(false)
, bPauseOnError(false)
, bTelnet(false)
, iDelay(20)
, iMaxErrors(-1)
, uSMTPPort(25)
, uTelnetPort(3100)
{
}

bool SConfig::ParseConfigFile( const char* const pConfig, SLogger* pLogger/* = NULL*/, JobList* const pJobList/* = NULL*/ )
{
	if ( !pConfig ) return false;

	ifstream file(pConfig);
	if ( !file.good() ) return false;

	// Now we can process the file line by line
	EParseResult eRes;
	int iLine = 0;
	do
	{
		eRes = ParseConfigLine( *this, file, iLine, pJobList );
	}
	while ( eRes == ePR_SUCCESS );

	file.close();

	// Last parse result should be EoF
	if ( eRes != ePR_EOF )
	{
		if ( pLogger )
		{
			pLogger->LogError( "Parsing config file line %i.\n", iLine );
		}
		else
		{
			fprintf( stderr, "[Error] Parsing config file line %i.\n", iLine );
		}
		return false;
	}

	FixSlashes(sMemoryStatsDir);
	FixSlashes(sReportOutDir);

	return true;
}
