////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek, 1999-2009.
// -------------------------------------------------------------------------
//  File name:   ConsolePS3.cpp
//  Version:     v1.00
//  Created:     28/09/2009 by Steve Barnett.
//  Description: Class for communication the PS3
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include <windows.h>
#include <algorithm>
#include <assert.h>
#include <conio.h>
#include <sstream>
#include <malloc.h>
#include <stdio.h>
#include <string>
#include <time.h>
#include <vector>

#include "tmver.h"
#include "PS3tmapi.h"

#include "ConsolePS3.h"

#define FORCE_DISCONNECT																								(1)
#define TIMEOUT																													(600)

using namespace std;

void sleep(unsigned int mseconds);	// Declared in StatsTool.cpp

// Static methods

int CConsolePS3::EnumCallBack( HTARGET hTarget, void* pInstance )
{
	CConsolePS3* pConcreteInstance = static_cast<CConsolePS3*>( pInstance );
	return pConcreteInstance->HandleEnumCallBack( hTarget );
}

void __stdcall CConsolePS3::ProcessTTYCallback( HTARGET hTarget, UINT uiType, UINT uiStreamId, SNRESULT result, UINT uiLength, BYTE *pData, void* pInstance )
{
	CConsolePS3* pConcreteInstance = static_cast<CConsolePS3*>( pInstance );
	pConcreteInstance->HandleProcessTTYCallback( hTarget, uiType, uiStreamId, result, uiLength, pData );
}

// End static methods

// Externally visible methods

CConsolePS3::CConsolePS3( const char* const pExePath, const char* const pArguments, const char* const pTarget, const char* const pPipeName, SLogger* const pLogger, bool verbose/* = false*/, bool outputTTY/* = true*/ )
: IConsoleInterface( pExePath, pArguments, pTarget, pPipeName, pLogger, verbose, outputTTY )
, m_existingConnection( false )
, m_pTTYStreams( NULL )
, m_numTTYStreams( 0 )
, m_pSelfFilename( NULL )
, m_pHomeDirectory( NULL )
, m_pipeBufferSize( BUFFERSIZE )
{
	// Process the exe path into a home directory and filename

	// Scan through for the last slash
	const char* pCur = pExePath;
	const char* pLastSlash = pExePath;
	while ( *pCur != '\0' )
	{
		if ( *pCur == '/' || *pCur == '\\' )
		{
			pLastSlash = pCur;
		}
		++pCur;
	}

	// Allocate buffers for path and self filename
	if ( m_pExePath )
	{
		const int pathLen = (pLastSlash - m_pExePath);
		m_pHomeDirectory = new char[pathLen + 1];
		if ( m_pHomeDirectory )
		{
			strncpy_s( m_pHomeDirectory, pathLen + 1, m_pExePath, pathLen );
			m_pHomeDirectory[pathLen] = '\0';
		}
		++pLastSlash;
		const int selfLen = strlen( pLastSlash );
		m_pSelfFilename = new char[selfLen + 1];
		if ( m_pSelfFilename )
		{
			strncpy_s( m_pSelfFilename, selfLen + 1, pLastSlash, selfLen );
			m_pSelfFilename[selfLen] = '\0';
		}
	}
}

CConsolePS3::~CConsolePS3( void )
{
	// Deallocate buffers
	delete m_pSelfFilename;
	m_pSelfFilename = NULL;
	delete m_pHomeDirectory;
	m_pHomeDirectory = NULL;
}

bool CConsolePS3::Connect( void )
{
	m_existingConnection = false;
	bool needsShutdown = false;
	char* pUsage = NULL;
	SNRESULT snr = SN_S_OK;

	// Initialise PS3tmapi.
	if (SN_FAILED( snr = SNPS3InitTargetComms() ))
	{
		PrintError( "Failed to initialize PS3TM SDK", snr );
		return false;
	}

	// Enumerate available targets.
	if (SN_FAILED( snr = SNPS3EnumTargetsEx( EnumCallBack, (void*)this ) ) )
	{
		PrintError( "Failed to enumerate targets", snr );
		return false;
		Shutdown();
	}

	// Check there is at least one target.
	if (m_Targets.empty())
	{
		PrintError( "No targets available" );
		Shutdown();
		return false;
	}

	// Attempt to get the target name from an environment variable if not set...
	if (m_pTarget == NULL)
	{
		m_pTarget = getenv("PS3TARGET");
	}

	// If there is only one target available, always use it.
	if (m_pTarget == NULL && m_Targets.size() == 1)
	{
		m_pTarget = (char*)m_Targets[0]->pszName;
	}
		
	// If no target has been selected then use the first one connected or the first one available.
	if (m_pTarget == NULL)
	{
		if (FindFirstConnectedTarget() == false && FindFirstAvailableTarget() == false)
		{
			PrintError( "No targets available" );
			Shutdown();
			return false;
		}
	}

	// Retrieve the target ID from the name or failing that IP.
	if (SN_FAILED( SNPS3GetTargetFromName(m_pTarget, &m_hTargetId) ) && GetTargetFromIP(m_pTarget, m_hTargetId) == false )
	{
		PrintError( "Target not found" );
		Shutdown();
		return false;
	}

	// Connect to the target.
	if ( SN_FAILED( snr = SNPS3Connect(m_hTargetId, NULL) ) )
	{
		if (snr == SN_E_TARGET_IN_USE && FORCE_DISCONNECT )
		{
			if (SN_FAILED( SNPS3ForceDisconnect(m_hTargetId) ))
			{
				PrintError( "Unable to force disconnect", pUsage );
				Shutdown();
				return false;
			}

			snr = SNPS3Connect(m_hTargetId, NULL);
		}
						
		if (SN_FAILED( snr ))
		{
			PrintError( "Failed to connect to target", snr );
			Shutdown();
			return false;
		}

		m_existingConnection = (snr == SN_S_CONNECTED);
	}

	if ( IsVerbose() )
		printf( "Info - [%s] Connected to target\n", m_pTarget );
												
	// Configure TTYs
	if (InitializeTTYStreams() == false)
		return false;
		
	u_int uChannel = -1;
	if ( SN_FAILED( SNPS3RegisterTTYEventHandler( m_hTargetId, uChannel, ProcessTTYCallback, (void*)this ) ) )
	{
		PrintError( "Failed to register for TTY notifications" );
		Shutdown();
		return false;
	}

	if ( IsVerbose() )
		printf("Info - [%s] Registered for TTY output on %s channel\n", m_pTarget, "every" );

	return true;
}

// hardReset defaults to true when called via Reset
bool CConsolePS3::ResetImpl( const vector<string> &cmdLineParams, bool hardReset )
{
		// Reset the console
		SNPS3ClearTTYCache(m_hTargetId);
			
		const UINT64 uBootParams = SNPS3TM_BOOTP_DEBUG_MODE | SNPS3TM_BOOTP_NIC_MASK;	// NIC_MASK is used here to set that 1 bit
		const UINT64 uBootMask = SNPS3TM_BOOTP_BOOT_MODE_MASK | SNPS3TM_BOOTP_NIC_MASK;

		const UINT64 uResetParams = hardReset ? SNPS3TM_RESETP_HARD_RESET : SNPS3TM_RESETP_SOFT_RESET;
		const UINT64 uResetMask = SNPS3TM_RESETP_ALL_MASK;

		const UINT64 uSystemParams = 0ui64;
		const UINT64 uSystemMask = 0ui64;

		SNRESULT snr = SNPS3ResetEx( m_hTargetId, uBootParams, uBootMask, uResetParams, uResetMask, uSystemParams, uSystemMask );

		SetHomeDir();
		SetFSDir();

		if (SN_FAILED( snr ))
		{
			PrintError( "Failed to reset target" );
			Shutdown();
			return false;
		}
		else
		{
			if ( IsVerbose() )
				printf( "Info - [%s] Reset target with system params %I64d\n\t(bootparam:0x%I64x, bootmask:0x%I64x)\n\t(resetparam:0x%I64x, resetmask:0x%I64x)\n", m_pTarget, uSystemParams, uBootParams, uBootMask, uResetParams, uResetMask );
			
			// Load the elf
			UINT	uModuleID;
			UINT64	uThreadID;
			
			// Download the ELF into target memory.
			TArguments sArguments = GetArguments( cmdLineParams );
			const int argc = sArguments.size();
			assert(argc != 0);
			const char** argv = &sArguments[0];
			snr = SNPS3ProcessLoad( m_hTargetId, SNPS3_DEF_PROCESS_PRI, m_pSelfFilename, argc, argv, 0, NULL, &uModuleID, &uThreadID, 0 );
			for (int i = 0; i < argc; ++i)
			{
				delete[] sArguments[i];
			}
			sArguments.clear();

			if (SN_FAILED( snr ))
			{
				PrintError( "Failed to load self", m_pSelfFilename );
				Shutdown();
				return false;
			}
			
			if ( IsVerbose() )
				printf( "Info - [%s] Loaded %s into memory, ID=0x%X\n", m_pTarget, m_pSelfFilename, uModuleID );
	}

		return true;
}

void CConsolePS3::Shutdown( void )
{
	// Pump pending TTY before exiting
	ClearMessagesBeforeExit();
	
	// Unregister TTY notifications before exiting.
	(void) DisconnectTTYEvents();
	
	// Check if we should disconnect from the target.
	if ( !m_existingConnection )
	{
		SNPS3Disconnect(m_hTargetId);

		if ( IsVerbose() )
			printf("Info - [%s] Disconnected from target\n", m_pTarget);
	}

	// Power off the target
	SNPS3PowerOff( m_hTargetId, 1 );
	
	// Uninitialise PS3tmapi.
	SNPS3CloseTargetComms();
	
	// Deallocate target memory.
	FreeTargetList();
}

void CConsolePS3::UpdateComms( void )
{

	SNRESULT snr = SN_S_OK;
	int nKicks = 0;

	do
	{
		snr = SNPS3Kick();
		++nKicks;

	} while (snr == SN_S_OK );

	if ( SN_FAILED( snr ) )
	{
		printf( "Info - [%s] Target requested shutdown\n", m_pTarget );
		m_quit = true;
	}
}

bool CConsolePS3::CreatePipe( void )
{
	string sPipePath( "\\\\.\\pipe\\CrysisTargetComms" );
	sPipePath += m_pPipeName;
	sPipePath += ".pipe";

	m_hPipe = ::CreateNamedPipeA( sPipePath.c_str(), PIPE_ACCESS_DUPLEX/*|FILE_FLAG_WRITE_THROUGH*/, PIPE_TYPE_BYTE|PIPE_NOWAIT, 1, 1024, 1024, 500/*ms*/, NULL );

	if (m_hPipe == INVALID_HANDLE_VALUE) 
	{
		PrintError( "Failed to create pipe" );
		return false;
	}

	if ( IsVerbose() )
		printf( "Info - [%s] Waiting for target using pipe %s\n", m_pTarget, sPipePath.c_str() );

	// Wait for the client to connect; if it succeeds,
	// the function returns a nonzero value. If the function
	// returns zero, GetLastError returns ERROR_PIPE_CONNECTED.
	bool success = false;
	int seconds = 0;
	while ( !success && seconds < TIMEOUT )
	{
		success = ::ConnectNamedPipe(m_hPipe, NULL) ? TRUE : (::GetLastError() == ERROR_PIPE_CONNECTED);
		sleep( 1000 ); seconds++;
	}
	if ( !success )
	{
		ClosePipe();
	}
	return success;
}

void CConsolePS3::SendToPipe( const char* const pCommand )
{
	DWORD sent = strlen( pCommand );
	DWORD received = -1;
	const char* pCur = pCommand;

	if ( pCur )
	{
		bool allSent = false;
		while ( !allSent )
		{
			::WriteFile( m_hPipe, pCommand, sent, &received, NULL );
			allSent = ( received >= sent );
			sent -= received;
			pCur += received;
		}
	}
}

const char* CConsolePS3::ReadFromPipe( char* const pBuffer, const int len )
{
	bool success = false;
	int seconds = 0;
	const int maxLen = min( len, BUFFERSIZE );
	assert( pBuffer );
	while ( !success && seconds < TIMEOUT )
	{
		success = ::ReadFile( m_hPipe, &m_pPipeBuffer, maxLen, &m_pipeReadChars, NULL );
		sleep( 1000 ); seconds++;
	}
	// Copy from pipe buffer to external buffer
	if ( success )
	{
		strncpy_s( pBuffer, maxLen, m_pPipeBuffer, maxLen - 1 );
		pBuffer[maxLen - 1] = '\0';
	}
	return success ? pBuffer : NULL;
}

void CConsolePS3::ClosePipe( void )
{
	::CloseHandle(m_hPipe);
}

bool CConsolePS3::WaitForConnected( void )
{
	// Should recieve "Connected." once target is ready
	char szBuffer[BUFFERSIZE];
	const char* const pOutput = ReadFromPipe( szBuffer, BUFFERSIZE );
	if ( !pOutput || strncmp( pOutput, "Connected.", 10 ) )
	{
		return false;
	}
	return true;
}

bool CConsolePS3::WaitForFinished( void )
{
	// Should recieve "Finished." once a command is complete
	char szBuffer[BUFFERSIZE];
	for ( u_int u = 0; u < 3; u++ )
	{
		const char* const pOutput = ReadFromPipe( szBuffer, BUFFERSIZE );
		if ( pOutput && !strncmp( pOutput, "Finished.", 9 ) )
		{
			return true;
		}
		if ( m_pLogger ) m_pLogger->LogWarning( "[%s] Attempt to read from pipe timed out. Strike %u!", m_pTarget, u + 1 );
		
	}
	if ( m_pLogger ) m_pLogger->LogWarning( "[%s] Task may have failed.", m_pTarget );
	return false;
}

const char* const CConsolePS3::GetMatchingResponse( const EResponseType eExpectedType )
{
	SResponseTarget responseInfo = GetResponseData( eExpectedType );
	char* const pTarget = responseInfo.m_pTarget;
	// On PS3 we don't have any prefix to match against so just take the first response we get into the target buffer
	return ReadFromPipe( pTarget, BUFFERSIZE );
}
bool CConsolePS3::GatherDumpFile( const char* const pMapName, const char* const pDumpFile, const char* const pOutputDir )
{
	char sourceFile[255];
	char destFile[255];
	if ( pMapName && pOutputDir )
	{
		// Make sure there is a directory for our level
		if ( pMapName[0] != '\0' )
		{
			if ( IsVerbose() )
				printf( "Info - [%s] Creating %s/%s if it does not already exist\n", m_pTarget, pOutputDir, pMapName );

			SetCurrentDirectoryA( pOutputDir );
			CreateDirectoryA( pMapName, NULL );
		}

		// Generate source and destination file names
		strcpy( sourceFile, m_pHomeDirectory );
		strcat( sourceFile, "/" );
		strcat( sourceFile, pDumpFile );

		strcpy( destFile, pOutputDir );
		strcat( destFile, "/" );
		strcat( destFile, pMapName );
		strcat( destFile, "/" );
		strcat( destFile, pDumpFile );

		if ( IsVerbose() )
			printf( "Info - [%s] Copying log from %s to %s\n", m_pTarget, sourceFile, destFile );

		CopyFileA( sourceFile, destFile, false );
		return true;
	}
	else
	{
		fprintf( stderr, "[Error] Invalid dump file or level name\n" );
		return false;
	}
}

// End externally visible methods

// Target enumeration methods

int CConsolePS3::HandleEnumCallBack(HTARGET hTarget)
{
	SNPS3TargetInfo ti;
	SNPS3TargetInfo* pti = new SNPS3TargetInfo;

	if (pti)
	{
		ti.hTarget = hTarget;
		ti.nFlags = SN_TI_TARGETID;

		if (SN_S_OK == SNPS3GetTargetInfo(&ti))
		{
			// Store target parameters.
			pti->hTarget = hTarget;
			pti->pszName = _strdup(ti.pszName);
			pti->pszHomeDir = _strdup(ti.pszHomeDir);
			pti->pszFSDir = _strdup(ti.pszFSDir);

			// Store this target.
			m_Targets.push_back(pti);
		}
		else
		{
			// Terminate enumeration.
			return 1;
		}
	}

	// Carry on with enumeration.
	return 0;
}

bool CConsolePS3::FindFirstConnectedTarget(void)
{
	ECONNECTSTATUS nStatus = (ECONNECTSTATUS) -1;
	char*  pszUsage = 0;

	vector<SNPS3TargetInfo*>::iterator iter = m_Targets.begin();

	while (iter != m_Targets.end())
	{
		SNRESULT snr = SNPS3GetConnectStatus((*iter)->hTarget,	&nStatus, &pszUsage);

		if (SN_SUCCEEDED( snr ))
		{
			if (nStatus == CS_CONNECTED)
			{
				SNPS3TargetInfo ti;

				ti.hTarget = (*iter)->hTarget;
				ti.nFlags = SN_TI_TARGETID;

				if (SN_S_OK == SNPS3GetTargetInfo(&ti))
				{
					// Store target parameters.
					m_hTargetId = ti.hTarget;
					m_pTarget = _strdup(ti.pszName);

					return true;
				}
			}
		}

		iter++;
	}

	return false;
}

bool CConsolePS3::FindFirstAvailableTarget(void)
{
	UINT   nStatus = -1;
	char*  pszUsage = 0;

	vector<SNPS3TargetInfo*>::iterator iter = m_Targets.begin();

	while (iter != m_Targets.end())
	{
		SNRESULT snr = SNPS3Connect((*iter)->hTarget, NULL);

		if (SN_SUCCEEDED( snr ))
		{
			SNPS3TargetInfo ti;

			ti.hTarget = (*iter)->hTarget;
			ti.nFlags = SN_TI_TARGETID;

			if (SN_S_OK == SNPS3GetTargetInfo(&ti))
			{
				// Store target parameters.
				m_hTargetId = ti.hTarget;
				m_pTarget = _strdup(ti.pszName);

				return true;
			}
		}

		iter++;
	}

	return false;
}

bool CConsolePS3::GetTargetFromIP(const char* pszIPAddr, HTARGET &hTarget)
{
	TMAPI_TCPIP_CONNECT_PROP oConnection;
	vector<SNPS3TargetInfo*>::iterator iter = m_Targets.begin();

	while (iter != m_Targets.end())
	{
		if (SN_SUCCEEDED( SNPS3GetConnectionInfo((*iter)->hTarget, &oConnection) ))
		{
			if (strcmp(pszIPAddr, oConnection.szIPAddress) == 0)
			{
				hTarget = (*iter)->hTarget;
				return true;
			}
		}

		++iter;
	}

	return false;
}


bool CConsolePS3::SetFSDir(void)
{
	SNRESULT snr = SN_S_OK;

	SNPS3TargetInfo ti;

	ti.hTarget  = m_hTargetId;
	ti.nFlags   = SN_TI_FILESERVEDIR | SN_TI_TARGETID;
	ti.pszFSDir = m_pHomeDirectory;

	if (SN_FAILED( snr = SNPS3SetTargetInfo(&ti) ))
	{
		fprintf( stderr, "Error - Failed to set file serving directory" );
		return false;
	}

	if ( IsVerbose() )
		printf( "Info - [%s] File serving directory changed to \"%s\"\n", m_pTarget, m_pHomeDirectory );

	return true;
}

bool CConsolePS3::SetHomeDir(void)
{
	SNRESULT snr = SN_S_OK;

	SNPS3TargetInfo ti;

	ti.hTarget    = m_hTargetId;
	ti.nFlags     = SN_TI_HOMEDIR | SN_TI_TARGETID;
	ti.pszHomeDir = m_pHomeDirectory;

	if (SN_FAILED( snr = SNPS3SetTargetInfo(&ti) ))
	{
		fprintf( stderr, "Error - Failed to set home directory" );
		return false;
	}

	if ( IsVerbose() )
	 printf( "Info - [%s] Home directory changed to \"%s\"\n", m_pTarget, m_pHomeDirectory );

	return true;
}

// End target enumeration methods

// TTY methods

bool CConsolePS3::InitializeTTYStreams()
{
	SNRESULT snr = SN_S_OK;

	if (m_pTTYStreams != NULL)
		return true;

	// Get number of available TTY channels.
	if (SN_FAILED( snr = SNPS3ListTTYStreams(m_hTargetId, &m_numTTYStreams, NULL) ))
	{
		fprintf( stderr, "Error - Failed to get count of TTY streams" );
		return false;
	}

	m_pTTYStreams = new TTYSTREAM[m_numTTYStreams];

	// Get list of available TTY channels.
	if (SN_FAILED( snr = SNPS3ListTTYStreams(m_hTargetId, &m_numTTYStreams, m_pTTYStreams) ))
	{
		fprintf( stderr, "Error - Failed to list TTY streams");
		
		delete [] m_pTTYStreams;
		m_pTTYStreams = NULL;

		m_numTTYStreams = 0;

		return false;
	}

	return true;
}

void CConsolePS3::HandleProcessTTYCallback(HTARGET /*hTarget*/, UINT uiType, UINT uiStreamId, 
						SNRESULT /*Result*/, UINT uiLength, BYTE *pData )
{
	if (uiType == SN_EVENT_TTY)
	{
		char* szData = new char[uiLength + 1];
		strncpy_s(szData, uiLength + 1, (char*) pData, uiLength);
		szData[uiLength] = '\0';

		char* pCtrlZ = strchr(szData, 26);

		if (pCtrlZ)
			*pCtrlZ = 0;

		// Output the received TTY text.
		if ( ShouldOutputTTY() )
			printf("%s", szData);

		// If Ctrl Z found, exit the message pump.
		if (pCtrlZ)
		{
			if (*(pCtrlZ - 1) != '\n')
				puts("");

			m_quit = true;
		}

		delete[] szData;

		fflush(stdout);
	}
}

// End of TTY methods

// Shutdown methods

void CConsolePS3::ClearMessagesBeforeExit(void)
{
	Sleep(500);

	while (SNPS3Kick() == SN_S_OK)
		/* Do nothing */;
}

void CConsolePS3::FinalizeTTYStreams()
{
	delete [] m_pTTYStreams;
	m_pTTYStreams = NULL;

	m_numTTYStreams = 0;
}

bool CConsolePS3::DisconnectTTYEvents(void)
{
	// Cancel TTY notifications (-1 is all channels).
	for (size_t i = 0 ; i < m_TTYChannels.size() ; ++i)
	{
		if (SN_FAILED( SNPS3CancelTTYEvents(m_hTargetId, m_TTYChannels[i]) ))
		{
			printf("Error - Failed to cancel TTY notifications on %s channel\n",
				(m_TTYChannels[i] == -1) ? "every" : m_pTTYStreams[m_TTYChannels[i]].szName);
		}
	}

	// Deallocate memory for TTY channels.
	FinalizeTTYStreams();

	return true;
}

void CConsolePS3::FreeTargetList()
{
	for (int i = 0; i < (int)m_Targets.size(); i++)
	{
		free((void*)m_Targets[i]->pszName);
		free((void*)m_Targets[i]->pszFSDir);
		free((void*)m_Targets[i]->pszHomeDir);

		delete m_Targets[i];
	}

	m_Targets.clear();
}

void CConsolePS3::ProcessArgument( TArguments& list, const string& sArgument )
{
		const int numChars = sArgument.size();
		if(numChars != 0)
		{
			char* szArgument = new char[numChars + 1];
			strncpy_s(szArgument, numChars + 1, sArgument.c_str(), numChars);
			szArgument[numChars] = '\0';
			list.push_back(szArgument);
		}
}

CConsolePS3::TArguments CConsolePS3::GetArguments( const vector<string> &cmdLineParams ) const
{
	TArguments arguments;
	string sArguments(m_pArguments);
	istringstream sArgumentsStream(sArguments);
	string sArgument;
	
	// Add the pipe name to use for target communication
	ProcessArgument( arguments, string( "-lt_pipename " ) + m_pPipeName );

	// Add the arguments from the config file
	while ( getline(sArgumentsStream, sArgument, ' ') )
	{
		ProcessArgument( arguments, sArgument );
	}

	// And any additional arguments generated by the tool for this particular reset only
	for ( vector<string>::const_iterator it = cmdLineParams.begin(), end = cmdLineParams.end(); it != end; it++ )
	{
		ProcessArgument( arguments, (*it) );
	}

	return arguments;
}

// End shutdown methods

// Console output methods

void CConsolePS3::PrintError( const char* const pError, SNRESULT res )
{
	if ( pError )
	{
		const char* pErrorCode = NULL;
		SNPS3TranslateError( res, &pErrorCode );
	
		PrintError( pError, pErrorCode );
	}
}

// End console output methods
