////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek, 1999-2009.
// -------------------------------------------------------------------------
//  File name:   ConsoleXBox.cpp
//  Version:     v1.00
//  Created:     19/10/2009 by Steve Barnett.
//  Description: Class for communication with the XBox360
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include "ConsoleXBox.h"
#include <assert.h>
#include <Xbdm.h>

void sleep(unsigned int mseconds);	// Declared in StatsTool.cpp

#define ENABLE_DEBUG()																											(0)

using namespace std;

CConsoleXBox* CConsoleXBox::s_pInstance = NULL;

CConsoleXBox::CConsoleXBox( const char* const pExePath, const char* const pArguments, const char* const pTarget, const char* const pPipeName, SLogger* const pLogger, bool verbose /*= false*/, bool outputTTY /*= true */ )
: IConsoleInterface( pExePath, pArguments, pTarget, pPipeName, pLogger, verbose, outputTTY )
, m_pXexDirectory( NULL )
, m_pRealTarget( NULL )
, m_hResponseWaiting( CreateEvent( NULL, 0, 0, NULL ) )
, m_hResponseLock( CreateMutex( NULL, FALSE, NULL ) )
{
	if ( !s_pInstance ) s_pInstance = this;

	// Process the xex path into a home directory and filename

	// Scan through for the last slash
	const char* pCur = pExePath;
	const char* pLastSlash = pExePath;
	while ( *pCur != '\0' )
	{
		if ( *pCur == '/' || *pCur == '\\' )
		{
			pLastSlash = pCur;
		}
		++pCur;
	}

	// Allocate buffers for path and self filename
	if ( m_pExePath )
	{
		const int pathLen = (pLastSlash - m_pExePath);
		m_pXexDirectory = new char[pathLen + 1];
		if ( m_pXexDirectory )
		{
			strncpy_s( m_pXexDirectory, pathLen + 1, m_pExePath, pathLen );
			m_pXexDirectory[pathLen] = '\0';
		}
	}

	if ( m_hResponseWaiting == NULL || m_hResponseLock == NULL )
	{
		PrintError( "Failed to create synchronisation objects", "" );
	}
}

CConsoleXBox::~CConsoleXBox( void )
{
	delete [] m_pXexDirectory;
	delete [] m_pRealTarget;
	if ( s_pInstance == this )
	{
		s_pInstance = NULL;
	}
	CloseHandle( m_hResponseLock );
	CloseHandle( m_hResponseWaiting );
}

bool CConsoleXBox::Connect( void )
{
	// ToDo: Ensure the image is up to date

	// If a target is specified then set the current XBox 360
	if ( m_pTarget && m_pTarget[0] != '\0' )
	{
		if ( IsVerbose() )
			printf( "Info - [%s] Setting target\n", m_pTarget );
		DmSetXboxName( m_pTarget );
	}

	// Get the actual target name
	// ToDo: Choose correct target if specified
	m_pRealTarget = new char[BUFFERSIZE];
	DWORD len = BUFFERSIZE;
	HRESULT res = DmGetNameOfXbox( m_pRealTarget, &len, false );
	if ( res != XBDM_NOERR )
	{
		PrintError( "Could not get name of XBox", res );
	}
	else
	{
		if ( IsVerbose() )
			printf( "Info - [%s] Connected to target\n", m_pRealTarget );
	}

	return ( res == XBDM_NOERR );
}

bool CConsoleXBox::ResetImpl( const vector<string> &cmdLineParams, bool )
{
	// Reboot the console
	string sParams(m_pArguments);
	for ( vector<string>::const_iterator it = cmdLineParams.begin(), end = cmdLineParams.end(); it != end; ++it )
	{
		sParams += string(" ") + *it;
	}
	HRESULT res = DmRebootEx( DMBOOT_STOP, m_pExePath, m_pXexDirectory, sParams.c_str() );
	if ( res == XBDM_NOERR )
	{
		// Give it 10 seconds to boot to prevent locking up the 360
		sleep( 10000 );
		int count = 0;
		while ( DmGo() != XBDM_NOERR && count < 10)
		{
			sleep( 10000 );
			count++;
		}
		if ( count >= 10 )
		{
			m_pLogger->LogError( "Failed to boot the XBox 360. It may have locked up and require a manual reboot." );
			return false;
		}
		DmConnectDebugger( true );
		while ( DmIsDebuggerPresent() == XBDM_NOERR && count < 10 )
		{
			sleep( 1000 );
			DmConnectDebugger( true );
			count++;
		}
		if ( count >= 10 )
		{
			m_pLogger->LogWarning( "Timeout trying to connect Debug Monitor. This may cause further issues." );
		}
		else if ( IsVerbose() )
		{
			printf( "Info - [%s] Attached to target\n", m_pRealTarget );
		}
	}
	return ( res == XBDM_NOERR );
}

void CConsoleXBox::Shutdown( void )
{
	// Delays to prevent problems caused by attempting to reset the console when bootanim.xex is running
	sleep( 10000 );
	HRESULT res = DmReboot( DMBOOT_COLD );
	sleep( 30000 );
}

void CConsoleXBox::UpdateComms( void )
{
	// Nothing here
}

bool CConsoleXBox::CreatePipe( void )
{
	// Initialise console connection
	HRESULT res = DmOpenConnection( &m_connection );
	if ( res == XBDM_NOERR )
	{
    // Make sure we'll be able to receive notifications
    res = DmOpenNotificationSession( 0, &m_session );
		if ( res != XBDM_NOERR )
		{
			PrintError( "Could not open notification session", res );
		}
		res = DmNotify( m_session, DM_DEBUGSTR, DebugStringHandler );
		if ( res != XBDM_NOERR )
		{
			PrintError( "Could not register notifications", res );
		}
		res = DmRegisterNotificationProcessor( m_session, "crysis_statsagent", NotificationHandler );
		if ( res != XBDM_NOERR )
		{
			PrintError( "Could not register notification processor", res );
		}
	}
	return ( res == XBDM_NOERR );
}

void CConsoleXBox::SendToPipe( const char* const pData )
{
	char pBuffer[BUFFERSIZE];
	char pResponse[256];
	DWORD bufferSize = 256;
	sprintf_s( pBuffer, BUFFERSIZE, "crysis_statsagent!write %s\n", pData );
	pResponse[0] = '\0';
	HRESULT res = DmSendCommand( m_connection, pBuffer, pResponse, &bufferSize );
	if ( IsVerbose() && ENABLE_DEBUG() )
		printf( "Info - [%s] Sending command '%s' response '%s'\n", m_pRealTarget, pBuffer, pResponse );
}

void CConsoleXBox::PopResponse( char* const pBuffer, int len )
{
		const string response = m_responseQueue.back();
		const int respLen = response.length();
		strncpy_s( pBuffer, len, response.c_str(), respLen );
		pBuffer[respLen] = '\0';
		m_responseQueue.pop();
}

const char* CConsoleXBox::ReadFromPipe( char* const pBuffer, const int len )
{
	assert( pBuffer );
	const int timeout = 300 * 1000;	// 5 minutes
	const char* pResult = NULL;
	// Lock the response queue
	// Timeout if 5 minutes pass to prevent a deadlock if the console crashes
	if ( WaitForSingleObject( m_hResponseLock, timeout ) == WAIT_TIMEOUT )
	{
		return pResult;
	}
	// If the (now locked) queue isn't empty then pop the data off, unlock and return
	else if ( !m_responseQueue.empty() )
	{
		PopResponse( pBuffer, len );
		pResult = pBuffer;
		ReleaseMutex( m_hResponseLock );
		return pResult;
	}
	// Finally if the queue was empty then unlock it again and wait for a signal from the
	// notification handler to say that the queue is populated. We can't just wait for the
	// signal initially as there is a potential deadlock if two threads were to wait for a
	// a single signal (it is on/off and not counted).
	else
	{
		ReleaseMutex( m_hResponseLock );
		// Query the Event to see if a response is waiting
		if ( WaitForSingleObject( m_hResponseWaiting, timeout ) != WAIT_TIMEOUT )
		{
			// Lock the response queue
			WaitForSingleObject( m_hResponseLock, INFINITE );
			if ( !m_responseQueue.empty() )
			{
				PopResponse( pBuffer, len );
				pResult = pBuffer;
			}
			else
			{
				PrintError( "Response signalled but the queue was empty", "" );
				pBuffer[0] = '\0';
			}
			ReleaseMutex( m_hResponseLock );
		}
	}
	return pResult;
}

void CConsoleXBox::ClosePipe( void )
{
	HRESULT res = DmCloseConnection( m_connection );
	if ( IsVerbose() )
	{
		if ( res == XBDM_NOERR )
		{
			printf( "Info - [%s] Closing connection\n", m_pRealTarget );
		}
		else
		{
			PrintError( "Failed to close connection", res );
		}
	}
	res = DmCloseNotificationSession( m_session );
	if ( IsVerbose() )
	{
		if ( res == XBDM_NOERR )
		{
			printf( "Info - [%s] Closing notification session\n", m_pRealTarget );
		}
		else
		{
			PrintError( "Failed to close notification session", res );
		}
	}
}

bool CConsoleXBox::WaitForConnected( void )
{
	static char pResponse[256];
	DWORD bufferSize = 256;
	HRESULT res;
	int tries = 24;
	// Once the agent is running we should receive a 200- response code
	pResponse[0] = '\0';
	res = DmSendCommand( m_connection, "crysis_statsagent!read", pResponse, &bufferSize );
	while ( strncmp( pResponse, "200- ", 5 ) && tries-- )
	{
		if ( IsVerbose() && ENABLE_DEBUG() )
			fprintf( stderr, "[WaitForConnected] Response: %s\n", pResponse );
		sleep( 10000 );
		pResponse[0] = '\0';
		res = DmSendCommand( m_connection, "crysis_statsagent!read", pResponse, &bufferSize );
	}
	if ( !pResponse || strncmp( pResponse, "200- ", 5 ) )
	{
		return false;
	}
	return true;
}

bool CConsoleXBox::WaitForFinished( void )
{
	// Should recieve "Finished." once command has completed
	int tries = 100;
	char szBuffer[s_DMBUFFERSIZE];
	const char* pOutput = ReadFromPipe( szBuffer, s_DMBUFFERSIZE );
	// Unknown means the stats agent isn't initialised yet - give it time
	while ( pOutput && strncmp( pOutput, "Finished.", 9 ) && tries-- )
	{
		sleep( 6000 );
		pOutput = ReadFromPipe( szBuffer, s_DMBUFFERSIZE );
	}
	if ( !pOutput || strncmp( pOutput, "Finished.", 10 ) )
	{
		return false;
	}
	return true;
}

const char* const CConsoleXBox::GetMatchingResponse( const EResponseType eExpectedType )
{
	SResponseTarget responseInfo = GetResponseData( eExpectedType );
	const char* const pHeader = responseInfo.m_pHeader;
	const int headerLen = strlen( pHeader );
	char* const pTarget = responseInfo.m_pTarget;
	// Should receive "<header>[filename]" once command has completed
	int tries = 30;
	const char* pOutput = ReadFromPipe( pTarget, BUFFERSIZE );
	// Unknown means the stats agent isn't initialised yet - give it time
	while ( pOutput && strncmp( pOutput, pHeader, headerLen ) && tries-- )
	{
		sleep( 5000 );
		pOutput = ReadFromPipe( pTarget, BUFFERSIZE );
	}
	if ( !pOutput || strncmp( pOutput, pHeader, headerLen ) )
	{
		return NULL;
	}
	return pOutput;
}

bool CConsoleXBox::GatherDumpFile( const char* const pMapName, const char* const pDumpFile, const char* const pOutputDir )
{
	const int BUFFER_SIZE = 155;
	char sourceFile[BUFFER_SIZE];
	char destFile[BUFFER_SIZE];
	if ( pMapName && pOutputDir )
	{
		// Make sure there is a directory for our level
		if ( pMapName[0] != '\0' )
		{
			if ( IsVerbose() )
				printf( "Info - [%s] Creating %s/%s if it does not already exist\n", m_pRealTarget, pOutputDir, pMapName );

			SetCurrentDirectoryA( pOutputDir );
			CreateDirectoryA( pMapName, NULL );
		}

		// Generate source and destination file names
		strcpy_s( sourceFile, m_pXexDirectory );
		strcat_s( sourceFile, "\\" );
		strcat_s( sourceFile, pDumpFile );

		strcpy_s( destFile, pOutputDir );
		strcat_s( destFile, "\\" );
		if ( pMapName[0] != '\0' )
		{
			strcat_s( destFile, pMapName );
			strcat_s( destFile, "\\" );
		}
		strcat_s( destFile, pDumpFile );

		if ( IsVerbose() )
			printf( "Info - [%s] Copying log from %s to %s\n", m_pRealTarget, sourceFile, destFile );

		for ( char* pCur = sourceFile; *pCur != '\0'; ++pCur )
		{
			if ( *pCur == '/' ) *pCur = '\\';
		}

		sleep( 10000 );	// Pause to allow the file to be closed before we try to gather

		HRESULT res = DmReceiveFileA( destFile, sourceFile );
		if ( res != XBDM_NOERR )
		{
			PrintError( sourceFile, res );
		}
	}
	else
	{
		PrintError( "Invalid dump file or level name\n", m_pRealTarget );
		fprintf( stderr, "[Error] \n" );
		return false;
	}
	return true;
}

// Console output
void CConsoleXBox::PrintError( const char* const pError, HRESULT res )
{
	char pErrorCode[256];
	DmTranslateErrorA( res, pErrorCode, 256 );
	PrintError( pError, pErrorCode );
}

// Handle notifications from the console
DWORD CConsoleXBox::ProcessNotification( const CHAR* pNotification )
{
	if ( !strncmp( pNotification, "crysis_statsagent!", 18 ) )
	{
		WaitForSingleObject( m_hResponseLock, INFINITE );
#if 0
		if ( IsVerbose() )
			fprintf( stdout, "[Notification:%s] %s", m_pRealTarget, pNotification );
#endif
		// Add the response to a queue
		m_responseQueue.push( string( pNotification + 18 ) );
		// Signal the consumer thread to let it know that the queue is not empty
		SetEvent( m_hResponseWaiting );
		ReleaseMutex( m_hResponseLock );
	}
	
	return S_OK;
}

DWORD __stdcall CConsoleXBox::NotificationHandler( const CHAR* pNotification )
{
	return s_pInstance ? s_pInstance->ProcessNotification( pNotification ) : S_FALSE;
}

DWORD __stdcall CConsoleXBox::ProcessDebugString( ULONG dwNotification, DWORD dwParam )
{
	(void)dwNotification;
	PDMN_DEBUGSTR p = ( PDMN_DEBUGSTR )dwParam;

	if ( m_outputTTY )
	{
		// The string may not be null-terminated, so make a terminated copy
		// for printing
		CHAR* strTemp = new CHAR[ p->Length + 1 ];
		memcpy( strTemp, p->String, p->Length * sizeof( CHAR ) );
		strTemp[ p->Length ] = 0;
		fprintf( stdout, "%s", strTemp );
		delete[] strTemp;
	}

	return S_OK;
}

DWORD __stdcall CConsoleXBox::DebugStringHandler( ULONG dwNotification, DWORD dwParam )
{
	return s_pInstance ? s_pInstance->ProcessDebugString( dwNotification, dwParam ) : S_FALSE;
}
